/*
 * JColln.cpp
 *
 * Collection implementation
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOS2.h"
#include "JPM.hpp"
#include "JColln.hpp"

// Collections (bag, set) are implemented using binary trees.
// No smartness or anything here.  Would be a good thing to reimplement
// using R-B or (for the brave) AVL trees.

// There are horrids all over this thing, casting to void *s and back again
// and stuff.  Solution would be (I guess) to insist that everything placed
// in a collection derive from some abc with the methods we need.  But then
// we'd be one step nearer to Java horribleness.

// Tree node class ----------------------------------------------------------
struct TreeNode : public JCollection::node
{
   TreeNode *left, *right, *parent;

   TreeNode( void *e) : left( 0), right( 0), parent( 0) { element = e; }
  ~TreeNode() {}

   // insert a new node
   void add( TreeNode *newnode, JCollection *cn)
   {
      int cmp = cn->compare( newnode->element, element, true);
      // exception could just have been thrown...
#ifndef REALLY_THROW_EXCEPTIONS
      if( JPM::current()->exn) return;
#endif
      if( cmp < 0) {
         if( right) right->add( newnode, cn);
         else { right = newnode; newnode->parent = this; }
      } else {
         if( left) left->add( newnode, cn);
         else { left = newnode; newnode->parent = this; }
      }
   }

   // return the (first) node with the element. Don't throw if it goes wrong.
   TreeNode *find( void *el, JCollection *cn)
   {
      int cmp = cn->compare( el, element, false);
      if( !cmp) return this;
      if( cmp < 0) {
         if( right) return right->find( el, cn);
      } else {
         if( left) return left->find( el, cn);
      }
      return 0;
   }

   // return the (first) node with the key. Don't throw if it goes wrong.
   TreeNode *findK( void *k, JKeyCollection *cn)
   {
      int cmp = cn->compareK( element, k);
      if( !cmp) return this;
      if( cmp < 0) {
         if( right) return right->findK( k, cn);
      } else {
         if( left) return left->findK( k, cn);
      }
      return 0;
   }

   // run an iterator
   void iterate( JVIterator &i)
   {
      if( left) left->iterate( i);
      i.iaction( element);
      if( right) right->iterate( i);
   }

   // remove node from tree, patching up
   void unlink( TreeNode * &referer, JCollection *cn)
   {
      TreeNode *new_me( 0);

      if( right && left) { right->add( left, cn); new_me = right; }
      else if( right)    new_me = right;
      else if( left)     new_me = left;

      left = right = 0;
      if( new_me) new_me->parent = parent;
      referer = new_me;
   }

   // parameterised destructor
   void destroy( JCollection *cn)
   {
      if( left) left->destroy( cn);
      cn->deleteElement( element);
      if( right) right->destroy( cn);
      delete this;
   }
};

// Base collection class ----------------------------------------------------
void JCollection::add( void *element)
{
   assertParms( element, "JColln::add");

   TreeNode *node = new TreeNode( element);
   if( root) ((TreeNode *)root)->add( node, this);
   else root = node;
#ifndef REALLY_THROW_EXCEPTIONS
   JPM *pm( JPM::current());
   if( !pm->exn)
#endif
   cElements++;
}

void JCollection::remove( void *element)
{
   assertParms( element, "JColln::remove");

   TreeNode *node = (TreeNode *)find( element);
   if( node) doRemove( node);
   else jlib_throw( new JElementNotFound);
}

void JCollection::doRemove( TreeNode *node)
{
   TreeNode *rt = (TreeNode *) root;
   if( node == rt) {
      node->unlink( rt, this);
      root = rt;
   } else
      node->unlink( node->parent->right == node ?
                    node->parent->right         :
                    node->parent->left, this);

   node->destroy( this);
   cElements--;
}

void JCollection::removeAt( const JVCursor &c)
{
   if( !c.isValid()) jlib_throw( new JInvalidCursor);
   else doRemove( (TreeNode *) c.cnode);
}

void JCollection::doForAll( const JVIterator &iterator)
{
   if( root) ((TreeNode *)root)->iterate( (JVIterator &)iterator);
}

JCollection::node *JCollection::find( void *element) const
{
   assertParms( element, "JColln::find");

   if( !root) return 0;

   return ((TreeNode *)root)->find( element, (JCollection *)this);
}

void JCollection::empty()
{
   if( root) ((TreeNode *)root)->destroy( this);
   root = 0;
   cElements = 0;
}

// Base cursor class --------------------------------------------------------
void JVCursor::toFirst()
{
   if( !cn.root) cnode = 0;
   else {
      TreeNode *c = (TreeNode *)cn.root;
      while( c->left) c = c->left;
      cnode = c;
   }
}

void JVCursor::toNext()
{
   if( !isValid()) jlib_throw( new JInvalidCursor);
   else {
      TreeNode *c = (TreeNode *)cnode;

      // if there's a `right' ptr then go to the first element of the tree
      // it points to
      if( c->right) for( c = c->right; c->left; c = c->left);
      else {
         // go up the tree until we find a left child...
         while( c->parent && c->parent->right == c) c = c->parent;
         // ...and go to its parent (may be 0)
         c = c->parent;
      }

      cnode = c;
   }
}

void JVCursor::toLast()
{
   if( !cn.root) cnode = 0;
   else {
      TreeNode *c = (TreeNode *)cn.root;
      while( c->right) c = c->right;
      cnode = c;
   }
}

void JVCursor::toPrevious()
{
   if( isValid()) jlib_throw( new JInvalidCursor);
   else {
      TreeNode *c = (TreeNode *)cnode;

      // if there's a left ptr then go to the last element of the tree
      // it points to
      if( c->left) for( c = c->left; c->right; c = c->right);
      else {
         // go up the tree until we find a right child...
         while( c->parent && c->parent->left == c) c = c->parent;
         // ...and go to its parent (0 if root)
         c = c->parent;
      }

      cnode = c;
   }
}

// Base key collection class ------------------------------------------------
JCollection::node *JKeyCollection::findK( void *k) const
{
   assertParms( k, "JKColln::findK");

   if( !root) return 0;

   return ((TreeNode *)root)->findK( k, (JKeyCollection *)this);
}

void *JKeyCollection::withK( void *k) const
{
   assertParms( k, "JKColln::withK");

   TreeNode *n = (TreeNode *)findK( k);
   void *ret = 0;
   if( !n) jlib_throw( new JKeyNotFound);
   else ret = n->element;

   return ret;
}

void JKeyCollection::removeK( void *k)
{
   assertParms( k, "JKColln::removeK");

   TreeNode *n = (TreeNode *)findK( k);
   if( !n) jlib_throw( new JKeyNotFound);
   else doRemove( n);
}

void JKeyCollection::removeAllK( void *k)
{
   // interesting algorithm here :-(

   // once allowed to throw...
   removeK( k);

#ifndef REALLY_THROW_EXCEPTIONS
   JPM *pm( JPM::current());
   if( pm->exn) return;
#else
   try {
#endif
      // ...then keep going until no more
      for( ;;) {
         removeK( k);
#ifndef REALLY_THROW_EXCEPTIONS
         if( pm->exn) {
            jlib_catch();
            break;
         }
      }
#else
      }
   } catch( JKeyNotFound *) {
   }
#endif
}

// Exception classes --------------------------------------------------------
JDuplicateElement::JDuplicateElement()
   : JException( err_duplicate_el, JExn::duplicate) {}

JInvalidCursor::JInvalidCursor()
   : JException( err_badcursor, JExn::badCursor) {}

JElementNotFound::JElementNotFound()
   : JException( err_notfound, JExn::base, "element") {}

JKeyNotFound::JKeyNotFound()
   : JException( err_notfound, JExn::badKey, "key") {}

JDuplicateKey::JDuplicateKey()
   : JException( err_duplicate_key, JExn::duplicateKey) {}
