/*
 * JDragSet.cpp
 *
 * Dragndrop operation object
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JDrag.hpp"
#include "JDragSet.hpp"
#include "JDrgItem.hpp"
#include "JWindow.hpp"
#include "JSystem.hpp"
#include "JPSpace.hpp"
#include <string.h>
#include "JSystem.hpp"

#define drgError( s) jlib_throw( new JException( err_drag, JExn::base, s))

// constants for which bits of the info to free
static const ulong freeInfo = 0x01;
static const ulong freeStrs = 0x02;

// Base dragset class; Ctors -------------------------------------------------

JDragSet::JDragSet( BOOL /*isLazy*/) : flFree( 0), info( 0)
{
// if( !isLazy)
//    flFree |= freeInfo;
}


// gets current drag-op.
JDragSet::JDragSet() : flFree( 0), info( 0)
{
   JDrag::dtype now = JDrag::currentDrag();

   if( now == JDrag::noDrag)
      jlib_throw( new JException( err_nodrag, JExn::nodrag));
   else {
      info = DrgQueryDraginfoPtr( 0);
      if( !DrgAccessDraginfo( (PDRAGINFO) info))
         pmError( 1, "DrgAccessDraginfo");
      else flFree = freeInfo;
   }
}

// wraps given pointer
// this one is called on *targets* at dm_dragover or dm_drop[help]
//                and on *sources* for dm_delete & dm_print
JDragSet::JDragSet( void *v, BOOL deleteStrs) : flFree( 0), info( 0)
{
   assertParms( v, "JBaseDragSet::");
   info = v;

   if( JDrag::Info.isSource && JDrag::Info.pDraginfo) {
      // we are the source as well as the target. Don't do anything about
      // the dragset, but free the string handles if appropriate
      info = JDrag::Info.pDraginfo;
   } else {
      // we are a target of someone else's drag.
      JDrag::Info.isSource = false;
      JDrag::Info.pDraginfo = v;
      if( !DrgAccessDraginfo( (PDRAGINFO) v))
         pmError( 1, "DrgAccessDraginfo");
      // free the info when we're done
      flFree |= freeInfo;
   }
   if( deleteStrs) flFree |= freeStrs;
}

// Dtor ----------------------------------------------------------------------
JDragSet::~JDragSet()
{
   // erm, is this right ??
   if( !JDrag::Info.isSource)
      JDrag::Info.pDraginfo = 0;

   if( flFree & freeStrs) {
      // target after she's happy
      DrgDeleteDraginfoStrHandles( (PDRAGINFO) info);
   }
   if( flFree & freeInfo) {
      // whenever
      DrgFreeDraginfo( (PDRAGINFO) info);
   }
}

// Access to dragitems -------------------------------------------------------
ulong JDragSet::items() const
{
   return ((PDRAGINFO) info)->cditem;
}

JDragItem JDragSet::operator [] ( ulong i )
{
   void *v = 0;
   if( i >= items())
      jlib_throw( new JBoundsError( err_bounds));
   else {
      v = DrgQueryDragitemPtr( (PDRAGINFO) info, i);
      if( !v) pmError( 2, "DrgQueryItemPtr");
   }
   return JDragItem( v);
}

// Operation in progress -----------------------------------------------------
JDrag::operation JDragSet::opType() const
{
   ushort op = ((PDRAGINFO) info)->usOperation;
   return (JDrag::operation) op;
}

// Criteria for the dragset to meet ------------------------------------------
BOOL JDragSet::willAllow( JDrag::operation op)
{
   int i, j = items();

   // check the actual operation happening is acceptable
   if( !( opType() == op || opType() == JDrag::def))
      return false;

   // then check that for each of the items the source can do the right thing
   for( i = 0; i < j; i++)
      if( ! (self)[ i].supports( op))
         return false;

   return true;
}

BOOL JDragSet::canRenderAs( const JDrag::rmf &r)
{
   int i, j = items();
   for( i = 0; i < j; i++)
      if( ! (self)[ i].hasRenderingMF( r))
         return false;

   return true;
}

BOOL JDragSet::hasRenderingMech( const JDrag::rm &r)
{
   int i, j = items();
   for( i = 0; i < j; i++)
      if( ! (self)[ i].hasRenderingMech( r))
         return false;

   return true;
}

BOOL JDragSet::hasRenderingFmt( const JDrag::rf &r)
{
   int i, j = items();
   for( i = 0; i < j; i++)
      if( ! (self)[ i].hasRenderingFmt( r))
         return false;

   return true;
}

BOOL JDragSet::isOfType( const JDrag::type &t)
{
   int i, j = items();
   for( i = 0; i < j; i++)
      if( ! (self)[ i].hasType( t))
         return false;

   return true;
}

// prepare source for rendering
BOOL JDragSet::prepare( void *xfer) const
{
   return JWindow::SendEvent( ((PDRAGINFO) info)->hwndSource,
                              DM_RENDERPREPARE, xfer);
}

// end a conversation
JDragSet &JDragSet::endConversation( BOOL ok)
{
   int i, m( items());
   for( i = 0; i < m; i++)
      (self)[ i].endConversation( ok);
   return self;
}

// Get a presentation space for the target window ----------------------------
JDragPS JDragSet::getPS( JWindow &w) const
{
   return JDragPS( &w);
}

// Derived class; interface for classes which start drags --------------------
JVDragSet::JVDragSet( JWindow *w, ulong cItems, BOOL lz) : JDragSet( lz)
{
   assertParms( w, "JStdDragSet::");
   JDrag::dtype now = JDrag::currentDrag();

   switch( now) {
      case JDrag::drag:
         if( !JDrag::Info.pDraginfo) {
            // hmm, either someone else is dragging & has got lost,
            // or a previous jlib drag went a bit screwy..
            // let's just let this one pass.
            case JDrag::noDrag:
               info = DrgAllocDraginfo( cItems);
               if( !info)
                  drgError( "allocDragInfo");
               else {
                  ((PDRAGINFO) info)->hwndSource = w->handle();
                  current = 0;
                  // reset current draginfo
                  JDrag::Info.pDraginfo = 0;
                  JDrag::Info.cDragitems = 0;
               }
               break;
         }
         jlib_throw( new JException( err_doubledrag, JExn::doubledrag));
      case JDrag::lazyDrag:
      {
         if( !lz)
            jlib_throw( new JException( err_doubledrag, JExn::doubledrag));
         else {
            PDRAGINFO old = DrgQueryDraginfoPtr( 0);
            if( !old)
               pmError( 2, "DrgQueryDraginfoPtr");
            ulong itms = old->cditem + cItems;
            info = DrgReallocDraginfo( old, itms);
            current = itms - 1;
            if( !info)
               pmError( 2, "DrgReallocDraginfo");
            else ((PDRAGINFO) info)->hwndSource = w->handle();
            break;
         }
      }
   }
}

JVDragSet::~JVDragSet()
{}

JVDragSet &JVDragSet::setOpType( JDrag::operation op)
{
   ((PDRAGINFO) info)->usOperation = (ushort) op;
   return self;
}

JVDragSet &JVDragSet::addDragItem( const JDragItem &item)
{
   // get the dragitem. First check there's space for it.
   if( current == items())
      jlib_throw( new JBoundsError( err_bounds));
   else {
   
      // if hwndItem is 0, set it to the source window in our structure
      if( ((PDRAGITEM) item.dragitem)->hwndItem == 0)
          ((PDRAGITEM) item.dragitem)->hwndItem = ((PDRAGINFO) info)->hwndSource;
   
      BOOL rc = DrgSetDragitem( (PDRAGINFO) info, (PDRAGITEM) item.dragitem,
                                sizeof( DRAGITEM), current++);
      if( !rc)
         pmError( 3, "DrgSetDragitem");
   }

   return self;
}

// First starter class; for standard drags -----------------------------------
JStdDragSet::JStdDragSet( JWindow *w, ulong c) : JVDragSet( w, c, false),
                                                 images( 0), cImages( 0)
{}

JStdDragSet::~JStdDragSet()
{
   if( cImages)
      free( images);
}

JStdDragSet &JStdDragSet::addDragItem( const JDragItem &item)
{
   // copy dragimage structure across
   cImages++;
   // gosh - a c++ realloc that doesn't need a cast.
   images = realloc( images, sizeof( DRAGIMAGE) * cImages);
   // we need to increase the image offset for the second & subsequent images
   if( cImages > 1) {
      ((PDRAGIMAGE) item.dragimage)->cxOffset +=
                   (cImages - 1) * (JSystem::query( JSystem::pointerWidth) / 2);
      ((PDRAGIMAGE) item.dragimage)->cyOffset +=
                   (cImages - 1) * (JSystem::query( JSystem::pointerHeight) / 2);
   }
   memcpy( ((char *)images) + (sizeof( DRAGIMAGE) * (cImages - 1)),
           item.dragimage, sizeof( DRAGIMAGE));

   // get the dragitem
   JVDragSet::addDragItem( item);

   return self;
}

// start the drag
JWindow *JStdDragSet::drag()
{
   void *f = (JException::debugEnabled ? ((void*) 0x80000000L) : 0);

   JDrag::Info.pDraginfo = info;
   JDrag::Info.cDragitems = items();
   JDrag::Info.isSource = true;

   ulong hwnd = DrgDrag( ((PDRAGINFO) info)->hwndSource,
                         (PDRAGINFO) info,
                         (PDRAGIMAGE) images,
                         cImages,
                         VK_ENDDRAG, f);

   // returns 0 if drag was cancelled, or dropped over a !do_drop window
   if( hwnd) {
      return JWindowPtr( hwnd);
   } else {
      // failed, free up the stuff when we go
      JDrag::Info.pDraginfo = 0;
      JDrag::Info.cDragitems = 0;
      flFree |= freeStrs | freeInfo;
   }
   return 0;
}

// Second starter class; for lazy drags --------------------------------------
JLazyDragSet::JLazyDragSet( JWindow *w, ulong c) : JVDragSet( w, c, true)
{}

JLazyDragSet::~JLazyDragSet()
{}

JLazyDragSet &JLazyDragSet::lazyDrag()
{
   ulong rc = DrgLazyDrag( ((PDRAGINFO) info)->hwndSource,
                           (PDRAGINFO) info, 0, 0, 0);
   if( !rc)
      pmError( 2, "DrgLazyDrag");
   else
      JDrag::Info.cLazyDrags++;

   return self;
}
