/*
 * JDrgItem.hpp
 *
 * Base classes for things wot get dragndrop'd
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JDrgItem.hpp"
#include "JIcon.hpp"
#include "JBitmap.hpp"
#include "JWindow.hpp"

// private static methods ----------------------------------------------------
ulong JDragItem::makeStr( const char *str)
{
   if( !str) return 0;
   ulong hStr = DrgAddStrHandle( str);
   if( !hStr)
      pmError( 2, "DrgAddStrHandle");
   return hStr;
}

JStr JDragItem::getStr( ulong hStr)
{
   JStr str;
   ulong l = DrgQueryStrNameLen( hStr);
   str.setsize( l + 1);
   DrgQueryStrName( hStr, l + 1, str.buffer());
   return str;
}

void JDragItem::replaceStr( ulong &hStr, const char *nu)
{
   DrgDeleteStrHandle( hStr);
   hStr = makeStr( nu);
}

// base class; ctors & dtor --------------------------------------------------
JDragItem::JDragItem( const JDrag::rmf &rmf, const JDrag::type &t,
                      ulong hIcon, const char *cnrName, const char *targetName,
                      const char *sourceName, ushort supportedOps, BOOL isBmp)
{
   // set up dragitem structure
   dragitem = calloc( sizeof( DRAGITEM), 1);
   ((PDRAGITEM) dragitem)->hstrType = makeStr( t.value());
   ((PDRAGITEM) dragitem)->hstrRMF = makeStr( rmf.value());
   ((PDRAGITEM) dragitem)->hstrContainerName = makeStr( cnrName);
   ((PDRAGITEM) dragitem)->hstrSourceName = makeStr( sourceName);
   ((PDRAGITEM) dragitem)->hstrTargetName = makeStr( targetName);
   ((PDRAGITEM) dragitem)->fsSupportedOps = supportedOps;

   // set up dragimage structure
   dragimage = calloc( sizeof( DRAGIMAGE), 1);
   ((PDRAGIMAGE) dragimage)->cb = sizeof( DRAGIMAGE);
   ((PDRAGIMAGE) dragimage)->hImage = hIcon;
   ((PDRAGIMAGE) dragimage)->fl = ( isBmp ? DRG_BITMAP : DRG_ICON);
}

JDragItem::JDragItem( void *drg) : dragitem( drg), dragimage( 0)
{}

JDragItem::~JDragItem()
{
   if( dragimage) {
      free( dragimage);
      free( dragitem);
   }
}

// Item ID -------------------------------------------------------------------
JDragItem &JDragItem::setItemID( ulong id)
{
   ((PDRAGITEM) dragitem)->ulItemID = id;
   return self;
}

ulong JDragItem::itemID() const
{
   return ((PDRAGITEM) dragitem)->ulItemID;
}

// Item window ---------------------------------------------------------------
JDragItem &JDragItem::setItemWindow( JWindow *w)
{
   ((PDRAGITEM) dragitem)->hwndItem = w->handle();
   return self;
}

ulong JDragItem::itemWindow() const
{
   return ((PDRAGITEM) dragitem)->hwndItem;
}

JDragItem &JDragItem::endConversation( BOOL happy)
{
   ulong fl = DMFL_TARGETSUCCESSFUL;
   if( !happy) fl = DMFL_TARGETFAIL;
   JWindow::SendEvent( ((PDRAGITEM) dragitem)->hwndItem, DM_ENDCONVERSATION,
                       itemID(), fl);
   return self;
}

JDragItem &JDragItem::setOffset( long x, long y)
{
  ((PDRAGIMAGE) dragimage)->cxOffset = x;
  ((PDRAGIMAGE) dragimage)->cyOffset = y;
  return self;
}

// Supported operations ------------------------------------------------------
JDragItem &JDragItem::setSupportedOps( ushort ops)
{
   ((PDRAGITEM) dragitem)->fsSupportedOps = ops;
   return self;
}

ushort JDragItem::supportedOps() const
{
   return ((PDRAGITEM) dragitem)->fsSupportedOps;
}

BOOL JDragItem::supports( JDrag::operation op)
{
   switch( op) {
      case JDrag::move:
         return supportedOps() & JDrag::moveable;
      case JDrag::copy:
         return supportedOps() & JDrag::copyable;
      case JDrag::link:
         return supportedOps() & JDrag::linkable;
      case JDrag::create:
         return supportedOps() & JDrag::createable;
      default:
         break;
   }
   return false;
}

// fsControl flags -----------------------------------------------------------
JDragItem &JDragItem::needPrepare( BOOL b)
{
   if( b)
      ((PDRAGITEM) dragitem)->fsControl |= DC_PREPARE;
   else
      ((PDRAGITEM) dragitem)->fsControl &= ~DC_PREPARE;
   return self;
}

BOOL JDragItem::needsPrepare() const
{
   return !!(((PDRAGITEM) dragitem)->fsControl & DC_PREPARE);
}

JDragItem &JDragItem::onRemovableMedia( BOOL b)
{
   if( b)
      ((PDRAGITEM) dragitem)->fsControl |= DC_REMOVEABLEMEDIA;
   else
      ((PDRAGITEM) dragitem)->fsControl &= ~DC_REMOVEABLEMEDIA;
   return self;
}

BOOL JDragItem::isOnRemovabelMedia() const
{
   return !!(((PDRAGITEM) dragitem)->fsControl & DC_REMOVEABLEMEDIA);
}

// Access to simpler string things -------------------------------------------
JStr JDragItem::containerName() const
{
   return getStr( ((PDRAGITEM) dragitem)->hstrContainerName);
}

JDragItem &JDragItem::setContainerName( const char *name)
{
   replaceStr( ((PDRAGITEM) dragitem)->hstrContainerName, name);
   return self;
}

JStr JDragItem::targetName() const
{
   return getStr( ((PDRAGITEM) dragitem)->hstrTargetName);
}

JDragItem &JDragItem::setTargetName( const char *name)
{
   replaceStr( ((PDRAGITEM) dragitem)->hstrTargetName, name);
   return self;
}

JStr JDragItem::sourceName() const
{
   return getStr( ((PDRAGITEM) dragitem)->hstrSourceName);
}

JDragItem &JDragItem::setSourceName( const char *name)
{
   replaceStr( ((PDRAGITEM) dragitem)->hstrSourceName, name);
   return self;
}

// supported rmf, types ------------------------------------------------------
BOOL JDragItem::hasType( const JDrag::type &t) const
{
   return DrgVerifyType( (PDRAGITEM) dragitem, t.value());
}

BOOL JDragItem::hasRenderingMF( const JDrag::rmf &r) const
{
   return DrgVerifyRMF( (PDRAGITEM) dragitem, r.RM(), r.RF());
}

BOOL JDragItem::hasRenderingMech( const JDrag::rm &r) const
{
   return DrgVerifyRMF( (PDRAGITEM) dragitem, r.value(), 0);
}

BOOL JDragItem::hasRenderingFmt( const JDrag::rf &r) const
{
   return DrgVerifyRMF( (PDRAGITEM) dragitem, 0, r.value());
}

// dubious support for adding extra pseudo rmf's -----------------------------
JDragItem &JDragItem::supportPrint()
{
   JStr rmf = getStr( ((PDRAGITEM) dragitem)->hstrRMF);
   JVStr newrmf( "%s,<DRM_PRINT,DRF_UNKNOWN>", rmf.buffer());
   replaceStr( ((PDRAGITEM) dragitem)->hstrRMF, newrmf.buffer());
   return self;
}

JDragItem &JDragItem::supportDiscard()
{
   JStr rmf = getStr( ((PDRAGITEM) dragitem)->hstrRMF);
   JVStr newrmf( "%s,<DRM_DISCARD,DRF_UNKNOWN>", rmf.buffer());
   replaceStr( ((PDRAGITEM) dragitem)->hstrRMF, newrmf.buffer());
   return self;
}

// 'Pulled' - ie. no source rendering - file ---------------------------------
ulong _fileIcon()
{
   static JIcon ic( JIcon::sysPointer( JIcon::file));
   return ic.handle();
}

JFileItem::JFileItem( const char *fileName, const char *cnrName, ulong icon,
                      const JDrag::rf &r, const JDrag::type &t)
          : JDragItem( JDrag::rmf( JDrag::file, r),
                       t, icon ? icon : _fileIcon(),
                       cnrName, fileName, fileName,
                       JDrag::copyable | JDrag::moveable)
{}

JFileItem::~JFileItem()
{}

// 'Pushed' - ie. source rendered - file -------------------------------------
// stoopid name, tho.
JPushedFileItem::JPushedFileItem( const char *fileName, ulong hIcon,
                                  const JDrag::rf &r, const JDrag::type &t)
                : JDragItem( JDrag::rmf( JDrag::file, r),
                             t, hIcon ? hIcon : _fileIcon(),
                             0, fileName, 0,
                             JDrag::copyable | JDrag::moveable)
{}

JPushedFileItem::~JPushedFileItem()
{}

// Message item; ie no faffing around ----------------------------------------
JCustomItem::JCustomItem( const JDrag::rmf &rmf, ulong itemId, ulong hIcon)
            : JDragItem( rmf, JDrag::t_unknown, hIcon, 0, 0, 0,
                         JDrag::linkable | JDrag::copyable | JDrag::moveable)
{
   setItemID( itemId);
}

JCustomItem::~JCustomItem()
{}
