/*
 * JEntFld.cpp
 *
 * Entry field gui control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JEntFld.hpp"
#include "JCoord.hpp"
#include "JWindow.hpp"
#include "JStr.hpp"
#include "JCtlH.hpp"
#include "JMParam.hpp"
#include "JIntern.hpp"

#define efError( s) jlib_throw( new JException( err_ef, JExn::base, s))

const unsigned long JEntryField::normal      = ES_AUTOSCROLL | ES_MARGIN | ES_LEFT;
const unsigned long JEntryField::autoSize    = ES_AUTOSIZE;
const unsigned long JEntryField::autoScroll  = ES_AUTOSCROLL;
const unsigned long JEntryField::autoTab     = ES_AUTOTAB;
const unsigned long JEntryField::margin      = ES_MARGIN;
const unsigned long JEntryField::readOnly    = ES_READONLY;
const unsigned long JEntryField::password    = ES_UNREADABLE;
const unsigned long JEntryField::alignLeft   = ES_LEFT;
const unsigned long JEntryField::alignRight  = ES_RIGHT;
const unsigned long JEntryField::alignCenter = ES_CENTER;

JEntryField::JEntryField( JWindow *p, const JPoint &pos, const JSize &size,
                          ulong Id, ulong style, const char *text) : JControl()
{
   assertParms( p, "JEntryField::JEntryField");

   JCreateWBlock b( p->handle(), WC_ENTRYFIELD, (char*) text, JWindow::visible | style,
                    pos, size, p->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
}

// clipboard actions
JEntryField &JEntryField::clear()
{
   BOOL rc = sendEvent( EM_CLEAR);
   if( !rc)
      efError( "clear selection");

   return self;
}

JEntryField &JEntryField::cut()
{
   BOOL rc = sendEvent( EM_CUT);
   if( !rc)
      efError( "cut selection");

   return self;
}

JEntryField &JEntryField::copy()
{
   BOOL rc = sendEvent( EM_COPY);
   if( !rc)
      efError( "copy selection");

   return self;
}

JEntryField &JEntryField::paste()
{
   BOOL rc = sendEvent( EM_PASTE);
   if( !rc)
      efError( "paste text");

   return self;
}

// selection
JEntryField &JEntryField::setSelection( short start, short count)
{
   BOOL rc = sendEvent( EM_SETSEL, JMP( start, count));
   if( !rc)
      efError( "set the selection");

   return self;
}

JStr JEntryField::getSelection() const
{
   JMR   rc( sendEvent( EM_QUERYSEL));
   JStr all( getText());

   return all.substring( rc.s1(), rc.s2());
}

// scroll to position
JEntryField &JEntryField::scrollTo( short offset)
{
   BOOL rc = sendEvent( EM_SETFIRSTCHAR, offset);
   if( !rc)
      jlib_throw( new JBoundsError( err_efbounds));

   return self;
}

// maximum field size
JEntryField &JEntryField::setMaxLength( short length)
{
   BOOL rc = sendEvent( EM_SETTEXTLIMIT, length);
   if( !rc)
      efError( "set the text limit");

   return self;
}

BOOL JEntryField::hasChanged() const
{
   return sendEvent( EM_QUERYCHANGED);
}

// readonly-ness
JEntryField &JEntryField::setReadonly( BOOL set)
{
   BOOL rc = sendEvent( EM_SETREADONLY, set);
   if( !rc)
      efError( "set the read-only state");

   return self;
}

BOOL JEntryField::isReadonly() const
{
   return sendEvent( EM_QUERYREADONLY);
}

// insertion mode
JEntryField &JEntryField::setInsert( BOOL set)
{
   BOOL rc = sendEvent( EM_SETINSERTMODE, set);
   if( !rc)
      efError( "set the insert state");

   return self;
}

// event handling
BOOL JEntryField::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case EN_CHANGE:
         return changed();
         break;
      case EN_KILLFOCUS:
         return lostFocus();
         break;
      case EN_SETFOCUS:
         return gainedFocus();
         break;
      case EN_SCROLL:
         return scrolled();
         break;
   }
   return false;
}
