/*
 * JFont.cpp
 *
 * Fonts
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOs2.h"
#include "JPM.hpp"
#include "JDC.hpp"
#include "JFont.hpp"
#include "JPSpace.hpp"
#include "JWindow.hpp"
#include "JAtoms.hpp"
#include "JFile.hpp"
#include "JBag.hpp"

#include <string.h>

// exceptions etc ------------------------------------------------------------
JFontNotFound::JFontNotFound() : JException( JExn::fontNotFound) {}

#define PFM (fd->metrics)

struct JFontData   // reference count around metrics
{
   PFONTMETRICS metrics;
   int          rc;

   JFontData() : metrics( new FONTMETRICS), rc( 1)
   {}
  ~JFontData()
   { delete metrics; }
};

// ctors, dtor ---------------------------------------------------------------
JFont::JFont( PFONTMETRICS pfm) : fd( new JFontData)
{
   assertParms( pfm, "JFont::ctor");
   memcpy( PFM, pfm, sizeof( FONTMETRICS));
}

JFont::JFont( const char *nm, JBPSpace *ps) : fd( new JFontData)
{
   long      cBuffer = 1;
   JWindowPS deskPS( &JWindow::theDesktopWindow);

   if( !ps) ps = &deskPS;
   long rc = GpiQueryFonts( ps->handle(), QF_PUBLIC | QF_PRIVATE, nm,
                            &cBuffer, sizeof( FONTMETRICS), PFM);
   if( rc == GPI_ALTERROR)
      pmError( 2, "GpiQueryFonts_3");

   if( rc == 1)
      jlib_throw( new JFontNotFound);
}

JFont::JFont( const char *nm, long ptSize, JBPSpace *ps) : fd( 0)
{
   JFontList          lst;
   JFontList::cursor  c( lst);
   JFont             *best = 0;

   // We try to return the requested face at the requested size.
   // If unavailable, we return the requested face in the next size down.
   // If we don't recognise the face, we throw an exception in disgust.
   JFont::list( lst, ps);

   for_cursor( c) {
      JFont *fnt = c.current();
      if( fnt->face() == nm) {
         if( fnt->pointSize() == ptSize) { best = fnt; break; }
         if( best == NULL ||
            (fnt->pointSize() < ptSize &&
             fnt->pointSize() > best->pointSize())) best = fnt;
      }
   }

   if( !c.isValid() && best == NULL)
      jlib_throw( new JFontNotFound);
   else {
      fd = best->fd;
      fd->rc++;
   }
}

JFont::JFont( const JFont &copy) : fd( copy.fd)
{ fd->rc++;
}

JFont::~JFont()
{
   fd->rc--;
   if( !fd->rc) delete fd;
}

JFont &JFont::operator = (const JFont &op2)
{
   if( !--(fd->rc)) delete fd;
   fd = op2.fd;
   fd->rc++;
   return self;
}

// member access : names -----------------------------------------------------
JStr JFont::face() const
{
   if( PFM->fsType & FM_TYPE_FACETRUNC)
      return JAtomTable::system.nameOf( JAtom( PFM->FaceNameAtom));
   else
      return JStr( PFM->szFacename);
}

JStr JFont::family() const
{
   if( PFM->fsType & FM_TYPE_FAMTRUNC)
      return JAtomTable::system.nameOf( JAtom( PFM->FamilyNameAtom));
   else
      return PFM->szFamilyname;
}

// ..and numbers
short JFont::pointSize() const
{
   return PFM->sNominalPointSize / 10;
}

long JFont::externalLeading() const
{
   return PFM->lExternalLeading;
}

long JFont::avgeCharWidth() const
{
   return PFM->lAveCharWidth;
}

JFont::wt JFont::weight() const
{
   return (JFont::wt) PFM->usWeightClass;
}

JFont::wd JFont::width() const
{
   return (JFont::wd) PFM->usWidthClass;
}

// Font attributes -----------------------------------------------------------
BOOL JFont::outline() const
{ return PFM->fsDefn & FM_DEFN_OUTLINE; }

BOOL JFont::italic() const
{ return PFM->fsSelection & FM_SEL_ITALIC; }

BOOL JFont::underlined() const
{ return PFM->fsSelection & FM_SEL_UNDERSCORE; }

BOOL JFont::inverse() const
{ return PFM->fsSelection & FM_SEL_NEGATIVE; }

BOOL JFont::hollow() const
{ return PFM->fsSelection & FM_SEL_OUTLINE; }

BOOL JFont::strikeout() const
{ return PFM->fsSelection & FM_SEL_STRIKEOUT; }

BOOL JFont::bold() const
{ return PFM->fsSelection & FM_SEL_BOLD; }

// Create a list of all public fonts -----------------------------------------
JFontList &JFont::list( JFontList &lst, JBPSpace *ps)
{
   long rc = 0, sysFonts;

   // set up a presentation space to query fonts from
   JWindowPS deskPS( &JWindow::theDesktopWindow);
   if( !ps)  ps = &deskPS;

   // how many fonts are there?
   sysFonts = GpiQueryFonts( ps->handle(), QF_PUBLIC | QF_PRIVATE , NULL, &rc,
                             sizeof( FONTMETRICS), NULL);
   if( sysFonts == GPI_ALTERROR)
      pmError( 2, "GpiQueryFonts_1");

   // allocate space for enough metrics
   FONTMETRICS *afm = new FONTMETRICS [ sysFonts];

   // query the font data
   rc = GpiQueryFonts( ps->handle(), QF_PUBLIC | QF_PRIVATE, NULL, &sysFonts,
                       sizeof( FONTMETRICS), afm);
   if( rc == GPI_ALTERROR)
      pmError( 1, "GpiQueryFonts_2");

   // set up the list
   lst.empty();

   // work out device font resolutions to validate image fonts
   long devRes[ 2] = { ps->dc().horzFontRes(),
                       ps->dc().vertFontRes() };

   for( rc = 0; rc < sysFonts; rc++) {
      // if it's an image font, check its res is okay for the device
      if( !( afm[ rc].fsDefn & FM_DEFN_OUTLINE)   &&
          ( afm[ rc].sXDeviceRes != devRes[ 0]  ||
            afm[ rc].sYDeviceRes != devRes[ 1])    ) continue;

      lst.append( JFont( afm + rc));
   }

   delete [] afm;

   return lst;
}

// Add a file full of fonts to the private area ------------------------------
class JFontManager
{
   JBag<JStr> list;

 public:
   void add( char *s) {
      list.add( JStr( s));
   }

  ~JFontManager() {
      JBag<JStr>::cursor c( list);

      for_cursor( c)
         GpiUnloadFonts( JPM::current()->hab(), *c.current());
   }
} fontManager;

void JFont::addFonts( const JFile &f)
{
   BOOL rc = GpiLoadFonts( JPM::current()->hab(), f.path());
   if( !rc)
      pmError( 2, "GpiLoadFonts");

   fontManager.add( f.path());
}

// Interface to WinFontDlg ---------------------------------------------------
JFont JFont::pickFont( JWindow *owner, const char *title, JFont *init)
{
   FONTDLG   fd;
   char      familyName[ FACESIZE] = "\0";
   JWindowPS deskPS( &JWindow::theDesktopWindow);

   // initialise fields in the dialog's structure
   memset( &fd, 0, sizeof( FONTDLG));
   fd.cbSize = sizeof( FONTDLG);
   fd.hpsScreen = deskPS.handle();
   fd.pszTitle = (char*)title;
   fd.pszFamilyname = familyName;
   fd.fl = FNTS_RESETBUTTON;
   fd.clrBack = CLR_WHITE;
   fd.clrFore = CLR_BLACK;
   fd.usFamilyBufLen = FACESIZE;

   // if we're given a 'current font', fill in some more fields
   if( init) {
      fd.fxPointSize = MAKEFIXED( init->pointSize(), 0);
      fd.usWeight = (ushort) init->weight();
      fd.usWidth = (ushort) init->width();

      fd.fl |= FNTS_INITFROMFATTRS;
      memcpy( &fd.fAttrs, init->fattrs().pvAddr(), sizeof( FATTRS));
   }

   // run the dialog
   BOOL rc = WinFontDlg( JWindow::theDesktopWindow, owner->handle(), &fd);
   if( !rc)
      pmError( 2, "WinFontDlg");

   // if the ok button wasn't pressed, give them the old one back or
   // system proportional if there is no old one.
   if( fd.lReturn != DID_OK)
      if( init) return *init;
      else return JFont( "System Proportional");

   // create a font object for this font
   JFont theFont( fd.fAttrs.szFacename, &deskPS);

   // need to set the point size: important to get image fonts right
   if( !theFont.outline()) {
      JFont newFont( fd.fAttrs.szFacename, fd.fxPointSize >> 16, &deskPS);
      theFont = newFont;
   } else
      theFont.fd->metrics->sNominalPointSize = (fd.fxPointSize >> 16) * 10;

   // hooray!
   return theFont;
}

// list scaffolding etc ------------------------------------------------------
// Obtain the fattrs buffer for the font -------------------------------------
JBuffer JFont::fattrs() const
{
   #define FS (*((PFATTRS) (fs.pvAddr())))

   JBuffer fs( sizeof( FATTRS));
   memset( fs.pvAddr(), 0, fs.cbSize());

   FS.usRecordLength = sizeof( FATTRS);
   if( italic())     FS.fsSelection |= FATTR_SEL_ITALIC;
   if( underlined()) FS.fsSelection |= FATTR_SEL_UNDERSCORE;
   if( hollow())     FS.fsSelection |= FATTR_SEL_OUTLINE;
   if( bold())       FS.fsSelection |= FATTR_SEL_BOLD;
   if( strikeout())  FS.fsSelection |= FATTR_SEL_STRIKEOUT;
   FS.lMatch = PFM->lMatch;
   strncpy( FS.szFacename, face(), FACESIZE);
   FS.szFacename[ FACESIZE - 1] = '\0';
   FS.idRegistry = PFM->idRegistry;
   FS.usCodePage = PFM->usCodePage;
   FS.lMaxBaselineExt = PFM->lMaxBaselineExt;
   FS.lAveCharWidth = PFM->lAveCharWidth;
   if( outline())
      FS.fsFontUse = FATTR_FONTUSE_OUTLINE;

   return fs;
}

PFONTMETRICS JFont::fontMetrics() { return PFM; }
