/*
 * JFrame.cpp
 *
 * Frame window implementation
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"
#include "JFrame.hpp"
#include "JWindow.hpp"
#include "JMother.hpp"
#include "JPM.hpp"
#include "JCoord.hpp"
#include "JIntern.hpp"
#include "JHandler.hpp"
#include "JEvent.hpp"
#include "JSWP.hpp"
#include "JSeq.hpp"
#include "JFrmeCtl.hpp"
#include "JToolbar.hpp"

#include <stdio.h>

// data area -----------------------------------------------------------------
typedef JSequence<JFrameCtl*> FrameCtlList;

struct JFrameData
{
   JWindow      *client;
   FrameCtlList frameCList;

   JFrameData() : client( 0) {}
};

// handler to manage frame extensions ----------------------------------------

class JFrameCtlHandler : public JHandler
{
 public:
   JFrameCtlHandler() : JHandler() {}

   BOOL handle( JEventData *e)
   {
      switch( e->msg) {
         case WM_QUERYFRAMECTLCOUNT:
            return queryCount( e);
         case WM_FORMATFRAME:
            return formatFrame( e);
         case WM_CALCFRAMERECT:
            return calcFrameRect( e);
         case WM_NEXTMENU:
            return nextMenu( e);
      }
      return false;
   }

   BOOL queryCount( JEventData *e)
   {
      JFrame *frame = (JFrame *) e->win;
      long    count = frame->defaultAction( e->msg);

      e->rc = JMR( count + frame->data->frameCList.elements());

      return true;
   }

   BOOL formatFrame( JEventData *e)
   {
      JFrame       *frame = (JFrame *) e->win;
      FrameCtlList &list  = frame->data->frameCList;
      long          i, count;

      // how many ctls has the def-procedure filled in?
      count = i = frame->defaultAction( e->msg, e->mp1, e->mp2);

      // how many more are we going to add?
      int extra = list.elements();

      // make a swparray of the controls the default routine filled in...
      JSWPArray defArray( (JSWP *) e->mp1(), count);
      // ...and another for us, containing all swps
      JSWPArray allSwps( (JSWP *) e->mp1(), count + extra);
      // get the client rect
      ulong ndxClient = allSwps.indexOf( JFrame::idClient);
      JRect rclClient = (*allSwps[ ndxClient])->rect();

      // for each frame extension...

      for( int j = 0; j < extra; j++, i++) {
         JSWP      *swp = *(allSwps[ i]);
         JFrameCtl *ctl = *(list[ j]);

         swp->hwnd()       = ctl->window()->handle();
         swp->hwndBehind() = HWND_TOP;
         swp->flags()      = JSWP::move | JSWP::resize;
         swp->flags()     |= (ctl->isVisible() ? JSWP::show : JSWP::hide);

         // give each of them a chance to fill in their size & position
         ctl->format( defArray, *swp, frame);

         // check the 'showme' flag in the framectl
         if( ctl->isVisible()) {
            ctl->adjustClientRect( &rclClient);
            swp = *(allSwps[ ndxClient]);
            swp->x()      = rclClient.min.x;
            swp->y()      = rclClient.min.y;
            swp->width()  = rclClient.size().x;
            swp->height() = rclClient.size().y;
         }
      }

      // fill in the return code - total number of frame controls
      e->rc = JMR( count + extra);

      // event handled Ok.
      return true;
   }

   BOOL calcFrameRect( JEventData *e)
   {
      JFrame       *frame = (JFrame *) e->win;
      FrameCtlList &list  = frame->data->frameCList;

      JRect        *rcl = (JRect *) e->mp1();
      BOOL          calcClient = e->mp2.s1();

      // first get the frame to have a go
      frame->defaultAction( e->msg, e->mp1, e->mp2);
      // now, the rectangle is almost right, except for changes for frame extns
      JFrameCtl *ctl;

      if( calcClient) { // want to calculate the client rect
         for( int i = 0; i < list.elements(); i++) {
            ctl = *(list[ i]);
            // actions are cumulative.
            if( ctl->isVisible())
               ctl->adjustClientRect( rcl);
         }
      } else {
         // we need to ADD on the extra space occupied by the controls.
         JRect rect;
         for( int i = 0; i < list.elements(); i++) {
            ctl = *(list[ i]);
            if( ctl->window()->isVisible())
               ctl->adjustClientRect( &rect);
         }

         rcl->min -= rect.min;
         rcl->max -= rect.max;
      }
      e->rc = true;
      return true;
   }

   // need to catch these to make alt-to-toolbar work
   // !! this is a very ugly hack and should be excised
   BOOL nextMenu( JEventData *e)
   {
      JFrame *frame      = (JFrame *) e->win;
      FrameCtlList &list = frame->data->frameCList;
      BOOL rc = true;

      // if a toolbar exists, we need to intervene. probably.
      FrameCtlList::cursor c( list);
      for_cursor( c)
         if( (*c.current())->getID() == JToolbar::idToolbar) break;

      if( !c.isValid()) {
         rc = false;
      } else {

         // work out what bits of menu-like regalia the window posesses
         ulong hwndSys = 0, hwndMenu = 0;
         ulong hwndTool = (*c.current())->window()->handle();

         JWindow *w = frame->childWithID( JFrame::idSysmenu);
         if( w) hwndSys = w->handle();
         w = frame->childWithID( JFrame::idMenu);
         if( w) hwndMenu = w->handle();

         ushort flStart = e->mp2.s1();

         if( e->mp1 == hwndSys) {
            if( flStart) e->rc = hwndTool;
            else e->rc = hwndMenu ? hwndMenu : hwndTool;
         } else if( e->mp1 == hwndMenu) {
            if( flStart) e->rc = hwndSys ? hwndSys : hwndTool;
            else e->rc = hwndTool;
         } else if( e->mp1 == hwndTool) {
            if( flStart)
               e->rc = hwndMenu ? hwndMenu : ( hwndSys ? hwndSys : hwndTool);
            else e->rc = hwndSys ? hwndSys : ( hwndMenu ? hwndMenu : hwndTool);
         } else
            // something kooky to do with context menu's going down.
            rc = false;
      }

      return rc;
   }
};

static JFrameCtlHandler _JLibFrameCH;

// Style ---------------------------------------------------------------------
const unsigned long JFrame::normal          = FCF_TITLEBAR | FCF_SYSMENU |
                                              FCF_SIZEBORDER | FCF_MINMAX;
const unsigned long JFrame::titleBar        = FCF_TITLEBAR;
const unsigned long JFrame::sysMenu         = FCF_SYSMENU;
const unsigned long JFrame::minButton       = FCF_MINBUTTON;
const unsigned long JFrame::maxButton       = FCF_MAXBUTTON;
const unsigned long JFrame::hideButton      = FCF_HIDEBUTTON;
const unsigned long JFrame::noByteAlign     = FCF_NOBYTEALIGN;
const unsigned long JFrame::noMoveWithOwner = FCF_NOMOVEWITHOWNER;
const unsigned long JFrame::sysModal        = FCF_SYSMODAL;
const unsigned long JFrame::screenAlign     = FCF_SCREENALIGN;
const unsigned long JFrame::mouseAlign      = FCF_MOUSEALIGN;

const ulong JFrame::idClient   = FID_CLIENT;
const ulong JFrame::idMenu     = FID_MENU;
const ulong JFrame::idMinMax   = FID_MINMAX;
const ulong JFrame::idSysmenu  = FID_SYSMENU;
const ulong JFrame::idTitlebar = FID_TITLEBAR;

// Constructors / Dtor -------------------------------------------------------
JFrame::JFrame( char *title, ulong flCreate,
                ulong ulFStyle, JWindow &parent ) : JMother(0 ),
                                                    data( new JFrameData)
{
   SWP swp;
   FRAMECDATA fcd = { sizeof( FRAMECDATA),
                      flCreate, 0, 0 };

   JCreateWBlock b( parent.handle(), WC_FRAME, title, ulFStyle, JPoint(),
                    JSize(), parent.handle(), HWND_BOTTOM, 0, &fcd, 0);

   setHwnd( JInternal::window::create( &b));

   subclass();
   setautodelete();

   // set the window's size to the next 'shell rectangle'.
   // do this here instead of with FCF_NEXTSHELLPOS bcos that doesn't allow
   // for people explicitly setting the size before the window appears.

   if( WinQueryTaskSizePos( JPM::current()->hab(), 0, &swp))
      pmError( 4, "WinQueryTaskSizePos");

   reshape( JPoint( swp.x, swp.y), JPoint( swp.cx, swp.cy));
}

JFrame::~JFrame()
{
   delete data;
}

// Client manipulation -------------------------------------------------------
JFrame &JFrame::setClient( JWindow *w)
{
   assertParms( w, "JFrame::setClient");

   data->client = w;
   data->client->setID( JFrame::idClient);

   // move the window to a sensible place.
   JRect rclClient = clientRect();
   data->client->moveTo( fromDesktopCoords( rclClient.min));
   data->client->resize( rclClient.size());

   return self;
}

JWindow *JFrame::client() const
{
   return data->client;
}

// Frame extensions ----------------------------------------------------------
JFrame &JFrame::addFrameExtension( JFrameCtl *ctl)
{
   data->frameCList.append( ctl);

   if( data->frameCList.elements() == 1)
      attach( &_JLibFrameCH);

   return self;
}

JFrame &JFrame::showFrameExtension( ulong id, BOOL shew)
{
   FrameCtlList::cursor c( data->frameCList);

   for_cursor( c)
      if( (*c.current())->getID() == id) break;

   if( c.isValid()) {
      (*c.current())->window()->show( shew);
      (*c.current())->setVisible( shew);
   }
   return updateFrameExtensions();
}

JFrame &JFrame::removeFrameExtension( ulong id)
{
   FrameCtlList::cursor c( data->frameCList);

   for_cursor( c)
      if( (*c.current())->getID() == id) break;

   data->frameCList.removeAt( c);

   return updateFrameExtensions();
}

JFrame &JFrame::updateFrameExtensions()
{
   sendEvent( WM_UPDATEFRAME);
   return self;
}

// overridden to get at extensions -------------------------------------------
// !! is this actually necessary?  Surely a frame control *is* a child of
// !! the frame, and so things work normally ?
JWindow *JFrame::childWithID( ushort id)
{
   JWindow *rc = 0;

   if( data->frameCList.elements()) {
      FrameCtlList::cursor c( data->frameCList);
      for_cursor( c)
         if( (*c.current())->getID() == id) {
            rc = (*c.current())->window(); break;
         }
   }

   if( !rc) rc = JWindow::childWithID( id);

   return rc;
}

// Set the mother's maxSize so the client is nice ----------------------------
JFrame &JFrame::setMaxClientSize( const JSize &sz)
{
   JSize oldSzFrame = size();

   sizeClientTo( sz);

   JSize szFrameFurniture = size() - clientRect().size();

   setMaximumSize( sz + szFrameFurniture);
   resize( oldSzFrame);
   return self;
}
