/*
 * JGButton.cpp
 *
 * Graphical pushbutton controls
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"

#ifdef __EMX__
// emx is a bit confused about which bits of it define what types.
#define INCL_OS2MM
#define GBS_MINIBUTTON 0x0001
#define GBN_SETFOCUS   0x0527
#include <os2me.h>
#define INCL_GRAPHICBUTTON
#include <sw.h>
#else
// vacpp isn't
#define INCL_SW
#include <os2me.h>
#endif

#include "JGButton.hpp"
#include "JIntern.hpp"
#include "JBuffer.hpp"
#include "JCtlH.hpp"

// constants, style ----------------------------------------------------------
const ulong JGButton::pushbutton   = 0;
const ulong JGButton::twostate     = GBS_TWOSTATE;
const ulong JGButton::autoTwostate = GBS_AUTOTWOSTATE;
const ulong JGButton::animated     = GBS_ANIMATION;
const ulong JGButton::autoAnimated = GBS_AUTOANIMATION;
const ulong JGButton::mini         = GBS_MINIBUTTON;

// constructor ---------------------------------------------------------------
JGButton::JGButton( JWindow *parent, const JPoint &pos, const JSize &size,
                    ulong id, const char *text, ulong style,
                    JGButton::textStyle t) : JControl()
{
   assertParms( parent, "JGButton::ctor");

   style |= JWindow::visible | GBS_HILITEBITMAP | GBS_DISABLEBITMAP;
   if( t == recessed) style |= GBS_3D_TEXTRECESSED;
   else if( t == embossed) style |= GBS_3D_TEXTRAISED;

   GBTNCDATA ctldata = { GB_STRUCTURE, (char*)text, 0, 0, 0 };

   JCreateWBlock b( parent->handle(), WC_GRAPHICBUTTON, (char*)text,
                    style, pos, size, parent->handle(),
                    HWND_TOP, id, &ctldata, 0);

   setHwnd( JInternal::window::create( &b));
}

// setting bitmap data -------------------------------------------------------
JGButton &JGButton::setBitmaps( ushort *usIDs, ulong cIDs, const JModule &mod)
{
   JTBuffer<GBTNCDATA> b( sizeof( GBTNCDATA) + cIDs * sizeof( ushort));
   JStr txt( getText());

   b[ 0].usReserved = GB_STRUCTURE;
   b[ 0].pszText = txt;
   b[ 0].hmod = mod.handle();
   b[ 0].cBitmaps = cIDs;
   memcpy( &(b[ 0].aidBitmap), usIDs, sizeof( ushort) * cIDs);

   BOOL rc = sendEvent( GBM_SETGRAPHICDATA, b());
   if( !rc) pmError( 2, "GBM_SETGD");

   return self;
}

JGButton &JGButton::setBitmaps( JSequence<ushort> &ids, const JModule &mod)
{
   ulong   cIDs = ids.elements();
   ushort *usIDs = new ushort[ cIDs];

   for( ulong i = 0; i < cIDs; i++)
      usIDs[ i] = *(ids[ i]);

   setBitmaps( usIDs, cIDs, mod);

   delete [] usIDs;

   return self;
}

JGButton &JGButton::setBitmap( JGButton::buttonState which, ulong index)
{
   BOOL rc = sendEvent( GBM_SETBITMAPINDEX, which, index);
   if( !rc) pmError( 2, "SETBMPNDX");
   return self;
}

// animation control ---------------------------------------------------------
JGButton &JGButton::startAnimation( BOOL startFromCurrent)
{
   BOOL rc = sendEvent( GBM_ANIMATE, true, startFromCurrent);
   if( !rc) pmError( 2, "GBM_ANIMATE");
   return self;
}

JGButton &JGButton::stopAnimation()
{
   BOOL rc = sendEvent( GBM_ANIMATE, false);
   if( !rc) pmError( 1, "GBM_ANIMATE2");
   return self;
}

BOOL JGButton::isAnimating() const
{
   return sendEvent( GBM_QUERYANIMATIONACTIVE);
}

ulong JGButton::getFrameDelay() const
{
   return sendEvent( GBM_QUERYANIMATIONRATE);
}

JGButton &JGButton::setFrameDelay( ulong ms)
{
   BOOL rc = sendEvent( GBM_SETANIMATIONRATE, ms);
   if( !rc) pmError( 4, "SETANIMRATE");
   return self;
}

// text style ----------------------------------------------------------------
JGButton::textStyle JGButton::getTextPos() const
{
   textStyle rc = flat;

   if( hasStyle( GBS_3D_TEXTRECESSED)) rc = recessed;
   else if( hasStyle( GBS_3D_TEXTRAISED)) rc = embossed;

   return rc;
}

JGButton &JGButton::setTextPos( JGButton::textStyle ts)
{
   ulong mask = ~(GBS_3D_TEXTRECESSED | GBS_3D_TEXTRAISED);
   ulong style = WinQueryWindowULong( handle(), QWL_STYLE);

   style &= mask;
   if( ts == recessed) style |= GBS_3D_TEXTRECESSED;
   else if( ts == embossed) style |= GBS_3D_TEXTRAISED;

   WinSetWindowULong( handle(), QWL_STYLE, style);

   return self;
}

// state ---------------------------------------------------------------------
JGButton::buttonState JGButton::getCurrentState() const
{
   ulong rc = sendEvent( GBM_QUERYSTATE);
   return (buttonState) rc;
}

JGButton &JGButton::setCurrentState( JGButton::buttonState b)
{
   BOOL rc = sendEvent( GBM_SETSTATE, b, true);
   if( !rc) pmError( 3, "GBM_SETSTATE");
   return self;
}

// events (my dear boy) ------------------------------------------------------
BOOL JGButton::event( const JCtlEvent &e)
{
   BOOL rc = false;

   switch( e.notify()) {
      case GBN_BUTTONUP:
         rc = buttonUp();
         break;
      case GBN_BUTTONDOWN:
         rc = buttonDown();
         break;
      case GBN_BUTTONHILITE:
         rc = buttonHilighted();
         break;
      case GBN_SETFOCUS:
         if( e.data())
            rc = lostFocus();
         else
            rc = gainedFocus();
         break;
   }

   return rc;
}
