/*
 * JGLPrim.cpp
 *
 * 'Logical' graphics primitives - settings objects
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOs2.h"
#include "JGLPrim.hpp"
#include "JColour.hpp"
#include "JPSpace.hpp"
#include "JBitmap.hpp"

// A pointer to class Bundle is kept in the base settings class, which the
// class can access via a virtual method. Classes derived from Bundle are
// stored by descendants of Settings, which overload the access method so
// that the base class transparently updates the correct structure.

// This is all a bit sordid.

// Base graphics settings class ----------------------------------------------
struct JBundle
{
   ulong  fc;
   ulong  bc;
   ushort fm;
   ushort bm;
   JBundle() : fc( JColour::black.asULong()), bc( JColour::white.asULong()),
               fm( JGfx::fgMix::normal), bm( JGfx::bgMix::normal) {}
};

JBundle *JGSettings::bundle() const
{ return baseB; }

JGSettings::JGSettings() : baseB( new JBundle), gstats( 0)
{}

JGSettings::~JGSettings()
{
   delete baseB;
}

// Mix and color setting
JGSettings &JGSettings::setFGMix( JGfx::fgMix::type t)
{ bundle()->fm = t; gstats |= 4; return self; }

JGfx::fgMix::type JGSettings::fgMix() const
{ return (JGfx::fgMix::type) bundle()->fm; }

JGSettings &JGSettings::setBGMix( JGfx::bgMix::type t)
{ bundle()->bm = t; gstats |= 8; return self; }

JGfx::bgMix::type JGSettings::bgMix() const
{ return (JGfx::bgMix::type) bundle()->bm; }

JGSettings &JGSettings::setFGColour( const JColour &c)
{ bundle()->fc = c.asULong(); gstats |= 1; return self; }

JColour JGSettings::fgColour() const
{ return JColour( bundle()->fc); }

JGSettings &JGSettings::setBGColour( const JColour &c)
{ bundle()->bc = c.asULong(); gstats |= 2; return self; }

JColour JGSettings::bgColour() const
{ return JColour( bundle()->bc); }

// rendering & defaults
JGSettings &JGSettings::makeDefaultsFor( JBPSpace &ps)
{
   // can't do this - just set the current
   renderIn( ps);
   return self;
}

BOOL JGSettings::renderIn( JBPSpace &ps) const
{
   BOOL rc = true;
   if( gstats & 1)
      rc &= GpiSetColor( ps.handle(), bundle()->fc);
   if( gstats & 2)
      rc &= GpiSetBackColor( ps.handle(), bundle()->bc);
   if( gstats & 4)
      rc &= GpiSetMix( ps.handle(), bundle()->fm);
   if( gstats & 8)
      rc &= GpiSetBackMix( ps.handle(), bundle()->bm);

   if( !rc)
      pmError( 1, "GpiKickTheCat");
   return false;
}

// Line settings -------------------------------------------------------------
struct JLineBundle : public JBundle
{
   ulong  width; // fixed
   long   gWidth;
   ushort type;
   ushort end;
   ushort join;
   ushort null;

   JLineBundle() : JBundle(),
                   width( JGfx::line::width::normal << 16),
                   gWidth( 0),
                   type( JGfx::line::type::normal),
                   end( JGfx::line::end::normal),
                   join( JGfx::line::join::normal),
                   null( 0)
   {}
};

JBundle *JLineSettings::bundle() const
{ return b; }

JLineSettings::JLineSettings() : JGSettings(), b( new JLineBundle)
{}

JLineSettings::JLineSettings( void *v, ulong m) : JGSettings(),
                                                  b( new JLineBundle)
{
   PLINEBUNDLE lb = (PLINEBUNDLE) v;
   gstats = m;
   if( m & LBB_COLOR) b->fc          = lb->lColor;
   if( m & LBB_BACK_COLOR) b->bc     = lb->lBackColor;
   if( m & LBB_MIX_MODE) b->fm       = lb->usMixMode;
   if( m & LBB_BACK_MIX_MODE) b->bm  = lb->usBackMixMode;
   if( m & LBB_WIDTH) b->width       = lb->fxWidth;
   if( m & LBB_GEOM_WIDTH) b->gWidth = lb->lGeomWidth;
   if( m & LBB_TYPE) b->type         = lb->usType;
   if( m & LBB_END) b->end           = lb->usEnd;
   if( m & LBB_JOIN) b->join         = lb->usJoin;
}

JLineSettings::~JLineSettings()
{ delete b; }

JLineSettings &JLineSettings::setEnds( JGfx::line::end::type x)
{ b->end = x; gstats |= LBB_END; return self; }

JGfx::line::end::type JLineSettings::ends() const
{ return (JGfx::line::end::type) b->end; }

JLineSettings &JLineSettings::setJoin( JGfx::line::join::type x)
{ b->join = x; gstats |= LBB_JOIN; return self; }

JGfx::line::join::type JLineSettings::join() const
{ return (JGfx::line::join::type) b->join; }

JLineSettings &JLineSettings::setType( JGfx::line::type::typ x)
{ b->type = x; gstats |= LBB_TYPE; return self; }

JGfx::line::type::typ JLineSettings::type() const
{ return (JGfx::line::type::typ) b->type; }

JLineSettings &JLineSettings::setLook( JGfx::line::width::type x)
{ b->width = x << 16; gstats |= LBB_WIDTH; return self; }

JGfx::line::width::type JLineSettings::look() const
{ return (JGfx::line::width::type) (b->width >> 16); }

JLineSettings &JLineSettings::setWidth( long l)
{ b->gWidth = l; gstats |= LBB_GEOM_WIDTH; return self; }

long JLineSettings::width() const
{ return b->gWidth; }

JLineSettings &JLineSettings::makeDefaultsFor( JBPSpace &ps)
{
   BOOL rc = GpiSetDefAttrs( ps.handle(), PRIM_LINE, gstats, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetDefAttrs");
   return self;
}

BOOL JLineSettings::renderIn( JBPSpace &ps) const
{
   BOOL rc = GpiSetAttrs( ps.handle(), PRIM_LINE, gstats, 0, (PBUNDLE) b);
   if( !rc)
      pmError( 2, "GpiSetAttrs");
   return false;
}

// Image Settings ------------------------------------------------------------
JImageSettings::JImageSettings() : JGSettings()
{}

JImageSettings::JImageSettings( void *v, ulong l) : JGSettings()
{
   PIMAGEBUNDLE ib = (PIMAGEBUNDLE) v;
   gstats = l;
   if( l & IBB_COLOR) baseB->fc         = ib->lColor;
   if( l & IBB_BACK_COLOR) baseB->bc    = ib->lBackColor;
   if( l & IBB_MIX_MODE) baseB->fm      = ib->usMixMode;
   if( l & IBB_BACK_MIX_MODE) baseB->bm = ib->usBackMixMode;
}

JImageSettings &JImageSettings::makeDefaultsFor( JBPSpace &ps)
{
   BOOL rc = GpiSetDefAttrs( ps.handle(), PRIM_IMAGE, gstats, (PBUNDLE) baseB);
   if( !rc)
      pmError( 3, "GpiSetDefAttrs");
   return self;
}

BOOL JImageSettings::renderIn( JBPSpace &ps) const
{
   BOOL rc = GpiSetAttrs( ps.handle(), PRIM_IMAGE, gstats, 0, (PBUNDLE) baseB);
   if( !rc)
      pmError( 2, "GpiSetAttrs");
   return false;
}

// Marker Settings -----------------------------------------------------------
struct JMarkerBundle : public JBundle
{
   ushort setID;
   ushort symbol;
   ulong  xCell;
   ulong  yCell;
   JMarkerBundle() : setID( 0), symbol( JGfx::marker::normal),
                     xCell( 16 << 16), yCell( 16 << 16), JBundle() {}
};

JBundle *JMarkerSettings::bundle() const
{ return b; }

JMarkerSettings::JMarkerSettings() : JGSettings(), b( new JMarkerBundle),
                                     f( "System Proportional")
{}

JMarkerSettings::JMarkerSettings( void *v, ulong m) : JGSettings(),
                                                      b( new JMarkerBundle),
                                                      f( "System Proportional")
{
   PMARKERBUNDLE mb = (PMARKERBUNDLE) v;
   gstats = m;

   if( m & MBB_COLOR) b->fc = mb->lColor;
   if( m & MBB_BACK_COLOR) b->bc = mb->lBackColor;
   if( m & MBB_MIX_MODE) b->fm = mb->usMixMode;
   if( m & MBB_BACK_MIX_MODE) b->bm = mb->usBackMixMode;
   if( m & MBB_SET) b->setID = mb->usSet;
   if( m & MBB_SYMBOL) b->symbol = mb->usSymbol;
   if( m & MBB_BOX) {
      b->xCell = mb->sizfxCell.cx;
      b->yCell = mb->sizfxCell.cy;
   }
}

JMarkerSettings::~JMarkerSettings()
{ delete b; }

JMarkerSettings &JMarkerSettings::setSymbol( JGfx::marker::sym s)
{ b->symbol = s; gstats |= MBB_SYMBOL; return self; }

JGfx::marker::sym JMarkerSettings::symbol() const
{ return (JGfx::marker::sym) b->symbol; }

JMarkerSettings &JMarkerSettings::setSize( const JSize &s)
{
   b->xCell = s.x << 16;
   b->yCell = s.y << 16;
   gstats |= MBB_BOX;
   return self;
}

JSize JMarkerSettings::size() const
{ return JSize( b->xCell >> 16, b->yCell >> 16); }

JMarkerSettings &JMarkerSettings::setFont( const JFont &fnt)
{ f = fnt; gstats |= MBB_SET; return self; }

JMarkerSettings &JMarkerSettings::makeDefaultsFor( JBPSpace &ps)
{
   if( gstats & MBB_SET)
      b->setID = ps.openFont( f);

   BOOL rc = GpiSetDefAttrs( ps.handle(), PRIM_MARKER, gstats, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetDefAttrs");
   return self;
}

BOOL JMarkerSettings::renderIn( JBPSpace &ps) const
{
   if( gstats & MBB_SET)
      b->setID = ps.openFont( f);

   BOOL rc = GpiSetAttrs( ps.handle(), PRIM_MARKER, gstats, 0, (PBUNDLE) b);
   if( !rc)
      pmError( 2, "GpiSetAttrs");
   return false;
}

// Area - fill - settings ----------------------------------------------------
struct JFillBundle : public JBundle
{
   ushort setID;
   ushort symbol;
   JPoint orig;
   JFillBundle() : setID( 0), symbol( JGfx::pattern::normal), JBundle() {}
};

JBundle *JFillSettings::bundle() const
{ return b; }

JFillSettings::JFillSettings() : b( new JFillBundle), JGSettings()
{}

JFillSettings::JFillSettings( void *v, ulong l) : b( new JFillBundle),
                                                  JGSettings()
{
   PAREABUNDLE ab = (PAREABUNDLE) v;
   gstats = l;

   if( l & ABB_COLOR) b->fc = ab->lColor;
   if( l & ABB_BACK_COLOR) b->bc = ab->lBackColor;
   if( l & ABB_MIX_MODE) b->fm = ab->usMixMode;
   if( l & ABB_BACK_MIX_MODE) b->bm = ab->usBackMixMode;
   if( l & ABB_SET) b->setID = ab->usSet;
   if( l & ABB_SYMBOL) b->symbol = ab->usSymbol;
   if( l & ABB_REF_POINT) {
      b->orig.x = ab->ptlRefPoint.x;
      b->orig.y = ab->ptlRefPoint.y;
   }
}

JFillSettings &JFillSettings::setPattern( JGfx::pattern::sym t)
{ b->symbol = t; gstats |= ABB_SYMBOL; return self; }

JFillSettings &JFillSettings::setPattern( const JBitmap &b)
{ hb = b.handle(); gstats |= ABB_SET; return self; }

JGfx::pattern::sym JFillSettings::pattern() const
{ return (JGfx::pattern::sym) b->symbol; }

JFillSettings &JFillSettings::setPatternOrigin( const JPoint &p)
{ b->orig = p; gstats |= ABB_REF_POINT; return self; }

JPoint JFillSettings::patternOrigin() const
{ return b->orig; }

JFillSettings &JFillSettings::makeDefaultsFor( JBPSpace &ps)
{
   if( gstats & ABB_SET) {
      b->setID = ps.newID();
      ulong rc = GpiSetBitmapId( ps.handle(), hb, b->setID);
      if( rc == GPI_ERROR)
         pmError( 3, "GpiSetBitmapId");
   }

   BOOL rc = GpiSetDefAttrs( ps.handle(), PRIM_AREA, gstats, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetDefAttrs");
   return self;
}

BOOL JFillSettings::renderIn( JBPSpace &ps) const
{
   if( gstats & ABB_SET) {
      b->setID = ps.newID();
      ulong rc = GpiSetBitmapId( ps.handle(), hb, b->setID);
      if( rc == GPI_ERROR)
         pmError( 3, "GpiSetBitmapId");
   }

   BOOL rc = GpiSetAttrs( ps.handle(), PRIM_AREA, gstats, 0, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetAttrs");
   return false;
}

// Text settings -------------------------------------------------------------
struct JTextBundle : public JBundle
{
  ushort setID;
  ushort precision;
  JSize  size;       // fixed
  JPoint angle;
  JPoint shear;
  ushort direction;
  ushort alignment;
  ulong  extra;      // fixed
  ulong  breakExtra; // fixed
  JTextBundle() : setID( 0), precision( 0), direction( 0), alignment( 0),
                  extra( 0), breakExtra( 0), JBundle() {}
};

JBundle *JTextSettings::bundle() const
{ return b; }

JTextSettings::JTextSettings() : b( new JTextBundle), f( "System Proportional"),
                                 JGSettings()
{}

JTextSettings::JTextSettings( void *v, ulong m) : b( new JTextBundle),
                                                  f( "System Proportional"),
                                                  JGSettings()
{
   PCHARBUNDLE pb = (PCHARBUNDLE) v;
   gstats = m;
   if( m & CBB_COLOR) b->fc = pb->lColor;
   if( m & CBB_BACK_COLOR) b->bc = pb->lBackColor;
   if( m & CBB_MIX_MODE) b->fm = pb->usMixMode;
   if( m & CBB_BACK_MIX_MODE) b->bm = pb->usBackMixMode;
   if( m & CBB_SET) b->setID = pb->usSet;
   if( m & CBB_MODE) b->precision = pb->usPrecision;
   if( m & CBB_BOX) {
      b->size.x = pb->sizfxCell.cx >> 16;
      b->size.y = pb->sizfxCell.cy >> 16;
   }
   if( m & CBB_ANGLE) {
      b->angle.x = pb->ptlAngle.x;
      b->angle.y = pb->ptlAngle.y;
   }
   if( m & CBB_SHEAR) {
      b->shear.x = pb->ptlShear.x;
      b->shear.y = pb->ptlShear.y;
   }
   if( m & CBB_DIRECTION) b->direction = pb->usDirection;
   if( m & CBB_TEXT_ALIGN) b->alignment = pb->usTextAlign;
   if( m & CBB_EXTRA) b->extra = pb->fxExtra;
   if( m & CBB_BREAK_EXTRA) b->breakExtra = pb->fxBreakExtra;
}

JTextSettings &JTextSettings::setGradient( const JPoint &p)
{ b->angle = p; gstats |= CBB_ANGLE; return self; }

JPoint JTextSettings::gradient() const
{ return b->angle; }

JTextSettings &JTextSettings::setShear( const JPoint &p)
{ b->shear = p; gstats |= CBB_SHEAR; return self; }

JPoint JTextSettings::shear() const
{ return b->shear; }

JTextSettings &JTextSettings::setFont( const JFont &fnt)
{ f = fnt; gstats |= CBB_SET; return self; }

JTextSettings &JTextSettings::setSize( const JSize &s)
{
   b->size.x = s.x << 16;
   b->size.y = s.y << 16;
   gstats |= CBB_BOX;
   return self;
}

JSize JTextSettings::size() const
{ return b->size; }

JTextSettings &JTextSettings::makeDefaultsFor( JBPSpace &ps)
{
   if( gstats & CBB_SET)
      b->setID = ps.openFont( f);

   BOOL rc = GpiSetDefAttrs( ps.handle(), PRIM_CHAR, gstats, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetDefAttrs");
   return self;
}

BOOL JTextSettings::renderIn( JBPSpace &ps) const
{
   if( gstats & CBB_SET)
      b->setID = ps.openFont( f);

   BOOL rc = GpiSetAttrs( ps.handle(), PRIM_CHAR, gstats, 0, (PBUNDLE) b);
   if( !rc)
      pmError( 3, "GpiSetAttrs");
   return false;
}

// Arc Settings --------------------------------------------------------------
JArcSettings::JArcSettings() : JGObject(), p( 1), q( 1), r( 0), s( 0)
{}

JArcSettings::JArcSettings( void *v) : JGObject()
{
   p = ((PARCPARAMS) v)->lP;
   q = ((PARCPARAMS) v)->lQ;
   r = ((PARCPARAMS) v)->lR;
   s = ((PARCPARAMS) v)->lS;
}

JPoint JArcSettings::majorPoint() const
{ return JPoint( p, s); }

JPoint JArcSettings::minorPoint() const
{ return JPoint( r, q); }

JArcSettings &JArcSettings::setMajorPoint( const JPoint &pt)
{ p = pt.x; s = pt.y; return self; }

JArcSettings &JArcSettings::setMajorPoint( long x, long y)
{ p = x; s = y; return self; }

JArcSettings &JArcSettings::setMinorPoint( const JPoint &pt)
{ r = pt.x; q = pt.y; return self; }

JArcSettings &JArcSettings::setMinorPoint( long x, long y)
{ r = x; q = y; return self; }

JArcSettings &JArcSettings::makeDefaultsFor( JBPSpace &ps)
{
   ARCPARAMS arcParams = { p, q, r, s };
   BOOL rc = GpiSetDefArcParams( ps.handle(), &arcParams);
   if( !rc)
      pmError( 3, "GpiSetDefArcParams");
   return self;
}

BOOL JArcSettings::renderIn( JBPSpace &ps) const
{
   ARCPARAMS arcParams = { p, q, r, s };
   BOOL rc = GpiSetArcParams( ps.handle(), &arcParams);
   if( !rc)
      pmError( 2, "GpiSetArcParams");
   return false;
}
