/*
 * JGPrim.cpp
 *
 * Physical graphics primitives
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOs2.h"
#include "JGPrim.hpp"
#include "JPSpace.hpp"
#include "JBitmap.hpp"

#include <math.h>

#define PI 3.1415926535897932384626433833
#define DEGREE 0.017453292519943295769236907684

// Macro for Gpi* error reporting ---------------------------------------------
#define GPIRC( a, b) if( a == GPI_ERROR)      \
                        pmError( 3, b);       \
                     return( a == GPI_HITS);

// Line object ---------------------------------------------------------------
BOOL JLine::renderIn( JBPSpace &p) const
{
   long rc = GpiLine( p.handle(), pt);
   GPIRC( rc, "GpiLine");
}

// Move object (Not popable) --------------------------------------------------
BOOL JMove::renderIn( JBPSpace &p) const
{
   long rc = GpiMove( p.handle(), pt);
   GPIRC( rc, "GpiMove");
}

// Box ------------------------------------------------------------------------
BOOL JBox::renderIn( JBPSpace &p) const
{
   JPoint curPos;
   long rc = GpiQueryCurrentPosition( p.handle(), curPos);
   rc = GpiBox( p.handle(), opts, curPos + sz, 0, 0);
   GPIRC( rc, "GpiBox");
}

// Polygons -------------------------------------------------------------------
JPolygons::JPolygons( const JPolygonList &l, BOOL outline) : JGObject()
{
   // plan here is to have an array of ulongs. First is cPolygons.
   // Second is flags. From third is an array of POLYGON structures, ie.
   // alternating cPoints, papointl.
   pgons = new ulong [ 2 + l.elements() * 2];
   pgons[ 0] = l.elements();
   pgons[ 1] = outline ? POLYGON_BOUNDARY : POLYGON_NOBOUNDARY;

   JPolygonList::cursor c( l);

   // for each polygon in the list...
   int i = 0;
   for( c.toFirst(); c.isValid(); c.toNext(), i++) {
      // ...record how many points are in it...
      pgons[ 2 + 2 * i] = (*c.current())->elements();
      // ...get some memory to hold the points data...
      long *aPolygons = new long [ 2 * pgons[ 2 + 2 * i]];
      // ...for each point in the polygon...
      for( ulong j = 0; j < pgons[ 2 + 2 * i]; j++) {
         // ...record its position.
         aPolygons[ j * 2 + 0] = (*c.current())->elementAt( j)->x;
         aPolygons[ j * 2 + 1] = (*c.current())->elementAt( j)->y;
      }
      // record the point array
      pgons[ 2 + 2 * i + 1] = (ulong) aPolygons;
   }
}

JPolygons::~JPolygons()
{
   for( ulong i = 0; i < pgons[ 0]; i++)
      delete [] (JPoint *) pgons[ 2 + 2 * i + 1]; // umm...
   delete [] pgons;
}

BOOL JPolygons::renderIn( JBPSpace &p) const
{
   long rc = GpiPolygons( p.handle(), pgons[ 0], (PPOLYGON) (pgons + 2),
                          pgons[ 1], POLYGON_INCL);
   GPIRC( rc, "GpiPolygons");
}

// Polylines ------------------------------------------------------------------
JPolyLine::JPolyLine( const JPointList &pnts) : JGObject()
{
   // alloc some memory to hold an array of points & the size
   points = new ulong [ pnts.elements() * 2 + 1];
   points[ 0] = pnts.elements();
   // for each point...
   for( ulong i = 0; i < points[ 0]; i++) {
      // ...record it in the array
      points[ 1 + i * 2 + 0] = pnts.elementAt( i)->x;
      points[ 1 + i * 2 + 1] = pnts.elementAt( i)->y;
   }
}

BOOL JPolyLine::renderIn( JBPSpace &ps) const
{
   long rc = GpiPolyLine( ps.handle(), points[ 0], (PPOINTL) (points + 1));
   GPIRC( rc, "GpiPolyLine");
}

JPolyLines::JPolyLines( const JPointList &pnts) : JGObject()
{
   // alloc some memory to hold an array of points & the size
   points = new ulong [ pnts.elements() * 2 + 1];
   points[ 0] = pnts.elements();
   // for each point...
   for( ulong i = 0; i < points[ 0]; i++) {
      // ...record it in the array
      points[ 1 + i * 2 + 0] = pnts.elementAt( i)->x;
      points[ 1 + i * 2 + 1] = pnts.elementAt( i)->y;
   }
}

BOOL JPolyLines::renderIn( JBPSpace &ps) const
{
   long rc = GpiPolyLineDisjoint( ps.handle(), points[ 0], (PPOINTL) (points + 1));
   GPIRC( rc, "GpiPolyLineDisjoint");
}

// Markers --------------------------------------------------------------------
BOOL JMarker::renderIn( JBPSpace &ps) const
{
   long rc = GpiMarker( ps.handle(), pt);
   GPIRC( rc, "GpiMarker");
}

JMarkers::JMarkers( const JPointList &pnts) : JGObject()
{
   // alloc some memory to hold an array of points & the size
   points = new ulong [ pnts.elements() * 2 + 1];
   points[ 0] = pnts.elements();
   // for each point...
   for( ulong i = 0; i < points[ 0]; i++) {
      // ...record it in the array
      points[ 1 + i * 2 + 0] = pnts.elementAt( i)->x;
      points[ 1 + i * 2 + 1] = pnts.elementAt( i)->y;
   }
}

BOOL JMarkers::renderIn( JBPSpace &ps) const
{
   long rc = GpiPolyMarker( ps.handle(), points[ 0], (PPOINTL) (points + 1));
   GPIRC( rc, "GpiPolyMarker");
}

// Text -----------------------------------------------------------------------
JText::JText( char *t, BOOL underline, BOOL strikeout) : JGObject(),
                                                         txt( t), opts( 0)
{
   if( underline) opts |= CHS_UNDERSCORE;
   if( strikeout) opts |= CHS_STRIKEOUT;
}

BOOL JText::underline() const { return opts & CHS_UNDERSCORE; }
BOOL JText::strikeout() const { return opts & CHS_STRIKEOUT; }
void JText::underline( BOOL u) { if( u) opts |= CHS_UNDERSCORE;
                                 else opts &= ~CHS_UNDERSCORE; }
void JText::strikeout( BOOL s) { if( s) opts |= CHS_STRIKEOUT;
                                 else opts &= ~CHS_STRIKEOUT; }

BOOL JText::renderIn( JBPSpace &ps) const
{
   long rc = GpiCharStringPos( ps.handle(), NULL, opts, txt.length() - 1,
                               (PCH) ((PCSZ) txt), NULL);
   GPIRC( rc, "GpiCharStringPos");
}

// Labels ---------------------------------------------------------------------
BOOL JLabel::renderIn( JBPSpace &ps) const
{
   BOOL rc = GpiLabel( ps.handle(), l);
   if( !rc)
      pmError( 3, "GpiLabel");
   return false;
}

// Comments -------------------------------------------------------------------
BOOL JComment::renderIn( JBPSpace &ps) const
{
   BOOL rc = GpiComment( ps.handle(), dat.cbSize(), (PBYTE) dat.pvAddr());
   if( !rc)
      pmError( 1, "GpiComment");
   return false;
}

// Triple-point arcs ----------------------------------------------------------
BOOL JArc::renderIn( JBPSpace &ps) const
{
   long rc = GpiPointArc( ps.handle(), *pts);
   GPIRC( rc, "GpiPartialArc");
}

// 'Partial' arcs -------------------------------------------------------------
BOOL JLineArc::renderIn( JBPSpace &ps) const
{
   long rc = GpiPartialArc( ps.handle(), pt, (FIXED) rad << 16,
                            (FIXED) st << 16, (FIXED) sw << 16);
   GPIRC( rc, "GpiPartialArc");
}

// 'Full' arcs ----------------------------------------------------------------
BOOL JCircle::renderIn( JBPSpace &ps) const
{
   long rc = GpiFullArc( ps.handle(), opts, rad << 16);
   GPIRC( rc, "GpiFullArc");
}

// Special oval figure --------------------------------------------------------
JEllipse::JEllipse( ulong major, ulong minor, ulong angle, BOOL o, BOOL f)
         : JGObject(), opts( 0)
{
   if( o) opts += 2;
   if( f) opts++;

   double radA = angle * DEGREE;
   double cosA = cos( radA);
   double sinA = sin( radA);

   arcs.setMajorPoint( JPoint( (long)(major * cosA), (long)(major * sinA)));
   arcs.setMinorPoint( JPoint( (long)(-(minor * sinA)), (long)(minor * cosA)));
}

BOOL JEllipse::renderIn( JBPSpace &ps) const
{
   JArcSettings old = ps.arcSettings();
   ps.render( arcs);
   long rc = GpiFullArc( ps.handle(), opts, 1 << 16);
   ps.render( old);
   GPIRC( rc, "GpiFullArc");
}

// Fillets --------------------------------------------------------------------
JFillet::JFillet( const JPointList &pnts) : JGObject()
{
   assertParms( pnts.elements() > 1, "JFillet::JFillet");

   // alloc some memory to hold an array of points & the size
   points = new ulong [ 2 * pnts.elements() + 1];
   points[ 0] = pnts.elements();
   // for each point...
   for( ulong i = 0; i < points[ 0]; i++) {
      // ...record it in the array
      points[ 1 + i * 2 + 0] = pnts.elementAt( i)->x;
      points[ 1 + i * 2 + 1] = pnts.elementAt( i)->y;
   }
}

BOOL JFillet::renderIn( JBPSpace &ps) const
{
   long rc = GpiPolyFillet( ps.handle(), points[ 0], (PPOINTL) (points+1));
   GPIRC( rc, "GpiPolyFillet");
}

// Splines --------------------------------------------------------------------
JSpline::JSpline( JPointList &pnts) : JGObject()
{
   assertParms( !( pnts.elements() % 3), "JSpline::JSpline");

   // alloc some memory to hold an array of points & the size
   points = new ulong [ 2 * pnts.elements() + 1];
   points[ 0] = pnts.elements();
   // for each point...
   for( ulong i = 0; i < points[ 0]; i++) {
      // ...record it in the array
      points[ 1 + i * 2 + 0] = pnts.elementAt( i)->x;
      points[ 1 + i * 2 + 1] = pnts.elementAt( i)->y;
   }
}

BOOL JSpline::renderIn( JBPSpace &ps) const
{
   long rc = GpiPolySpline( ps.handle(), points[ 0], (PPOINTL) (points+1));
   GPIRC( rc, "GpiPolySpline");
}

// Blitted images -------------------------------------------------------------
JBmpImage::JBmpImage( const JBPSpace &ps, const JRect &src,
                      const JRect &dst, JGfx::rop::type t)
          : JGObject(), hThing( ps.handle()), rop( t), useWC( false)
{
   if( dst == JRect()) {
      pts[ 0] = JPoint();
      pts[ 1] = src.size();
   } else {
      pts[ 0] = dst.min;
      pts[ 1] = dst.max;
   }
   pts[ 2] = src.min;
   pts[ 3] = src.max;
}

JBmpImage::JBmpImage( const JBitmap &bmp, const JRect &src,
                      const JRect &dst, JGfx::rop::type t)
          : JGObject(), hThing( bmp.handle()), rop( t), useWC( true)
{
   if( src == JRect()) {
      pts[ 2] = JPoint();
      pts[ 3] = bmp.size();
   } else {
      pts[ 2] = src.min;
      pts[ 3] = src.max;
   }

   if( dst == JRect()) {
      pts[ 0] = JPoint();
      pts[ 1] = pts[ 3];
   } else {
      pts[ 0] = dst.min;
      pts[ 1] = dst.max;
   }
}

BOOL JBmpImage::renderIn( JBPSpace &ps) const
{
   long rc;
   if( useWC)
      rc = GpiWCBitBlt( ps.handle(), hThing, 4, *pts, rop, BBO_IGNORE);
   else
      rc = GpiBitBlt( ps.handle(), hThing, 4, *pts, rop, BBO_IGNORE);

   GPIRC( rc, useWC ? "GpiWCBitBlt" : "GpiBitBlt");
}
// List of drawing objects ----------------------------------------------------
BOOL JGList::renderIn( JBPSpace &ps) const
{
   BOOL hits = false;

   for( long i = 0; i < elements(); i++)
      hits |= (*elementAt( i))->renderIn( ps);

   return hits;
}
