/*
 * JListBox.cpp
 *
 * Base Listbox control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JWindow.hpp"
#include "JControl.hpp"
#include "JListBox.hpp"
#include "JCoord.hpp"
#include "JCtlH.hpp"
#include "JStr.hpp"
#include "JMParam.hpp"
#include "JIntern.hpp"
#include "JColln.hpp"

#define lbError( s) jlib_throw( new JException( err_lbox1, JExn::base, s))

// style
const unsigned long JListBox::horzScroll     = LS_HORZSCROLL;
const unsigned long JListBox::multiSelection = LS_MULTIPLESEL | LS_EXTENDEDSEL;
const unsigned long JListBox::noAdjustPos    = LS_NOADJUSTPOS;
const ulong JListBox::normal = 0;

// constructors for ListBox
JListBox::JListBox( JWindow *parent, const JPoint &pos, const JSize &size,
                    ulong Id, ulong style) : JControl()
{
   assertParms( parent, "JListBox::JListBox");

   JCreateWBlock b( parent->handle(), WC_LISTBOX, "", JWindow::visible | style,
                    pos, size, parent->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
}

// listbox apis - these do the work, and are called by the item sub-object
// they deal in terms of indices into the list - you can construct an LB::item
// from this index if so desired.

// remove all the entries in the list box
JListBox &JListBox::empty()
{
   BOOL rc = sendEvent( LM_DELETEALL);
   if( !rc)
      lbError( "delete the contents");

   return self;
}

// number of things in the listbox
ushort JListBox::items() const
{
   return sendEvent( LM_QUERYITEMCOUNT);
}

// top entry
ushort JListBox::topEntry() const
{
   short rc = sendEvent( LM_QUERYTOPINDEX);
   if( rc == LIT_NONE)
       notFound( "listbox item");

   return rc;
}

BOOL JListBox::check( ushort i) const
{
   BOOL rc = true;
   if( i >= items()) {
      jlib_throw( new JBoundsError( err_lbox2));
      rc = false;
   }
   return rc;
}

JListBox &JListBox::scrollTo( ushort index)
{
   if( check( index)) {
      BOOL rc = sendEvent( LM_SETTOPINDEX, index);
      if( !rc)
         lbError( "set the top index");
   }

   return self;
}

// remove a single item
JListBox &JListBox::deleteItem( ushort index)
{
   if( check( index))
      sendEvent( LM_DELETEITEM, index);
   return self;
}

// get & set selection
JListBox &JListBox::select( ushort index, BOOL set)
{
   if( check( index)) {
      BOOL rc = sendEvent( LM_SELECTITEM, index, set);
      if( !rc)
         lbError( "select an item");
   }

   return self;
}

ushort JListBox::choice() const
{
   JMR rc = sendEvent( LM_QUERYSELECTION);

   if( rc.s1() == LIT_NONE)
      notFound( "item");

   return rc;
}

// adding items to the listbox
const unsigned short JListBox::addAtEnd      = (ushort)LIT_END;
const unsigned short JListBox::addSortedUp   = (ushort)LIT_SORTASCENDING;
const unsigned short JListBox::addSortedDown = (ushort)LIT_SORTDESCENDING;

JListBox &JListBox::insert( const JListBox::item &itm, short type, BOOL sel)
{
   JMR rc = sendEvent( LM_INSERTITEM, type, itm.text().buffer());
   if( rc.s1() == LIT_ERROR || rc.s1() == LIT_MEMERROR)
      lbError( "insert an item");
   else {

      if( sel)
         select( rc);

      if( itm.handle()) { // setting handle = 0 => PM_ERROR .. doh.
         BOOL rc2 = sendEvent( LM_SETITEMHANDLE, rc, itm.handle());
         if( !rc2)
            lbError( "set an item's handle");
      }
   }
   return self;
}

// here we go.. insert a collection of items into the box at once
JListBox &JListBox::insert( JLBoxItemList &list, short type)
{
   JLBoxItemList::cursor c( list);
   LBOXINFO              info;
   ulong                 count;
   char                **cands;
   JMR                   rc;
   int                   i;

   count = list.elements();

   info.lItemIndex = type;
   info.ulItemCount = count;
   info.reserved = info.reserved2 = 0; // I mean, what ???

   cands = new char* [ count];

   for( c.toFirst(), i = 0; c.isValid(); c.toNext(), i++)
      cands[ i] = c.current()->text().buffer();

   rc = sendEvent( LM_INSERTMULTITEMS, &info, cands);

   delete [] cands; // added 21/10/97

   if( rc != count)
      lbError( "inserting multiple items");

   return self;
}

// find the index of the item which looks like... this!
ushort JListBox::find( const char *txt, BOOL starts) const
{
   ushort flags = LSS_CASESENSITIVE; // an obligatory option brought to you by ibm
   JMR    rc;

   assertParms( txt, "JListBox::find");

   if( starts)
      flags |= LSS_PREFIX;
   else
      flags |= LSS_SUBSTRING;

   rc = sendEvent( LM_SEARCHSTRING, JMP( flags, 0), txt);

   if( rc.s1() == LIT_NONE)
      notFound( "string" );
   else if( rc.s1() == LIT_ERROR)
      lbError( "search for a string");

   return rc;
}

// a cursor for the listbox. This is of most use with a multi-selection listbox,
// to step through the selection.
JListBox::cursor::cursor( JListBox *LB, JListBox::cursorType T)
         : type( T), curr( 0), lb( LB), last( 0)
{
   assertParms( LB, "JListbox::cursor::cursor");
}

// there must be a less painful way of doing this...
// curr = 0 signals invalid cursor...
void JListBox::cursor::check()
{
   if( curr) {
      delete curr;
      curr = 0;
   }
}

// move to the first choice
JListBox::cursor &JListBox::cursor::toFirst()
{
   if( type == JListBox::selection)
      last = lb->choice();
   else
      last = 0;

   check();

   if( lb->items() > 0)
      curr = new JListBox::item( lb, last);

   return self;
}

// move to next choice
JListBox::cursor &JListBox::cursor::toNext()
{
   if( type == JListBox::selection)
      last = lb->sendEvent( LM_QUERYSELECTION, last);
   else
      last++;

   check();

   if( last != (ushort)LIT_NONE && lb->items() > last)
      curr = new JListBox::item( lb, last);

   return self;
}

// is valid
BOOL JListBox::cursor::isValid() const
{
   return !!curr;
}

// element at position
JListBox::item *JListBox::cursor::current() const
{
   JListBox::item *rc = 0;

   if( !isValid())
      jlib_throw( new JInvalidCursor);
   else rc = curr;

   return rc;
}

JListBox::item *JListBox::cursor::operator -> ()
{
   JListBox::item *rc = 0;

   if( !isValid())
      jlib_throw( new JInvalidCursor);
   else rc = curr;

   return rc;
}

// Listbox item class. This has two purposes, to act as a wrapper for the
// data which gets inserted into the box, and as a wrapper of an actual item.
JListBox::item::item( JListBox *Lb, ushort index)
               : lb( Lb), ndex( index), txt( 0), hndle( 0)
{
   assertParms( Lb, "JListBOx::item::item");
}

JListBox::item::item( const char *t, ulong h)
               : lb( 0), ndex( 0), txt( new JStr( t)), hndle( h)
{}


JListBox::item::item( const JListBox::item &copy)
               : lb( copy.getListBox()), ndex( copy.index()),
                 txt( new JStr( copy.text())), hndle( copy.handle())
{}

JListBox::item::~item()
{
   if( txt)
      delete txt;
}

JStr JListBox::item::text() const
{
   if( txt) return *txt;

   JStr  rc;
   short length = lb->sendEvent( LM_QUERYITEMTEXTLENGTH, ndex);

   if( length == LIT_ERROR)
      lbError( "querying the length of an item's text");
   else {

      rc.setsize( length + 1);
      lb->sendEvent( LM_QUERYITEMTEXT, JMP( ndex, length + 1), rc.buffer());
   }

   return rc;
}

JListBox::item &JListBox::item::setText( const char *s)
{
   if( txt)
      *txt = s;
   else {
      BOOL rc = lb->sendEvent( LM_SETITEMTEXT, ndex, s);
      if( !rc)
         lbError( "setting the text of an item");
   }

   return self;
}

JListBox::item &JListBox::item::setHandle( ulong ul)
{
   if( txt)
      hndle = ul;
   else {
      BOOL rc = lb->sendEvent( LM_SETITEMHANDLE, ndex, ul);
      if( !rc)
         lbError( "setting the handle of an item");
   }

   return self;
}

ulong JListBox::item::handle() const
{
   if( txt)
      return hndle;

   return lb->sendEvent( LM_QUERYITEMHANDLE, ndex);
}

JListBox::item &JListBox::item::remove()
{
   if( txt)
      jlib_throw( new JException( err_lboxitem, JExn::base));
   else
      lb->deleteItem( ndex);

   return self;
}

JListBox::item &JListBox::item::scrollTo()
{
   if( txt)
      jlib_throw( new JException( err_lboxitem, JExn::base));
   else
      lb->scrollTo( ndex);

   return self;
}

JListBox::item &JListBox::item::select( BOOL f)
{
   if( txt)
      jlib_throw( new JException( err_lboxitem, JExn::base));
   else
      lb->select( ndex, f);

   return self;
}

// event handling - notify codes from the listbox
BOOL JListBox::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case LN_KILLFOCUS:
         return lostFocus();
         break;
      case LN_SETFOCUS:
         return gainedFocus();
         break;
      case LN_SCROLL:
         return scrolled();
         break;
      case LN_SELECT:
      {
         JListBox::cursor c( this);
         return selected( c);
         break;
      }
      case LN_ENTER:
      {
         JListBox::cursor c( this);
         return choice( c);
         break;
      }
   }
   return false;
}
