/*
 * JMLE.cpp
 *
 * Multi-line entry field control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JMLE.hpp"
#include "JCoord.hpp"
#include "JColour.hpp"
#include "JControl.hpp"
#include "JWindow.hpp"
#include "JStr.hpp"
#include "JCtlH.hpp"
#include "JMParam.hpp"
#include "JIntern.hpp"
#include <string.h>

#define mleError( s) jlib_throw( new JException( err_mle, JExn::base, s))

const unsigned long JMLE::border     = MLS_BORDER;
const unsigned long JMLE::readonly   = MLS_READONLY;
const unsigned long JMLE::wordwrap   = MLS_WORDWRAP;
const unsigned long JMLE::horzScroll = MLS_HSCROLL;
const unsigned long JMLE::vertScroll = MLS_VSCROLL;
const unsigned long JMLE::ignoreTab  = MLS_IGNORETAB;
const unsigned long JMLE::noUndo     = MLS_DISABLEUNDO;
const unsigned long JMLE::scrollBars = MLS_HSCROLL | MLS_VSCROLL;
const unsigned long JMLE::normal     = MLS_BORDER | MLS_WORDWRAP;

JMLE::JMLE( HWND hwnd) : JControl( hwnd)
{
   sendEvent( MLM_FORMAT, MLFIE_NOTRANS);
}

JMLE::JMLE( JWindow *w, ulong id) : JControl( w, id)
{
   sendEvent( MLM_FORMAT, MLFIE_NOTRANS);
}

JMLE::JMLE( JWindow *parent, const JPoint &pos, const JSize &size,
            ulong Id, ulong style) : JControl()
{
   assertParms( parent, "JMLE::JMLE");

   JCreateWBlock b( parent->handle(), WC_MLE, "", JWindow::visible | style,
                    pos, size, parent->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
   sendEvent( MLM_FORMAT, MLFIE_NOTRANS);
}

// Event call-back handler functions

BOOL JMLE::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case MLN_HSCROLL:
         return scrolled( JMLE::horzScroll);
         break;
      case MLN_VSCROLL:
         return scrolled( JMLE::vertScroll);
         break;
      case MLN_CHANGE:
         return changed();
         break;
      case MLN_SETFOCUS:
         return gainedFocus();
         break;
      case MLN_KILLFOCUS:
         return lostFocus();
         break;
      case MLN_MARGIN:
      {
         PMARGSTRUCT pMarg = (PMARGSTRUCT) e.data();
         assertParms( pMarg, "MLN_MARGIN");
         return borderCrossed( (JMLE::margin) pMarg->afMargins,
                               pMarg->usMouMsg, pMarg->iptNear);
         break;
      }
      case MLN_SEARCHPAUSE:
         return continueSearch( (long) e.data());
         break;
   }
   return false;
}

// interaction with clipboard
JMLE &JMLE::clear()
{
   sendEvent( MLM_CLEAR);
   return self;
}

JMLE &JMLE::cut()
{
   sendEvent( MLM_CUT);
   return self;
}

JMLE &JMLE::copy()
{
   sendEvent( MLM_COPY);
   return self;
}

JMLE &JMLE::paste()
{
   sendEvent( MLM_PASTE);
   return self;
}

JMLE &JMLE::setSelection( long start, long count)
{
   BOOL rc = sendEvent( MLM_SETSEL, start, start + count);
   if( !rc) mleError( "set the selection");
   return self;
}

// query the selected text. I don't think this should be so hard...
JStr JMLE::getSelection() const
{
   long selstart = sendEvent( MLM_QUERYSEL, MLFQS_MINSEL);
   long selend = sendEvent( MLM_QUERYSEL, MLFQS_MAXSEL);
   long size = sendEvent( MLM_QUERYFORMATTEXTLENGTH, selstart, selend - selstart);

   JStr selection;
   selection.setsize( size + 1);
   sendEvent( MLM_QUERYSELTEXT, selection.buffer());

   return selection;
}

// cursor movement
JMLE &JMLE::moveCursorTo( long offset)
{
   if( offset > textLength())
      jlib_throw( new JBoundsError( err_efbounds));
   else {
      BOOL rc = sendEvent( MLM_SETFIRSTCHAR, offset);
      if( !rc) mleError( "set the cursor position");
   }
   return self;
}

long JMLE::cursorPos() const
{
   return sendEvent( MLM_QUERYFIRSTCHAR);
}

// line things
long JMLE::lines() const
{
   return sendEvent( MLM_QUERYLINECOUNT);
}

JMLE &JMLE::scrollToLine( long line)
{
   long offset = sendEvent( MLM_CHARFROMLINE, line);
   return moveCursorTo( offset);
}

long JMLE::lineLength( long line) const
{
   long offset = sendEvent( MLM_CHARFROMLINE, line);
   return sendEvent( MLM_QUERYFORMATLINELENGTH, offset);
}

// get the text of a specified line... should add error checking
JStr JMLE::line( long line) const
{
   JStr buffer;
   long start  = sendEvent( MLM_CHARFROMLINE, line);
   long length = sendEvent( MLM_QUERYFORMATLINELENGTH, start);

   buffer.setsize( length + 1);

   BOOL rc = sendEvent( MLM_SETIMPORTEXPORT, buffer.buffer(), length + 1);
   if( !rc) mleError( "set the ie bufer");
   else {
      sendEvent( MLM_EXPORT, &start, &length);
      if( length)
         mleError( "export some text");
   }

   return buffer;
}

long JMLE::textLength() const
{
   long length;
   // how many characters in the text?
   length = sendEvent( MLM_QUERYTEXTLENGTH);
   // convert to bytes - cr/lf misery
   length = sendEvent( MLM_QUERYFORMATTEXTLENGTH, JMP(), length);
   return length;
}

// add a line to the end of the mle. This reveals why the mle is so damn slow -
// each of these event things requires at least one thunk ( mle is 16-bit code).
JMLE &JMLE::addLine( const char *text)
{
   char *buffer;
   ulong insertPt = textLength();
   ulong buffsize;
   ulong oldIP;
   ulong pos = 0;

   // set up the import buffer
   buffsize = strlen( text);
   if( buffsize > 64000)
      buffsize = 64000;

   buffer = (char *) malloc( buffsize);
   BOOL rc = sendEvent( MLM_SETIMPORTEXPORT, buffer, buffsize);
   if( !rc) mleError( "set the i/e buffer");
   else {
   
      // add the text
      while( pos < strlen( text)) {
         oldIP = insertPt;
         strncpy( buffer, text + pos, buffsize);
         sendEvent( MLM_IMPORT, &insertPt, buffsize);
         pos += ( oldIP - insertPt);
      }
   }

   free( buffer);

   return self;
}

// need to override normal method 'cos of import/export misery
JStr JMLE::getText() const
{
   return export();
}

// set contents, again need to override JWindow's method
JMLE &JMLE::setText( const char *s)
{
   // delete current text
   empty();
   // use append-er to add the text
   addLine( s);
   return self;
}

JMLE &JMLE::empty()
{
   sendEvent( MLM_DELETE, JMP(), textLength());
   return self;
}

// insert some text at the current cursor position
JMLE &JMLE::insert( const char *s)
{
   sendEvent( MLM_INSERT, JMP(s));
   return self;
}

// import and export text - no stripping
JStr JMLE::export() const
{
   JStr text( ""), buffer;
   long length = textLength();
   long start = 0;
   long buffsize;
   long prelength;

   // set up the export buffer
   buffsize = 64000;
   if( length < buffsize)
      buffsize = length + 1;

   buffer.setsize( buffsize);
   BOOL rc = sendEvent( MLM_SETIMPORTEXPORT, buffer.buffer(), buffsize - 1);
   if( !rc) mleError( "set the i/e buffer");
   else {
   
      // export the text.
      // Because the mle is stupid, we have to faff around endlessly (hope not..)
      while( length) {
         prelength = length;
         // this call updates start & length to be ready for next time
         sendEvent( MLM_EXPORT, &start, &length);
         buffer[ prelength - length] = 0;
         text = text + buffer;
      }
   }

   return text;
}

// umm... what's this for, j?
JMLE &JMLE::import( const char *s)
{
   setText( s);
   return self;
}

// screen refresh - if the mle was written properly then you wouldn't have to
// worry about this. REWRITE the mle!
JMLE &JMLE::enableRefresh( BOOL doit)
{
   BOOL rc = sendEvent( doit ? MLM_ENABLEREFRESH : MLM_DISABLEREFRESH);
   if( !rc) mleError( "set the refresh state");
   return self;
}

JMLE &JMLE::setReadonly( BOOL f)
{
   sendEvent( MLM_SETREADONLY, f);
   return self;
}

BOOL JMLE::isReadonly() const
{
   return sendEvent( MLM_QUERYREADONLY);
}

JMLE &JMLE::setTabStop( long pels)
{
   int rc = sendEvent( MLM_SETTABSTOP, pels);
   if( rc < 0)
      mleError( "setting the tabstop");
   return self;
}

long JMLE::getTabStop() const
{
   return sendEvent( MLM_QUERYTABSTOP);
}

JMLE &JMLE::setWrap( BOOL set)
{
   BOOL rc = sendEvent( MLM_SETWRAP, set);
   if( !rc) mleError( "set the wrap state");
   return self;
}

BOOL JMLE::hasWrap() const
{
   return sendEvent( MLM_QUERYWRAP);
}

BOOL JMLE::hasChanged() const
{
   return sendEvent( MLM_QUERYCHANGED);
}

JMLE &JMLE::setChanged( BOOL yes)
{
   sendEvent( MLM_SETCHANGED, yes);
   return self;
}

JMLE &JMLE::setMaxLength( long count)
{
   BOOL rc = sendEvent( MLM_SETTEXTLIMIT, count);
   if( !rc) jlib_throw( new JException( err_mletoobig, JExn::base));
   return self;
}

long JMLE::getMaxLength() const
{
   return sendEvent( MLM_QUERYTEXTLIMIT);
}

BOOL JMLE::canUndo() const
{
   JMR rc = sendEvent( MLM_QUERYUNDO);
   return rc.s2();
}

JMLE &JMLE::resetUndo()
{
   sendEvent( MLM_RESETUNDO);
   return self;
}

JMLE &JMLE::undo()
{
   BOOL rc = sendEvent( MLM_UNDO);
   if( !rc) mleError( "do an undo");
   return self;
}

BOOL JMLE::find( const char *string, BOOL match, long start) const
{
#ifdef __EMX__
   MLE_SEARCHDATA m = { sizeof( MLE_SEARCHDATA), 0,
                        (char*) string, 0, strlen( string),
                        0, start, -1, 0 };
#else
   MLE_SEARCHDATA m = { sizeof( MLE_SEARCHDATA),
                        (char*) string, 0, strlen( string),
                        0, start, -1, 0 };
#endif

   ulong style = MLFSEARCH_SELECTMATCH;
   if( match) style |= MLFSEARCH_CASESENSITIVE;

   return sendEvent( MLM_SEARCH, style, &m);
}

BOOL JMLE::replace( const char *find, const char *change,
                    BOOL match, long start) const
{
#ifdef __EMX__
   MLE_SEARCHDATA m = { sizeof( MLE_SEARCHDATA), 0,
                        (char*)find, (char*)change, strlen( find),
                        strlen( change), start, -1, 0 };
#else
   MLE_SEARCHDATA m = { sizeof( MLE_SEARCHDATA),
                        (char*)find, (char*)change, strlen( find),
                        strlen( change), start, -1, 0 };
#endif

   ulong style = MLFSEARCH_CHANGEALL;
   if( match) style |= MLFSEARCH_CASESENSITIVE;

   return sendEvent( MLM_SEARCH, style, &m);
}
