/*
 * JMenu.cpp
 *
 * Menu - bar, pulldown, popup - class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JMenu.hpp"
#include "JWindow.hpp"
#include "JStr.hpp"
#include "JCoord.hpp"
#include "JPointer.hpp"
#include "JMParam.hpp"
#include "JResID.hpp"
#include "JModule.hpp"
#include "JBitmap.hpp"
#include "JIntern.hpp"

// Helpful shortcuts ---------------------------------------------------------
#define menuError( s) jlib_throw( new JException( err_menu, JExn::base, s))

// access to the MENUITEM structure for the entry
static void queryMI( PMENUITEM pMenuItem, JMenu *m, ushort id)
{
   BOOL rc = m->sendEvent( MM_QUERYITEM, JMP( id, true), pMenuItem);
   if( !rc)
      menuError( "query for a menu item");
}

static void setMI( PMENUITEM pMI, JMenu *m)
{
   BOOL rc = m->sendEvent( MM_SETITEM, JMP( 0, true), pMI);
   if( !rc)
      menuError( "set an item");
}

// ctors ---------------------------------------------------------------------
JMenu::JMenu( const JResID &id, JWindow *w, const JModule &mod) : JWindow()
{
   HWND hwnd = WinLoadMenu( w->handle(), mod.handle(), id.value());
   if( !hwnd)
      resError( "menu");

   setHwnd( hwnd);
}

JMenu::JMenu( JMother *owner, JWindow *parent, ulong ID, BOOL bar) : JWindow()
{
   JCreateWBlock b( parent->handle(), WC_MENU, 0, bar ? MS_ACTIONBAR : 0,
                    JPoint(), JSize(), owner->handle(), 0, ID, 0, 0);

   setHwnd( JInternal::window::create( &b));
   setautodelete();
}

// removing & adding items ---------------------------------------------------
JMenu &JMenu::removeItem( short Id, BOOL del)
{
   ulong msg = del ? MM_DELETEITEM : MM_REMOVEITEM;
   sendEvent( msg, JMP( Id, true));
   return self;
}

JMenu &JMenu::addItem( const char *text, short Id, short pos, JMenu *sub)
{
   MENUITEM mI = { pos, MIS_TEXT | (sub ? MIS_SUBMENU : 0), 0, Id,
                   sub ? sub->handle() : 0, 0 };

   ulong rc = sendEvent( MM_INSERTITEM, &mI, text);
   if( (long)rc == MIT_ERROR)
      menuError( "insert");
   return self;
}

JMenu &JMenu::addItem( const JBitmap &b, short Id, short pos, JMenu *sub)
{
   MENUITEM mI = { pos, MIS_BITMAP | (sub ? MIS_SUBMENU : 0), 0, Id,
                   sub ? sub->handle() : 0, b.handle() };

   ulong rc = sendEvent( MM_INSERTITEM, &mI);
   if( (long)rc == MIT_ERROR)
      menuError( "insert");
   return self;
}

JMenu &JMenu::addSeparator( short Id, short pos)
{
   MENUITEM mI = { pos, MIS_SEPARATOR, 0, Id, 0, 0 };

   ulong rc = sendEvent( MM_INSERTITEM, &mI);
   if( (long)rc == MIT_ERROR)
      menuError( "insert");
   return self;
}

JMenu &JMenu::addSubMenu( short Id, JMenu *sub)
{
   // get data for item at id
   MENUITEM mI; queryMI( &mI, this, Id);

   // add submenu bits
   mI.hwndSubMenu = sub->handle();
   mI.afStyle |= MIS_SUBMENU;

   // get text
   JStr menuText( itemText( Id));

   // remove item
   removeItem( Id);

   // insert new item with submenu bits set
   short rc = sendEvent( MM_INSERTITEM, &mI,
                         (mI.afStyle & MIS_TEXT) ? menuText.buffer() : 0);
   if( rc == MIT_ERROR)
      menuError( "insert");

   // phew.
   return self;
}

// item text -----------------------------------------------------------------
JStr JMenu::itemText( short Id) const
{
   JStr  menuText;
   short cbText = sendEvent( MM_QUERYITEMTEXTLENGTH, Id);
   menuText.setsize( cbText + 1);
   sendEvent( MM_QUERYITEMTEXT, JMP( Id, cbText + 1), menuText.buffer());

   return menuText;
}

JMenu &JMenu::setItemText( short Id, char *s)
{
   BOOL rc = sendEvent( MM_SETITEMTEXT, Id, s);
   if( !rc)
      menuError( "settext");
   return self;
}

// item attributes -----------------------------------------------------------
BOOL JMenu::toggleItem( short Id) const
{
   BOOL hasTick = true;

   // get data
   MENUITEM mI; queryMI( &mI, (JMenu *) this, Id);

   // process data
   if( mI.afAttribute & MIA_CHECKED) {
      mI.afAttribute &= ~MIA_CHECKED;
      hasTick = false;
   } else
      mI.afAttribute |= MIA_CHECKED;

   // reset data
   setMI( &mI, (JMenu *) this);

   return hasTick;
}

JMenu &JMenu::tick( short Id, BOOL set)
{
   MENUITEM mI; queryMI( &mI, this, Id);

   if( set) mI.afAttribute |= MIA_CHECKED;
   else     mI.afAttribute &= ~MIA_CHECKED;

   setMI( &mI, this);

   return self;
}

JMenu &JMenu::enableItem( short Id, BOOL enable)
{
   MENUITEM mI; queryMI( &mI, this, Id);

   if( enable)
      mI.afAttribute &= ~MIA_DISABLED;
   else
      mI.afAttribute |= MIA_DISABLED;

   setMI( &mI, this);

   return self;
}

JMenu &JMenu::enableItems( BOOL enable)
{
   short cItems = sendEvent( MM_QUERYITEMCOUNT);
   for( int i = 0; i < cItems; i++) {
      ushort usID = sendEvent( MM_ITEMIDFROMPOSITION, i);
      enableItem( usID, enable);
   }
   return self;
}

// getting submenus ----------------------------------------------------------
JMenu JMenu::getSubMenu( short Id) const
{
   MENUITEM mI; queryMI( &mI, (JMenu *) this, Id);
   return JMenu( mI.hwndSubMenu); // !! erm, this is a bit dangerous
}

// conditional cascade bits --------------------------------------------------
JMenu &JMenu::makeCascade( short defaultEntry)
{
   setStyle( MS_CONDITIONALCASCADE);
   return setDefault( defaultEntry);
}

JMenu &JMenu::setDefault( short Id)
{
   BOOL rc = sendEvent( MM_SETDEFAULTITEMID, Id);
   if( !rc)
      menuError( "set the default item");

   return self;
}

short JMenu::getDefault() const
{
   return sendEvent( MM_QUERYDEFAULTITEMID);
}

// context menus -------------------------------------------------------------
JMenu &JMenu::popUp( JWindow &owner, const JPoint &pos)
{
   setOwner( &owner);
   BOOL rc =
   WinPopupMenu( JWindow::theDesktopWindow,
                 owner,
                 handle(),
                 pos.x,
                 pos.y,
                 0,
                 PU_HCONSTRAIN | PU_VCONSTRAIN | PU_NONE |
                 PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 );
   if( !rc)
      pmError( 3, "WinPopupMenu");

   return self;
}

JMenu &JMenu::popUp( JWindow &owner)
{
   return popUp( owner, JPointer::pos());
}
