/*
 * JMother.cpp
 *
 * Class that provides frame-like behaviour
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JMother.hpp"
#include "JWindow.hpp"
#include "JProfile.hpp"
#include "JCoord.hpp"
#include "JMenu.hpp"
#include "JAccel.hpp"
#include "JIcon.hpp"
#include "JMParam.hpp"
#include "JPM.hpp"
#include "JResID.hpp"
#include "JSystem.hpp"
#include "JCmdH.hpp"
#include "JFocusH.hpp"
#include "JStr.hpp"
#include "JHelp.hpp"

#include <string.h>
// this handler is for setting the minimum & maximum sizes of a frame
struct JMaResizeHandler : public JHandler
{
   JMaResizeHandler() : JHandler()
   {}
   BOOL handle( JEventData *e)
   {
      switch( e->msg) {
         // need this one for tasklist->tile/cascade
         case WM_ADJUSTWINDOWPOS:
         {
            JMother *mama = (JMother *) e->win;
            JSize    max = mama->maximumSize();
            PSWP    pSwp = (PSWP) e->mp1();

            if( pSwp->cx > max.x) pSwp->cx = max.x;
            if( pSwp->cy > max.y) pSwp->cy = max.y;

            return false;
         }
         case WM_QUERYTRACKINFO:  // set the minimum / maximum tracking sizes
         {
            JMother *mama = (JMother *) e->win;

            // get the frame to fill in the complicated bits
            mama->defaultAction( e->msg, e->mp1, e->mp2);

            // now alter the minimum and maximum track position things
            PTRACKINFO pTrack = (PTRACKINFO) e->mp2();
            JSize &min = mama->minimumSize();
            if( min.x && min.y) {
               pTrack->ptlMinTrackSize.x = min.x;
               pTrack->ptlMinTrackSize.y = min.y;
            }
            pTrack->ptlMaxTrackSize.x = mama->maximumSize().x;
            pTrack->ptlMaxTrackSize.y = mama->maximumSize().y;

            // set rc to true, so that things keep working, and return true
            e->rc = true;
            return true;
         }
         case WM_MINMAXFRAME:
         {
            PSWP     pSwp = (PSWP) e->mp1();
            JMother *mama = (JMother *) e->win;
            // if the frame is being maximized...
            if( pSwp->fl & SWP_MAXIMIZE) {
               // get the maximum size of the window  & screen
               JSize &max = mama->maximumSize();
               JSize screen = JSystem::screen();
               // if max is screen or larger, let the system do its stuff
               if( !(max >= screen)) {
                  JPoint pos = mama->origin();
                  pSwp->cx = (max.x > screen.x) ? screen.x : max.x;
                  pSwp->cy = (max.y > screen.y) ? screen.y : max.y;
                  // if we can max the window without moving it, do it
                  // else max the window and move it so there's a corner in the
                  // top-left of the screen
                  if( pos + max <= screen) {
                     pSwp->x = pos.x;
                     pSwp->y = pos.y;
                  } else {
                     pSwp->x = screen.x - pSwp->cx;
                     pSwp->y = screen.y - pSwp->cy;
                  }
                  // ret 1
               }
            }
            return false;
         }
         default:
            return false;
      }
   }
};

static JMaResizeHandler mothersizehandler;

// Private data area ----------------------------------------------------------
struct JMotherData
{
   JIcon       *icon;
   JAccelTable *accel;
   JMenu       *menu;
   JMenu       *sysMenu;
   JProfile    *prf;
   JStr         app;
   ulong        hSwitch;
   JHelpID      helpid;
   JSize        max, min;
   BOOL         isDlg;
   JMotherData( BOOL d) : icon( 0), accel( 0), menu( 0), sysMenu( 0), prf( 0),
                          hSwitch( 0), max( JSystem::screen()), isDlg( d)
   {}
  ~JMotherData() {
      if( accel) delete accel;
      if( menu) delete menu;
      if( icon) delete icon;
      if( sysMenu) delete sysMenu;
      // we don't own the profile
   }
};

// Handler to intercept close events and general command events ---------------
class JMotherHandler : public JCmdHandler
{
 public:
   JMotherHandler() : JCmdHandler()
   {}
   BOOL syscommand( const JCmdEvent &e)
   {
      JMother *mama = (JMother *) e.window();
      BOOL       rc = false;

      if( e.value() == SC_CLOSE) {

         rc = mama->closing();

         if( !rc) { // yup, quit the app's q right away.
            mama->hide();
            if( !mama->data->isDlg) {
               // for a frame, turn the message into a wm_close.
               // this is handled by the defproc & turned into a wm_quit
               mama->postEvent( WM_CLOSE);
               rc = true;
            } else
               // for a dialog, post a wm_quit to the queue
               JWindow::PostEvent( 0, WM_QUIT);
         }
      } else
         rc = mama->sysCommand( e);

      return rc;
   }
   BOOL command( const JCmdEvent &e)
   {
      JMother *mama = (JMother *) e.window();
      return mama->command( e);
   }
};

static JMotherHandler motherhandler;

// Handler to intercept wm_help messages --------------------------------------
class JHelpMsgHandler : public JHandler
{
 public:
   JHelpMsgHandler() : JHandler()
   {}
   BOOL handle( JEventData *e)
   {
      switch( e->msg) {
         case WM_HELP:
            if( HelpManager.helpModel() == JHelpManager::dynamic) {
               JMother *mama = (JMother *) e->win;
               JHelpID id = mama->helpID();
               HelpManager.showHelp( id);
               e->rc = true;
               return true;
            }
            break;
         case HM_QUERY_KEYS_HELP:
            e->rc = HelpManager.keysHelp.ID();
            return true;
         case HM_TUTORIAL:
            if( HelpManager.tut)
               (*HelpManager.tut)();
            return true;
         case HM_GENERAL_HELP_UNDEFINED:
            if( HelpManager.generalID.ID())
               HelpManager.showHelp( HelpManager.generalID);
            else
               HelpManager.showContents();
            return true;
      }
      return false;
   }
};

static JHelpMsgHandler motherhelphandler;

// Activation handler; for passing the help buck around -----------------------
class JMotherActiveHandler : public JFocusChangeHandler
{
 public:
   JMotherActiveHandler() : JFocusChangeHandler() {}
   BOOL activated( const JFocusChangeEvent &e)
   {
      if( HelpManager.helpModel() != JHelpManager::none) {
         HelpManager.setActiveWindow( e.window());
      }
      return false;
   }
   BOOL deactivated( const JFocusChangeEvent &)
   {
      if( HelpManager.helpModel() != JHelpManager::none) {
         HelpManager.setActiveWindow( 0);
      }
      return false;
   }
};

static JMotherActiveHandler motheractivehandler;

// JMother class; common behaviour between dlgs and frames --------------------
JMother::JMother( BOOL isDlg) : JWindow(), data( new JMotherData( isDlg))
{
   attach( &motheractivehandler);
   attach( &motherhandler);
   attach( &mothersizehandler);
   attach( &motherhelphandler);
}

JMother::~JMother()
{
   if( data->prf) savePos( *data->prf, data->app);
   delete data;

   detach( &mothersizehandler);
   detach( &motherhandler);
   detach( &motherhelphandler);
   detach( &motheractivehandler);
}

// Setting up frame furniture -------------------------------------------------
JMother &JMother::setIcon( JResID id, const JModule &mod)
{
   JIcon *ic = new JIcon( id, mod);
   return setIcon( ic);
}

JMother &JMother::setIcon( JIcon *ic)
{
   if( data->icon) delete data->icon;
   data->icon = ic;
   BOOL rc = sendEvent( WM_SETICON, JMP( data->icon->handle()));
   if( !rc)
      pmError( 2, "wm_seticon failed");

   return self;
}

JIcon *JMother::getIcon()
{
   return data->icon;
}

JMother &JMother::setAccel( JResID id, const JModule &m)
{
   return setAccel( new JAccelTable( id, m));
}

JMother &JMother::setAccel( JAccelTable *table)
{
   if( data->accel) delete data->accel;
   data->accel = table;

   BOOL rc = WinSetAccelTable( JPM::current()->hab(),
                               data->accel->handle(), handle());
   if( !rc)
      pmError( 1, "WinSetAccelTable");

   return self;
}

JAccelTable *JMother::getAccelTable()
{
   return data->accel;
}

JMother &JMother::setMenu( JResID id, const JModule &mod)
{
   return setMenu( new JMenu( id, this, mod));
}

JMother &JMother::setMenu( JMenu *m)
{
   if( data->menu) delete data->menu;
   data->menu = m;
   data->menu->setID( FID_MENU);
   sendEvent( WM_UPDATEFRAME, FCF_MENU);
   return self;
}

JMenu *JMother::getMenu()
{
   return data->menu;
}

JMother &JMother::showMenu( BOOL shw)
{
   if( data->menu) {
      if( shw)
         data->menu->setParent( this);
      else
         data->menu->setParent( &JWindow::theObjectWindow);
      sendEvent( WM_UPDATEFRAME, FCF_MENU);
      invalidate();
   }

   return self;
}

JMother &JMother::setAll( JResID id, const JModule &mod)
{
   setIcon( id, mod);
   setAccel( id, mod);
   setMenu( id, mod);

   return self;
}

// load window position from ini ----------------------------------------------
JMother &JMother::loadPos( JProfile &prfl, const char *appl, BOOL resave)
{
   if( resave) { // need to save bits in private data
      data->prf = &prfl;
      data->app = appl;
   }

   JPoint orig, size;

#ifdef REALLY_THROW_EXCEPTIONS
   try {                                 // if entry not in ini, fall thru'
#endif
      orig.x = prfl.readLong( "x", appl);
      orig.y = prfl.readLong( "y");
      size.x = prfl.readLong( "cx");
      size.y = prfl.readLong( "cy");

#ifndef REALLY_THROW_EXCEPTIONS
      if( JPM::current()->type == JExn::notFound)
         jlib_catch();
      else
#endif
         reshape( orig, size);

#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JNotFoundError *) {
   }
#endif

   return self;
}

// save position again
JMother &JMother::savePos( JProfile &prfl, const char *appl)
{
   JPoint pos = origin();
   JPoint ext = size();

   // !! ought to check if we're minimized or have otherwise gone crazy

   prfl.write( pos.x, "x", appl);
   prfl.write( pos.y, "y");
   prfl.write( ext.x, "cx");
   prfl.write( ext.y, "cy");

   return self;
}

// get the system menu --------------------------------------------------------
JMenu *JMother::getSystemMenu()
{
   if( !data->sysMenu) {
      MENUITEM mI;
      ulong    hwndBox = WinWindowFromID( handle(), FID_SYSMENU);
      short    boxID = JWindow::SendEvent( hwndBox, MM_ITEMIDFROMPOSITION);

      JWindow::SendEvent( hwndBox, MM_QUERYITEM, JMP( boxID, false) , &mI);

      // !! erm, why don't menus know about JWindowPtr ?
      data->sysMenu = new JMenu( mI.hwndSubMenu);
   }
   return data->sysMenu;
}

// adding window to tasklist --------------------------------------------------
JMother &JMother::showInTasklist( BOOL show)
{
   SWCNTRL swcntrl;

   if( !data->hSwitch) {   // need to create the entry
      if( show) {
         swcntrl.hwnd = handle();
         if( data->icon) swcntrl.hwndIcon = data->icon->handle();
         swcntrl.hprog = 0;
         swcntrl.idProcess = 0;  // means current
         swcntrl.idSession = 0;  //  "  "   "  "
         swcntrl.uchVisibility = SWL_VISIBLE;  // why /uch/Visibility ?
         swcntrl.fbJump = SWL_JUMPABLE; // what on earth does this do?
         strcpy( swcntrl.szSwtitle, getText());
         swcntrl.bProgType = PROG_PM; // rah!
         data->hSwitch = WinCreateSwitchEntry( JPM::current()->hab(), &swcntrl);
         if( !data->hSwitch)
            pmError( 3, "WinCreateSwitchEntry");
      }
   } else {   // just alter the entry which we've already created
      if( WinQuerySwitchEntry( data->hSwitch, &swcntrl))
         pmError( 4, "WinQuerySwitchEntry");
      swcntrl.uchVisibility = show ? SWL_VISIBLE : SWL_INVISIBLE;
      if( WinChangeSwitchEntry( data->hSwitch, &swcntrl))
         pmError( 1, "WinChangeSwitchEntry");
   }
   return self;
}

JMother &JMother::setText( const char *txt)
{
   JWindow::setText( txt);
   if( data->hSwitch) { // need to update switchlist
      SWCNTRL swcntrl;
      if( WinQuerySwitchEntry( data->hSwitch, &swcntrl))
         pmError( 3, "WinQuerySwitchEntry");
      strcpy( swcntrl.szSwtitle, getText());
      if( WinChangeSwitchEntry( data->hSwitch, &swcntrl))
         pmError( 1, "WinChangeSwitchEntry");
   }
   return self;
}

// Client rectangle control ---------------------------------------------------
JRect JMother::clientRect() const
{
   JRect rcl = rect();
   BOOL rc = WinCalcFrameRect( handle(), rcl, true); // given = frame
   if( !rc)
      pmError( 2, "WinCalcFrameRect");

   return rcl;
}

JMother &JMother::sizeClientTo( const JSize &sz)
{
   JRect rcl( 0, 0, sz);
   BOOL rc = WinCalcFrameRect( handle(), rcl, false); // given = client
   if( !rc)
      pmError( 2, "WinCalcFrameRect");

   resize( rcl.size());
   return self;
}

// maximum & minimum sizes ----------------------------------------------------
JMother &JMother::setMinimumSize( const JSize &s)
{
   data->min = s;
   return self;
}

JMother &JMother::setMaximumSize( const JSize &s)
{
   data->max = s;
   return self;
}

JSize &JMother::minimumSize()
{ return data->min; }

JSize &JMother::maximumSize()
{ return data->max; }

// Help manager support -------------------------------------------------------
JMother &JMother::setHelpID( const JHelpID &id)
{
   data->helpid = id;
   return self;
}

JHelpID JMother::helpID() const
{
   if( data->helpid.ID() == 0)
      return JHelpID( ID());
   else
      return data->helpid;
}
