/*
 * JMsgBox.hpp
 *
 * JLib message boxes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JMsgBox.hpp"

#include "JLibRes.h"
#include "JModule.hpp"
#include "JSystem.hpp"

#include "JWrap.hpp"
//#include "JStaticT.hpp"
#include "JIcon.hpp"
#include "JStaticG.hpp"
#include "JPushBut.hpp"
#include "JResID.hpp"

#include "JMenu.hpp"

// constants -----------------------------------------------------------------

const ushort JMsgBox::Ok     = 0x01;
const ushort JMsgBox::cancel = 0x02;
const ushort JMsgBox::yes    = 0x04;
const ushort JMsgBox::no     = 0x08;
const ushort JMsgBox::retry  = 0x10;
const ushort JMsgBox::abort  = 0x20;
const ushort JMsgBox::ignore = 0x40;
const ushort JMsgBox::help   = 0x80;

// private data class --------------------------------------------------------
struct JMsgBoxData
{
   JWrapText    text;
// JStaticText  text;
   JPushButton  butt1;
   JPushButton  butt2;
   JPushButton  butt3;
   JPushButton  buttHelp;
   JStaticIcon  icon;
   JWindow     *owner;
   ulong        butts;

   JMsgBoxData( JMsgBox *mbox, JWindow *wn) : text( mbox, j_control_text),
                                              butt1( mbox, j_butt_one),
                                              butt2( mbox, j_butt_two),
                                              butt3( mbox, j_butt_three),
                                              buttHelp( mbox, j_butt_help),
                                              icon( mbox, j_icon_msg),
                                              owner( wn)
   {}
};

// ctor/dtor -----------------------------------------------------------------
JMsgBox::JMsgBox( JWindow *owner, const char *msg, const char *title,
                  JMsgBox::buttons b, JMsgBox::type t)
        : JDialog( owner, JResID(j_dlg_mbox),
                   &JWindow::theDesktopWindow,
                   JModule::myResMod),
          data( new JMsgBoxData( this, owner))
{
   setFontNameSize( JSystem::getGeneralFont());

   // set the window title
   setText( title);

   // set up the wrap text, which repositions the controls
   setMessage( msg);

   // set the icon as requested
   JIcon::sys iccy;
   switch( t) {
      case JMsgBox::informational:
         iccy = JIcon::info;
         break;
      case JMsgBox::question:
         iccy = JIcon::question;
         break;
      case JMsgBox::warning:
         iccy = JIcon::warning;
         break;
      case JMsgBox::error:
         iccy = JIcon::error;
         break;
   }
   data->icon.setImage( JIcon::sysPointer( iccy, false));

   // set the buttons
   ulong ulButtons = ((ulong) b) & ~JMsgBox::help;
   data->butts = ulButtons;
   JPushButton *cHelp = &data->butt2;

   if( b & JMsgBox::Ok)
      data->butt1.setText( "~Ok");

   if( b & JMsgBox::cancel) {
      data->butt2.setText( "~Cancel").show();
      cHelp = &data->butt3;
   }
   if( b & JMsgBox::yes) {
      data->butt1.setText( "~Yes");
      data->butt2.setText( "~No").show();
      cHelp = &data->butt3;
   }
   if( b & JMsgBox::retry) {
      data->butt1.setText( "~Retry");
      data->butt2.setText( "~Abort").show();
      data->butt3.setText( "~Ignore").show();
      cHelp = &data->buttHelp;
   }
   data->butt1.show();  // always have at least button 1

   // check for help bit
   if( ((ulong) b) & JMsgBox::help) {
      cHelp->setText( "~Help").show().setStyle( JPushButton::help);
   }

   // sort out window menu
   JMenu *m = getSystemMenu();
   m->removeItem( SC_SIZE, true)
     .removeItem( SC_MINIMIZE)
     .removeItem( SC_MAXIMIZE)
     .removeItem( SC_RESTORE)
     .removeItem( SC_HIDE);
}

JMsgBox::~JMsgBox()
{
   delete data;
}

// Methods used to set the msg text. This is quite complex --------------------
JMsgBox &JMsgBox::setMessage( const char *msg)
{
   // set the control up
   data->text.setText( msg);

   // lay out the controls.
   // First get a general spacing measurement in window coords
   JPoint space( 4, 4);
   mapPoint( space);
   // then get the size of a button
   JSize buttonSize( data->butt1.size());

   // get the height required to display the text
   ulong htText( data->text.height());
// ulong htText( data->text.size().y);

   // okay, height of window is space.y + buttonSize.y + space.y + htText
   // + space.y + titlebar.y
   // ie. 3 * space.y + buttonSize.y + htText + titlebar.y

   // checkcheckcheck there's enough space for the icon
   JSize icSize( data->icon.size());
   if( htText < (ulong) icSize.y)
      htText = icSize.y;

   // Resize the window...
   JSize aSize( size());
   aSize.y = 3 * space.y + buttonSize.y + htText +
             JSystem::query( JSystem::titlebarHeight);
   resize( aSize);
   // ...the text...
   aSize = data->text.size();
   aSize.y = htText;
   data->text.resize( aSize);
   // ...and finally move the icon.
   aSize = data->icon.origin();
   aSize.y = 2 * space.y + buttonSize.y +
             (htText - icSize.y) / 2 + space.y / 2;
   data->icon.moveTo( aSize);

   return self;
}

// Showing the box ------------------------------------------------------------
ulong JMsgBox::openModally( BOOL sys)
{
   // check we have an owner
   assertParms( data->owner, "JMsgBox::showModally");
   centreOver( data->owner);
   return runDlg( sys);
}

ulong JMsgBox::openOverPointer( BOOL sys)
{
   moveOverCursor();
   return runDlg( sys);
}

ulong JMsgBox::open( JWindow *papa)
{
   centreOver( papa);
   return runDlg();
}

ulong JMsgBox::runDlg( BOOL s)
{
   ulong rc = showModally( s);
   switch( data->butts) {
      case JMsgBox::Ok:
         return JMsgBox::Ok;
      case JMsgBox::Ok | JMsgBox::cancel:
         if( rc == j_butt_one)
            return JMsgBox::Ok;
         else return JMsgBox::cancel;
      case JMsgBox::yes | JMsgBox::no:
         if( rc == j_butt_one)
            return JMsgBox::yes;
         else return JMsgBox::no;
      case JMsgBox::retry | JMsgBox::abort | JMsgBox::ignore:
         switch( rc) {
            case j_butt_one:
               return JMsgBox::retry;
            case j_butt_two:
               return JMsgBox::abort;
            case j_butt_three:
               return JMsgBox::ignore;
         }
   }
   return JMsgBox::help;
}

// JMother scaffolding --------------------------------------------------------
BOOL JMsgBox::closing()
{
   dismiss( j_butt_two);
   return true;
}
