/*
 * JNoteBk.cpp
 *
 * Notebook control and page object
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include <string.h>
#include "JOs2.h"
#include "JNoteBk.hpp"
#include "JWindow.hpp"
#include "JCtlH.hpp"
#include "JColour.hpp"
#include "JPSpace.hpp"
#include "JIntern.hpp"
#include "JSystem.hpp"

#define book_error( s) jlib_throw( new JException( err_notebook, JExn::base, s))

void JNotebook::adjustTabs( const char *s)
{
   JWindowPS ps( this);
   JPoint    pts[ TXTBOX_COUNT];

   GpiQueryTextBox( ps, strlen( s), (PCH)s, TXTBOX_COUNT, (PPOINTL) &pts);
   JSize szNew = pts[TXTBOX_TOPRIGHT] - pts[TXTBOX_BOTTOMLEFT] + JSize( 8, 8);

   if( szNew > tabSz)
      setTabSize( szNew);
}

// Setup methods; we only support one type of notebook -----------------------
void JNotebook::setup()
{
   sendEvent( BKM_SETNOTEBOOKCOLORS, JColour::middleButton.asULong(),
              BKA_BACKGROUNDPAGECOLOR);
   sendEvent( BKM_SETNOTEBOOKCOLORS, JColour::middleButton.asULong(),
              BKA_BACKGROUNDMAJORCOLOR);
   BOOL rc = sendEvent( BKM_SETDIMENSIONS, JMP( 24, 24), BKA_PAGEBUTTON);
   if( !rc)
      book_error( "size somethin");
}

JNotebook::JNotebook( ulong hwnd) : JControl( hwnd)
{ setup(); }

JNotebook::JNotebook( JWindow *ma, ulong id) : JControl()
{
   JWindow *ctrl = ma->childWithID( id);
   if( JSystem::isWarp4()) {
      // erm, come again ?
      JSize sz = ctrl->size();
      JPoint wh = ctrl->origin();
      ctrl->destroy();

      JCreateWBlock b( ma->handle(), WC_NOTEBOOK, 0, JWindow::visible | 0x800,
                       wh, sz, ma->handle(), HWND_TOP, id, 0, 0);
      setHwnd( JInternal::window::create( &b));

   } else {
      setHwnd( ctrl->handle());
      setup();
   }
}

JNotebook::JNotebook( JWindow *ma, const JPoint &pos, const JSize &sz,
                      ulong Id, ulong style) : JControl()
{
   assertParms( ma, "JNotebook::ctor");

   style |= ( BKS_SPIRALBIND | BKS_BACKPAGESBR | BKS_MAJORTABRIGHT |
              BKS_SQUARETABS | BKS_STATUSTEXTRIGHT | BKS_TABTEXTLEFT);

   if( JSystem::isWarp4()) style |= 0x800;

   JCreateWBlock b( ma->handle(), WC_NOTEBOOK, 0, style, pos, sz,
                    ma->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
   setup();
}

JNotebook::~JNotebook()
{
   // remove all the pages
   sendEvent( BKM_DELETEPAGE, JMP(), BKA_ALL);
}

// Page manipulation ---------------------------------------------------------
JPage JNotebook::addSection( JWindow *w, const char *tab,
                             const char *status) const
{
   ulong flags = BKA_MAJOR | BKA_AUTOPAGESIZE;
   if( status) flags |= BKA_STATUSTEXTON;

   ulong pageID = sendEvent( BKM_INSERTPAGE, JMP(), JMP( flags, BKA_LAST));
   if( !pageID) {
      book_error( "insert a page");
      return JPage();
   }

   BOOL rc = sendEvent( BKM_SETPAGEWINDOWHWND, pageID, w->handle());
   if( !rc) {
      book_error( "create a page");
      return JPage();
   }

   JPage page( this, pageID);

   if( status)
      page.setStatusText( status);

   if( tab)
      page.setTabText( tab);

   return page;
}

JPage JNotebook::addPage( JWindow *w, const char *statusField) const
{
   ulong flags = BKA_AUTOPAGESIZE;
   if( statusField) flags |= BKA_STATUSTEXTON;

   ulong pageID = sendEvent( BKM_INSERTPAGE, JMP(), JMP( flags, BKA_LAST));
   if( !pageID) {
      book_error( "insert a page");
      return JPage();
   }

   BOOL rc = sendEvent( BKM_SETPAGEWINDOWHWND, pageID, w->handle());
   if( !rc) {
      book_error( "create a page");
      return JPage();
   }

   JPage page( this, pageID);

   if( statusField)
      page.setStatusText( statusField);

   return page;
}

// located insertion
JPage JNotebook::addSectionAfter( JWindow *w, const JPage &pg,
                                  const char *tab, const char *field) const
{
   ulong flags = BKA_MAJOR | BKA_AUTOPAGESIZE;
   if( field) flags |= BKA_STATUSTEXTON;

   ulong pageID = sendEvent( BKM_INSERTPAGE, pg.pageID, JMP( flags, BKA_NEXT));
   if( !pageID) {
      book_error( "insert a page");
      return JPage();
   }

   BOOL rc = sendEvent( BKM_SETPAGEWINDOWHWND, pageID, w->handle());
   if( !rc) {
      book_error( "create a page");
      return JPage();
   }

   JPage page( this, pageID);

   if( field)
      page.setStatusText( field);

   if( tab)
      page.setTabText( tab);

   return page;
}

JPage JNotebook::addPageAfter( JWindow *w, const JPage &pg,
                               const char *field) const
{
   ulong flags = BKA_AUTOPAGESIZE;
   if( field) flags |= BKA_STATUSTEXTON;

   ulong pageID = sendEvent( BKM_INSERTPAGE, pg.pageID, JMP( flags, BKA_NEXT));
   if( !pageID) {
      book_error( "insert a page");
      return JPage();
   }

   BOOL rc = sendEvent( BKM_SETPAGEWINDOWHWND, pageID, w->handle());
   if( !rc) {
      book_error( "create a page");
      return JPage();
   }

   JPage page( this, pageID);

   if( field)
      page.setStatusText( field);

   return page;
}

JNotebook &JNotebook::removePage( const JPage &pg)
{
   BOOL rc = sendEvent( BKM_DELETEPAGE, pg.pageID, BKA_SINGLE);
   if( !rc)
      book_error( "delete a page");
   return self;
}

// Misc things ---------------------------------------------------------------
int JNotebook::pages() const
{
   ulong rc = sendEvent( BKM_QUERYPAGECOUNT, JMP(), BKA_END);
   return rc - 1;
}

JRect JNotebook::pageRect() const
{
   JRect rcl( 0, 0, size());
   BOOL  rc = sendEvent( BKM_CALCPAGERECT, &rcl, true);
   if( !rc)
      book_error( "calc a rect");
   return rcl;
}

// Page & section tabs -------------------------------------------------------
JNotebook &JNotebook::refreshTabs()
{
   BOOL rc = sendEvent( BKM_INVALIDATETABS);
   if( !rc)
      book_error( "invalidate the tabs");
   return self;
}

JNotebook &JNotebook::setTabSize( const JSize &s)
{
   BOOL rc = sendEvent( BKM_SETDIMENSIONS, JMP( s.x, s.y), BKA_MAJORTAB);
   if( !rc)
      book_error( "resize the tabs");
   tabSz = s;
   return self;
}

// Setting & querying visible page -------------------------------------------
JNotebook &JNotebook::turnToPage( const JPage &pg)
{
   BOOL rc = sendEvent( BKM_TURNTOPAGE, pg.pageID);
   if( !rc)
      book_error( "setting the toppage");
   return self;
}

JNotebook &JNotebook::turnToTop()
{
   ulong Id = sendEvent( BKM_QUERYPAGEID, JMP(), BKA_FIRST);

   BOOL  rc = sendEvent( BKM_TURNTOPAGE, Id);
   if( !rc)
      book_error( "turn to the start");
   return self;
}

JPage JNotebook::topPage() const
{
   ulong Id = sendEvent( BKM_QUERYPAGEID, JMP(), BKA_TOP);
   if( !Id)
      book_error( "get the top page");

   return JPage( this, Id);
}

// Help id -------------------------------------------------------------------
JNotebook &JNotebook::setHelpID( const JHelpID &i)
{ hID = i; return self; }

JHelpID JNotebook::helpID() const
{
   if( !hID.ID())
      return JHelpID( ID());
   else
      return hID;
}

// Page cursor class ---------------------------------------------------------
JNotebook::cursor::cursor( JNotebook *bk) : pg( 0), book( bk)
{
   assertParms( bk, "JNbk::cursor::ctor");
}


JNotebook::cursor::~cursor()
{ delete pg; }

JNotebook::cursor &JNotebook::cursor::toFirst()
{
   ulong id = book->sendEvent( BKM_QUERYPAGEID, JMP(), BKA_FIRST);
   delete pg;
   pg = id ? new JPage( book, id) : 0 ;
   return self;
}

JNotebook::cursor &JNotebook::cursor::toNext()
{
   if( !isValid()) jlib_throw( new JException( err_badcursor, JExn::base));
   else {
      ulong id = book->sendEvent( BKM_QUERYPAGEID, pg->pageID, BKA_NEXT);
      delete pg;
      pg = id ? new JPage( book, id) : 0 ;
   }
   return self;
}

BOOL JNotebook::cursor::isValid() const
{ return pg != 0; }

JPage *JNotebook::cursor::current() const
{
   JPage *rc = 0;
   if( !isValid()) jlib_throw( new JException( err_badcursor, JExn::base));
   else rc = pg;
   return rc;
}

JPage *JNotebook::cursor::operator -> () const
{ return current(); }

// Event handling ------------------------------------------------------------
BOOL JNotebook::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case BKN_HELP:
         if( HelpManager.helpModel() != JHelpManager::none)
            HelpManager.showHelp( helpID());
         return true;
      case BKN_PAGESELECTED:
      {
         PAGESELECTNOTIFY *pThing = (PPAGESELECTNOTIFY) e.data();
         return justShownPage( JPage( this, pThing->ulPageIdNew),
                               JPage( this, pThing->ulPageIdCur));
      }
      case BKN_PAGESELECTEDPENDING:
      {
         PAGESELECTNOTIFY *pThing = (PPAGESELECTNOTIFY) e.data();
         BOOL rc = aboutToShowPage( JPage( this, pThing->ulPageIdCur),
                                    JPage( this, pThing->ulPageIdNew));
         if( rc) pThing->ulPageIdNew = 0;
         return false;
      }
      case BKN_NEWPAGESIZE:
         return pageSizeChanged( pageRect());
   }
   return false;
}

// Page class ----------------------------------------------------------------
JPage::JPage() : pageID( 0), book( 0)
{}

JPage::JPage( const JPage &copy) : pageID( copy.pageID), book( copy.book)
{}

JPage &JPage::operator = ( const JPage &op2)
{
   book = op2.book; pageID = op2.pageID;
   return self;
}

JPage::JPage( const JNotebook *nb, ulong i) : pageID( i), book( (JNotebook *) nb)
{
   assertParms( nb, "Jpage::ctor");
}

JPage::JPage( JNotebook *nb, ulong i) : pageID( i), book( nb)
{
   assertParms( nb, "Jpage::ctor");
}

// Status text
JStr JPage::statusText() const
{
   BOOKTEXT txt;
   long len = book->sendEvent( BKM_QUERYSTATUSLINETEXT, pageID, &txt);

   if( len == BOOKERR_INVALID_PARAMETERS) {
      book_error( "get the stat text");
      return JStr();
   }

   return JStr( txt.pString);
}

JPage &JPage::setStatusText( const char *s)
{
   BOOL rc = book->sendEvent( BKM_SETSTATUSLINETEXT, pageID, s);
   if( !rc)
      book_error( "set the stat text");
   return self;
}

// Page data
ulong JPage::data() const
{
   return book->sendEvent( BKM_QUERYPAGEDATA, pageID);
}

JPage &JPage::setData( ulong u)
{
   BOOL rc = book->sendEvent( BKM_SETPAGEDATA, pageID, u);
   if( !rc)
      book_error( "set the page data");
   return self;
}

// tab text
JStr JPage::tabText() const
{
   BOOKTEXT txt;
   long len = book->sendEvent( BKM_QUERYTABTEXT, pageID, &txt);

   if( len == BOOKERR_INVALID_PARAMETERS) {
      book_error( "get the tab text");
      return JStr();
   }

   return JStr( txt.pString);
}

JPage &JPage::setTabText( const char *s)
{
   BOOL rc = book->sendEvent( BKM_SETTABTEXT, pageID, s);
   if( !rc)
      book_error( "set the tab text");
   book->adjustTabs( s);
   return self;
}

// tab bitmaps
JBitmap JPage::tabBitmap() const
{
   ulong hBm = book->sendEvent( BKM_QUERYTABBITMAP, pageID);
   if( hBm == (ulong)BOOKERR_INVALID_PARAMETERS)
      book_error( "get the tab bitmap");

   return JBitmap( hBm);
}

JPage &JPage::setTabBitmap( const JBitmap &b)
{
   BOOL rc = book->sendEvent( BKM_SETTABBITMAP, pageID, b.handle());
   if( !rc)
      book_error( "set the tab bitmap");
   return self;
}

JWindow *JPage::window() const
{
   JWindow *rc = 0;

   ulong hWnd = book->sendEvent( BKM_QUERYPAGEWINDOWHWND, pageID);
   if( !hWnd)
      book_error( "get the page hwnd");
   else rc = JWindowPtr( hWnd);

   return rc;
}
