/*
 * JPSpace.cpp
 *
 * Presentation space classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "JOs2.h"
#include "JPSpace.hpp"
#include "JDC.hpp"
#include "JFont.hpp"
#include "JGPrim.hpp"
#include "JGLPrim.hpp"
#include "JWindow.hpp"
#include "JPM.hpp"
#include "JBitmap.hpp"
#include "JCoord.hpp"
#include "JIcon.hpp"
#include "JGfx.hpp"
#include "JColour.hpp"
#include "JSet.hpp"

// 'set-id' management doubries ----------------------------------------------
struct FontInfo
{
   ulong id; JStr name;
   FontInfo( ulong i, const JStr &s) : id( i), name( s) {}
   FontInfo( const FontInfo &c) : id( c.id), name( c.name) {}
};

const JStr &key( const FontInfo &f) { return f.name; }

typedef JKeySet<FontInfo,JStr> KnownFontList;

struct JPSData
{
   ulong         topID;
   KnownFontList fontlist;

   JPSData() : topID( 0) {}
};

ulong JBPSpace::newID() const
{ return ++(data->topID); }

ulong JBPSpace::openFont( const JFont &f) const
{
   ulong rc = 0;
   // look to see if we've already created a logical font for this Font
#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
      FontInfo *fi = data->fontlist.elementWithKey( f.face());
#ifdef REALLY_THROW_EXCEPTIONS
      rc = fi->id;
#else
      if( fi) rc = fi->id;
      else {
         jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JNotFoundError *e) {
#endif
      // create a new logical font
      ulong id = newID();
      long  r = GpiCreateLogFont( hps, NULL, id,
                                  (PFATTRS) f.fattrs().pvAddr());
      if( r == GPI_ERROR)
         pmError( 2, "GpiCreateLogFont");
      else {
         // remember it
         data->fontlist.add( FontInfo( id, f.face()));
         rc = id;
      }
   }
   return rc;
}

// Setup method ---------------------------------------------------------------
void JBPSpace::setup()
{
   assertParms( hps, "JBPSpace::setup");

   BOOL rc = GpiSetCharMode( hps, CM_MODE2);
   if( !rc)
      pmError( 3, "GpiSetCharMode");

   // this call means that when you specify colours in the PS, they should
   // be 24-bit rgb values, not indices into the system colour table.
   rc = GpiCreateLogColorTable( hps, 0, LCOLF_RGB, 0, 0, NULL);
   if( !rc)
      pmError( 2, "GpiCreateLogColorTable");
}

// Base ps class --------------------------------------------------------------
JBPSpace::JBPSpace() : data( new JPSData), hps( 0)
{}

JBPSpace::JBPSpace( ulong wrapme) : data( new JPSData), hps( wrapme)
{
   setup();
}

JBPSpace::~JBPSpace()
{
   for( ulong i = 1; i <= data->topID; i++)
      GpiDeleteSetId( hps, i);
   delete data;
}

// Win apis -------------------------------------------------------------------
JBPSpace &JBPSpace::fillRect( const JRect &rcl, const JColour &col)
{
   BOOL rc = WinFillRect( handle(), rcl, col.asULong());
   if( !rc)
      pmError( 1, "WinFillRect");
   return self;
}

JBPSpace &JBPSpace::invertRect( const JRect &rcl)
{
   BOOL rc = WinInvertRect( handle(), rcl);
   if( !rc)
      pmError( 3, "WinInvertRect - well done!");
   return self;
}

JBPSpace &JBPSpace::drawBitmapAt(const JBitmap &bmp, const JPoint &pos)
{
   BOOL rc = WinDrawBitmap( hps, bmp.handle(), NULL, pos, 0, 0, DBM_NORMAL);
   if( !rc)
      pmError( 4, "WinDrawBitmap_1");
   return self;
}

JBPSpace &JBPSpace::drawBitmapScaled( const JBitmap &bmp, const JRect &dest)
{
   BOOL rc = WinDrawBitmap( hps, bmp.handle(), NULL, (PPOINTL) &dest,
                            0, 0, DBM_NORMAL | DBM_STRETCH);
   if( !rc)
      pmError( 4, "WinDrawBitmap_2");
   return self;
}

JBPSpace &JBPSpace::drawIcon( const JIcon &ic, const JPoint &pt,
                             JGfx::icon::type t)
{
   BOOL rc = WinDrawPointer( hps, pt.x, pt.y, ic.handle(), t);
   if( !rc)
      pmError( 4, "WinDrawPointer");

   return self;
}

// Mixing DCs ----------------------------------------------------------------
JBPSpace &JBPSpace::disassociate()
{
   BOOL rc = GpiAssociate( hps, 0);
   if( !rc)
      pmError( 1, "GpiAssociate");

   return self;
}

JBPSpace &JBPSpace::associate( const JDeviceContext &dc)
{
   BOOL rc = GpiAssociate( hps, dc.handle());
   if( !rc)
      pmError( 1, "GpiAssociate");

   return self;
}

JReadOnlyDC JBPSpace::dc() const
{
   ulong hdc = GpiQueryDevice( hps);
   if( !hdc || hdc == HDC_ERROR)
      pmError( 4, "GpiQueryDevice");
   return JReadOnlyDC( hdc);
}

// Clear the ps --------------------------------------------------------------
JBPSpace &JBPSpace::clear()
{
   BOOL rc = GpiErase( hps);
   if( !rc)
      pmError( 1, "GpiErase");

   return self;
}

// Save & restore ps state ---------------------------------------------------
JSavedPS JBPSpace::save() const
{
   ulong sid = GpiSavePS( hps);
   if( sid < 1)
      pmError( 3, "GpiSavePS");

   return JSavedPS( sid);
}

JBPSpace &JBPSpace::restore( const JSavedPS &sps)
{
   BOOL rc = GpiRestorePS( hps, sps.handle());
   if( !rc)
      pmError( 1, "GpiRestorePS");

   return self;
}

// Resetting things ----------------------------------------------------------
JBPSpace &JBPSpace::reset( BOOL f1, BOOL f2)
{
   ulong fl = GRES_ATTRS;
   if( f2) fl = GRES_ALL;
   else if( f1) fl = GRES_SEGMENTS;

   BOOL rc = GpiResetPS( hps, fl);
   if( !rc)
      pmError( 2, "GpiResetPS");

   return self;
}

// Selecting bitmaps ---------------------------------------------------------
JBPSpace &JBPSpace::deselectBitmap()
{
   ulong rc = GpiSetBitmap( hps, 0);
   if( rc == HBM_ERROR)
      pmError( 2, "GpiSetBitmap");
   return self;
}

JBPSpace &JBPSpace::select( const JBitmap &bmp)
{
   ulong rc = GpiSetBitmap( hps, bmp.handle());
   if( rc == HBM_ERROR)
      pmError( 3, "GpiSetBitmap");
   return self;
}

// Units ---------------------------------------------------------------------
JGfx::units::type JBPSpace::units() const
{
   return (JGfx::units::type) ( GpiQueryPS( hps, NULL) & PS_UNITS);
}

JBPSpace &JBPSpace::setUnits( JGfx::units::type t)
{
   BOOL rc = GpiSetPS( hps, NULL, t);
   if( !rc)
      pmError( 3, "GpiSetPS_1");
   return self;
}

// Attributes - undoing ------------------------------------------------------
JBPSpace::undoT JBPSpace::undoState() const
{
   long rc = GpiQueryAttrMode( hps);
   if( rc == AM_ERROR)
      pmError( 3, "GpiQueryAttrMode");
   return (JBPSpace::undoT) rc;
}

JBPSpace &JBPSpace::setUndo( JBPSpace::undoT t)
{
   BOOL rc = GpiSetAttrMode( hps, t);
   if( !rc)
      pmError( 3, "GpiSetAttrMode");
   return self;
}

JBPSpace &JBPSpace::undo( ulong count)
{
   BOOL rc = GpiPop( hps, count);
   if( !rc)
      pmError( 3, "GpiPop");
   return self;
}

// Various settings ----------------------------------------------------------
// General
JGSettings JBPSpace::settings() const
{
   JGSettings setts;
   setts.setFGMix( (JGfx::fgMix::type) GpiQueryMix( hps));
   setts.setBGMix( (JGfx::bgMix::type) GpiQueryBackMix( hps));
   setts.setFGColour( JColour( GpiQueryColor( hps)));
   setts.setBGColour( JColour( GpiQueryBackColor( hps)));
   return setts;
}

// Lines
JLineSettings JBPSpace::lineSettings() const
{
   LINEBUNDLE b;
   long rc = GpiQueryAttrs( hps, PRIM_LINE, 0x200 - 1, (PBUNDLE) &b);
   if( rc == GPI_ALTERROR)
      pmError( 2, "GpiQueryAttrs");

   return JLineSettings( &b, (0x200 - 1) ^ rc);
}

JLineSettings JBPSpace::defLineSettings() const
{
   LINEBUNDLE b;
   BOOL rc = GpiQueryDefAttrs( hps, PRIM_LINE, 0x200 - 1, (PBUNDLE) &b);
   if( !rc)
      pmError( 4, "GpiQueryDefAttrs");

   return JLineSettings( &b, 0x200 - 1);
}

// Images
JImageSettings JBPSpace::imageSettings() const
{
   IMAGEBUNDLE b;
   long rc = GpiQueryAttrs( hps, PRIM_IMAGE, 0x10 - 1, (PBUNDLE) &b);
   if( rc == GPI_ALTERROR)
      pmError( 2, "GpiQueryAttrs");

   return JImageSettings( &b, (0x10 - 1) ^ rc);
}

JImageSettings JBPSpace::defImageSettings() const
{
   IMAGEBUNDLE b;
   BOOL rc = GpiQueryDefAttrs( hps, PRIM_IMAGE, 0x10 - 1, (PBUNDLE) &b);
   if( !rc)
      pmError( 2, "GpiQueryDefAttrs");

   return JImageSettings( &b, 0x10 - 1);
}

// Markers
JMarkerSettings JBPSpace::markerSettings() const
{
   MARKERBUNDLE b;
   long rc = GpiQueryAttrs( hps, PRIM_MARKER, 0x80 - 1, (PBUNDLE) &b);
   if( rc == GPI_ALTERROR)
      pmError( 2, "GpiQueryAttrs");

   return JMarkerSettings( &b, (0x80 - 1) ^ rc);
}

JMarkerSettings JBPSpace::defMarkerSettings() const
{
   MARKERBUNDLE b;
   BOOL rc = GpiQueryDefAttrs( hps, PRIM_MARKER, 0x80 - 1, (PBUNDLE) &b);
   if( !rc)
      pmError( 2, "GpiQueryDefAttrs");

   return JMarkerSettings( &b, 0x80 - 1);
}

// Areas
JFillSettings JBPSpace::fillSettings() const
{
   AREABUNDLE b;
   long rc = GpiQueryAttrs( hps, PRIM_AREA, 0x80 - 1, (PBUNDLE) &b);
   if( rc == GPI_ALTERROR)
      pmError( 3, "GpiQueryAttrs");

   return JFillSettings( &b, (0x80 - 1) ^ rc);
}

JFillSettings JBPSpace::defFillSettings() const
{
   AREABUNDLE b;
   BOOL rc = GpiQueryDefAttrs( hps, PRIM_AREA, 0x80 - 1, (PBUNDLE) &b);
   if( !rc)
      pmError( 1, "GpiQueryDefAttrs");

   return JFillSettings( &b, 0x80 - 1);
}

// Text
JTextSettings JBPSpace::textSettings() const
{
   CHARBUNDLE b;
   long rc = GpiQueryAttrs( hps, PRIM_CHAR, 0x2000 - 1, (PBUNDLE) &b);
   if( rc == GPI_ALTERROR)
      pmError( 2, "GpiQueryAttrs");

   return JTextSettings( &b, (0x2000 - 1) ^ rc);
}

JTextSettings JBPSpace::defTextSettings() const
{
   CHARBUNDLE b;
   BOOL rc = GpiQueryDefAttrs( hps, PRIM_CHAR, 0x2000 - 1, (PBUNDLE) &b);
   if( !rc)
      pmError( 2, "GpiQueryDefAttrs");

   return JTextSettings( &b, 0x2000 - 1);
}

// Arcs
JArcSettings JBPSpace::arcSettings() const
{
   ARCPARAMS arcParams;
   BOOL rc = GpiQueryArcParams( hps, &arcParams);
   if( !rc)
      pmError( 3, "GpiQueryArcParams");
   return JArcSettings( &arcParams);
}

JArcSettings JBPSpace::defArcSettings() const
{
   ARCPARAMS arcParams;
   BOOL rc = GpiQueryDefArcParams( hps, &arcParams);
   if( !rc)
      pmError( 3, "GpiQueryDefArcParams");
   return JArcSettings( &arcParams);
}

// Shortcuts -----------------------------------------------------------------
JBPSpace &JBPSpace::setColour( const JColour &c)
{
   JGSettings s;
   s.setFGColour( c);
   render( s);
   return self;
}

JBPSpace &JBPSpace::setMix( JGfx::fgMix::type t)
{
   JGSettings s;
   s.setFGMix( t);
   render( s);
   return self;
}

// Rendering primitives ------------------------------------------------------
JBPSpace &JBPSpace::render( const JGObject &g)
{ g.renderIn( self); return self; }

// Current things ------------------------------------------------------------
JFont JBPSpace::font() const
{
   FONTMETRICS fm;
   BOOL        rc = GpiQueryFontMetrics( hps, sizeof( FONTMETRICS), &fm);
   if( !rc)
      pmError( 4, "GpiQueryFontMetrics");

   return JFont( &fm);
}

JPoint JBPSpace::position() const
{
   JPoint p;
   BOOL rc = GpiQueryCurrentPosition( hps, p);
   if( !rc)
      pmError( 3, "GpiQueryCurrentPos");
   return p;
}

// Boundary data collecting class ---------------------------------------------
JPSBoundsData::JPSBoundsData( JBPSpace &p) : ps( p)
{
   BOOL rc = GpiSetDrawControl( ps.handle(), DCTL_BOUNDARY, DCTL_ON);
   if( !rc)
      pmError( 2, "GpiSetDrawControl");
   reset();
}

JPSBoundsData::~JPSBoundsData()
{
   BOOL rc = GpiSetDrawControl( ps.handle(), DCTL_BOUNDARY, DCTL_OFF);
   if( !rc)
      pmError( 2, "GpiSetDrawControl");
}

JPSBoundsData &JPSBoundsData::reset()
{
   BOOL rc = GpiResetBoundaryData( ps.handle());
   if( !rc)
      pmError( 2, "GpiResetBoundaryData");
   return self;
}

JRect JPSBoundsData::bounds() const
{
   JRect bounds;
   BOOL  rc = GpiQueryBoundaryData( ps.handle(), bounds);
   if( !rc)
      pmError( 2, "GpiQueryBoundaryData");
   return bounds;
}

// A micro ps for a window ---------------------------------------------------
JWindowPS::JWindowPS( JWindow *w) : JBPSpace()
{
   hps = WinGetPS( w->handle());
   if( !hps)
      pmError( 2, "WinGetPS");

   setup();
}

JWindowPS::~JWindowPS()
{
   BOOL rc = WinReleasePS( hps);
   if( !rc)
      pmError( 4, "WinReleasePS");
}

// A clip ps for a window. Don't really understand ---------------------------
JClippedPS::JClippedPS( JWindow *w, JWindow::clipType t) : JBPSpace()
{
   hps = WinGetClipPS( w->handle(), 0, t);
   if( !hps)
      pmError( 1, "WinGetClipPS");

   setup();
}

JClippedPS::~JClippedPS()
{
   BOOL rc = WinReleasePS( hps);
   if( !rc)
      pmError( 4, "WinReleasePS");
}

JDragPS::JDragPS( JWindow *w)
{
   hps = DrgGetPS( w->handle());
   if( !hps)
      pmError( 2, "DrgGetPS");

   setup();
}

JDragPS::~JDragPS()
{
   BOOL rc = DrgReleasePS( hps);
   if( !rc)
      pmError( 1, "DrgReleasePS");
}

// A standard ps -------------------------------------------------------------
JPSpace::JPSpace( const JDeviceContext &dc, JGfx::units::type u) : JBPSpace()
{
   SIZEL sizel = { 0, 0};
   hps = GpiCreatePS( JPM::current()->hab(),
                      dc.handle(),
                      (PSIZEL) &sizel,
                      ((ushort) u) | GPIT_MICRO | GPIA_ASSOC );
   if( !hps)
      pmError( 1, "GpiCreatePS");

   setup();
}

JPSpace::~JPSpace()
{
   BOOL rc = GpiDestroyPS( hps);
   if( !rc)
      pmError( 3, "GpiDestroyPS");
}

// A normal ps - for retained graphics (urgh) --------------------------------
JNormalPS::JNormalPS( const JDeviceContext &dc, JGfx::units::type u) : JBPSpace()
{
   SIZEL sizel = { 0, 0};
   hps = GpiCreatePS( JPM::current()->hab(),
                      dc.handle(),
                      (PSIZEL) &sizel,
                      ((ushort) u) | GPIT_NORMAL | GPIA_ASSOC );
   if( !hps)
      pmError( 1, "GpiCreatePS");

   setup();
}

JNormalPS::~JNormalPS()
{
   BOOL rc = GpiDestroyPS( hps);
   if( !rc)
      pmError( 3, "GpiDestroyPS");
}
