/*
 * JProfile.cpp
 *
 * Profile - ini - class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JProfile.hpp"
#include "JPM.hpp"
#include "JStr.hpp"

#include <stdio.h>  // sprintf

// ctor from a name - for an app's ini file
JProfile::JProfile( const char *file) : name( file), hIni( 0)
{
   hIni = PrfOpenProfile( JPM::current()->hab(), name);
   if( !hIni)
      pmError( 2, "PrfOpenProfile");
}

// ctor from a HINI - for OS/2 system & user ini files
JProfile::JProfile( HINI h) : hIni( h) {}

// dtor
JProfile::~JProfile()
{
   if( name.isValid())
      PrfCloseProfile( hIni);
}

// I've been advised - by gurus on OS2PMPRG FORUM - that it is *bad* to open
// the ini once, ie. in ctor, and then close it at the end. Instead, each
// operation ( read/write) is meant to be surrounded by an open / close. This
// prevents OS/2 having to write it out to disk intermittently, and so that's
// what I'm going to do.

// - 17/July - ummm... this makes prf things *very* slow, so going to do it
//             the other way.

void JProfile::check( const char **a)
{
   assertParms( a, "JProfile::check");
   // we're passed a ptr to the req'd app title.
   if( !*a)  // if it's null, use the last one
      *a = lastapp;
   else { // otherwise, set the 'last app given' member
      lastapp = *a;
   }
}

// write-access to profiles
JProfile &JProfile::write( const char *text, const char *key, const char *app)
{
   check( &app);
   BOOL rc = PrfWriteProfileString( hIni, app, key, text);
   if( !rc)
      pmError( 1, "PrfWriteProfileString");

   return self;
}

JProfile &JProfile::write( long l, const char *key, const char *app)
{
   // use PrfWriteProfileString
   return write( JVStr( "%d", l).buffer(), key, app);
}

JProfile &JProfile::write( const JVBuffer &buff, const char *key, const char *app)
{
   check( &app);
   BOOL rc = PrfWriteProfileData( hIni, app, key, buff.pvAddr(), buff.cbSize());
   if( !rc)
      pmError( 2, "PrfWriteProfileData");

   return self;
}

// read-access
JProfile &JProfile::read( JVBuffer &buff, const char *key, const char *app)
{
   BOOL rc;

   check( &app);
   ulong cb = buff.cbSize();
   rc = PrfQueryProfileData( hIni, app, key, buff.pvAddr(), &cb);
   if( !rc)
      notFound( "profile entry");

   return self;
}

// get a string value from thr ini
JStr JProfile::readString( const char *key, const char *app)
{
   JStr  returned;
   BOOL  rc;
   ulong len;

   check( &app);
   rc = PrfQueryProfileSize( hIni, app, key, &len);
   if( rc) {
      returned.setsize( len);
      PrfQueryProfileString( hIni, app, key, "", returned.buffer(), len);
   } else
      notFound( "profile entry");

   return returned;
}

long JProfile::readLong( const char *key, const char *app)
{
   long result;

   check( &app);
   result = PrfQueryProfileInt( hIni, app, key, -666);

   if( result == -666)
      notFound( "profile entry");

   return result;
}

// call to delete all keys and their values from the app name given
JProfile &JProfile::emptyApp( const char *app)
{
   return write( JStr(), 0, app);
}

// static data members to provide access to user & system ini files
JProfile JProfile::systemINI( HINI_SYSTEMPROFILE);
JProfile JProfile::userINI( HINI_USERPROFILE);
