/*
 * JSaveDlg.hpp
 *
 * The save dialog class, lifted from riscos.
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"
#include "JPM.hpp"
#include "JLibRes.h"
#include "JSaveDlg.hpp"
#include "JFile.hpp"
#include "JMsgBox.hpp"
#include "JMenu.hpp"
#include "JCmdH.hpp"

#include "JDragH.hpp"
#include "JDrgItem.hpp"
#include "JDrgXfer.hpp"

#include "JStaticG.hpp"
#include "JEntFld.hpp"
#include "JBorder.hpp"
#include "JSystem.hpp"

// drag starting handler -----------------------------------------------------
class JSaveDragHandler : public JDragHandler
{
 public:
   JSaveDragHandler() : JDragHandler() {}
   JDragItem *beginDrag( const JBeginDragEvent &e)
   {
      if( e.isLazy()) return 0;

      JSaveDlg *dlg = (JSaveDlg *) e.window()->parent();

      return dlg->beginDrag();
   }
};

static JSaveDragHandler filedraghandler;

// renderer handler ----------------------------------------------------------
class JSaveRenderHandler : public JSourceRenderHandler
{
 public:
   JSaveRenderHandler() : JSourceRenderHandler() {}
   BOOL ableToRender( JSourceRenderEvent &)
   {
      return true;
   }
   BOOL render( JSourceRenderEvent &e)
   {
      JSaveDlg *dlg = (JSaveDlg *) e.window()->parent();
      dlg->doSave( e.renderTo());
      return true;
   }
};

static JSaveRenderHandler filerenderhandler;

// private data --------------------------------------------------------------
struct JSaveDlgData
{
   // dialog components
   JStaticIcon icon;
   JEntryField ef;
   JBorder     border;
   // saveas data
   JStr       &filename;
   JStr        filetype;
   JVBuffer   *buffer;
   JIcon      *fileicon;
   // owner data
   JWindow    *owner;
   ulong       id;
   ushort      isModal;
   // client data
   JSaveDlgClient *client;

   JSaveDlgData( JSaveDlg *wnd, JWindow *wner, JVBuffer *buffr, const char *ft,
                 JStr &fn, const JResID &rID)
      : icon( wnd, j_icon_saveas),
        ef( wnd, j_control_ef),
        border( wnd, j_control_frame),
        filename( fn),
        filetype( ft ? ft : "Unknown"),
        buffer( buffr),
        owner( wner),
        id( 0),
        client( 0)
   {
      if( rID.value())
         fileicon = new JIcon( rID);
      else
         fileicon = new JIcon( j_icon_file, JModule::myResMod);
   }
  ~JSaveDlgData() { delete fileicon; }
};

// file dialog class ---------------------------------------------------------
JSaveDlg::JSaveDlg( JWindow *owner, JVBuffer *buffer, JStr &fn,
                    const JResID &resId, const char *ft)
         : JDialog( owner, j_dlg_saveas, &JWindow::theDesktopWindow,
                    JModule::myResMod),
           data( new JSaveDlgData( this, owner, buffer, ft, fn, resId))
{
   setFontNameSize( JSystem::getGeneralFont());
   // make the border look pretty
   data->border.setType( JBorder::bevelIn);

   // set up the entryfield
   data->ef.setMaxLength( 260);
   data->ef.setText( data->filename);
   data->ef.setSelection();

   data->icon.setImage( *data->fileicon);

   // give us a nice window icon (mother looks after it)
   JIcon *fileicon = new JIcon( j_icon_file, JModule::myResMod);
   setIcon( fileicon);

   // set up the system menu
   JMenu *m = getSystemMenu();
   m->removeItem( SC_SIZE, true)
     .removeItem( SC_MINIMIZE)
     .removeItem( SC_MAXIMIZE)
     .removeItem( SC_RESTORE)
     .removeItem( SC_HIDE);

   // attach a draghandler to the icon
   data->icon.attach( &filedraghandler);
   // attach a renderer to this window
   data->icon.attach( &filerenderhandler);

   // put us in the right place
   moveOverCursor();
}

void JSaveDlg::run( JSaveDlgClient *c, ulong id)
{
   data->client = c; data->id = id;
   data->isModal = false;
   show();
}

BOOL JSaveDlg::runModally( JSaveDlgClient *c, ulong id)
{
   data->client = c; data->id = id;
   data->isModal = true;
   return showModally();
}

JSaveDlg::~JSaveDlg()
{
   delete data;
}

BOOL JSaveDlg::closing()
{
   die( true);
   return true;
}

// process button clicks
BOOL JSaveDlg::command( const JCmdEvent &e)
{
   switch( e.value()) {
      case j_butt_ok:
      {
         // test the filename's validity.
         JStr fn = data->ef.getText();
         if( fn.length() > 4 && fn[ 1] == ':' && fn[ 2] == '\\')
            // yup, seems pretty valid to me
            doSave( fn);
         else {
            JMsgBox mbox( this, JStr( j_str_id_save1, JModule::myResMod),
                          "Error", JMsgBox::ok);
            mbox.open( data->owner);
         }
         return true;
      }
      case j_butt_cancel:
         die( true);
         return true;
   }
   return true;
}

// called to close down the window
void JSaveDlg::die( BOOL isCancel)
{
   hide();
   if( data->client)
      data->client->savingComplete( data->buffer, data->id, isCancel);

   if( data->isModal)
      dismiss( !isCancel);

   delete this;
}

// called to do the save; updates the wotsit & starts the thread
BOOL JSaveDlg::doSave( const char *path)
{
   data->ef.setText( path);
   start( this);
   return true;
}

ulong JSaveDlg::operator () ()
{
   BOOL rc = true;
   // okay, we assume the path's valid.
#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
      JStr path( data->ef.getText());
      // first give the owner a chance to do it
      if( !( data->client &&
             true == data->client->savingImminent( data->buffer,
                                                   path,
                                                   data->id)))
      {
         // if it declines, open a stream
         JWFile file( path); // , JStream::denyAll);

         // write the buffer to it and set the filetype
         file.write( *data->buffer);
         file.setType( data->filetype);
         // update the filename JStr
         data->filename = path;
      }
#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JException *) {
#else
   if( JPM::current()->exn) {
      jlib_catch();
#endif
      JMsgBox mb( this, JStr( j_str_id_save2, JModule::myResMod),
                  "Error", JMsgBox::ok, JMsgBox::error);
      mb.open( data->owner);
      rc = false;
   }
   return rc;
}

BOOL JSaveDlg::preHandleEvents( JEventData *e)
{
   if( e->msg == JThread::overMsgID) {
      if( e->mp1 == true) die( false);
      return true;
   }
   return false;
}

// create a dragitem
JPushedFileItem *JSaveDlg::beginDrag()
{
   JStr name = data->ef.getText();
   if( !name.isValid()) name = "File";
   JPushedFileItem *i = new JPushedFileItem( name,
                                             data->icon.getImage().handle(),
                                             JDrag::unknown,
                                             JDrag::type( data->filetype));
   i->setSupportedOps( JDrag::moveable);
   return i;
}
