/*
 * JSlider.cpp
 *
 * GUI Slider components
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "JOs2.h"

#include "JSlider.hpp"
#include "JCtlH.hpp"
#include "JCoord.hpp"
#include "JIntern.hpp"
#include "JStr.hpp"

// All base style flag definitions --------------------------------------------
const ulong JBSlider::snapToTick         = SLS_SNAPTOINCREMENT;
const ulong JBSlider::ownerDraw          = SLS_OWNERDRAW;

const ulong JVertSlider::normal          = SLS_CENTER | SLS_PRIMARYSCALE1 |
                                           SLS_HOMEBOTTOM | SLS_BUTTONSBOTTOM;
const ulong JVertSlider::sliderInMiddle  = SLS_CENTER;
const ulong JVertSlider::sliderOnLeft    = SLS_LEFT;
const ulong JVertSlider::sliderOnRight   = SLS_RIGHT;
const ulong JVertSlider::scaleOnRight    = SLS_PRIMARYSCALE1;
const ulong JVertSlider::scaleOnLeft     = SLS_PRIMARYSCALE2;
const ulong JVertSlider::homeAtBottom    = SLS_HOMEBOTTOM;
const ulong JVertSlider::homeAtTop       = SLS_HOMETOP;
const ulong JVertSlider::buttonsAtTop    = SLS_BUTTONSBOTTOM;
const ulong JVertSlider::buttonsAtBottom = SLS_BUTTONSTOP;

const ulong JHorzSlider::normal          = SLS_CENTER | SLS_PRIMARYSCALE1 |
                                           SLS_HOMELEFT | SLS_BUTTONSRIGHT;
const ulong JHorzSlider::sliderInMiddle  = SLS_CENTER;
const ulong JHorzSlider::sliderAtTop     = SLS_TOP;
const ulong JHorzSlider::sliderAtBottom  = SLS_BOTTOM;
const ulong JHorzSlider::scaleAtTop      = SLS_PRIMARYSCALE1;
const ulong JHorzSlider::scaleAtBottom   = SLS_PRIMARYSCALE2;
const ulong JHorzSlider::homeOnLeft      = SLS_HOMELEFT;
const ulong JHorzSlider::homeOnRight     = SLS_HOMERIGHT;
const ulong JHorzSlider::buttonsOnLeft   = SLS_BUTTONSLEFT;
const ulong JHorzSlider::buttonsOnRight  = SLS_BUTTONSRIGHT;

const ulong JProgressBar::normal         = JHorzSlider::normal | SLS_READONLY |
                                           SLS_RIBBONSTRIP;
// Base slider class ----------------------------------------------------------
// Constructors
JBSlider::JBSlider( JWindow *parent, const JPoint &pos, const JSize &size,
                    ulong ID, ulong style, ushort i, ushort s) : JControl()
{
   assertParms( parent, "JValueset::Jvalueset");

   SLDCDATA slData = { sizeof( SLDCDATA), i, s, i, s };

   JCreateWBlock b( parent->handle(), WC_SLIDER, 0, style | JWindow::visible,
                    pos, size, parent->handle(), HWND_TOP, ID, &slData, 0);

   setHwnd( JInternal::window::create( &b));
}

// Detent management
JDetent JBSlider::addDetent( ushort pelOffset) const
{
   ulong hDetent = sendEvent( SLM_ADDDETENT, pelOffset);
   if( !hDetent)
      pmError( 1, "SLM_ADDDETENT");
   return JDetent( hDetent);
}

JDetent JBSlider::addDetentAtArm() const
{
   return addDetent( armPosInPels());
}

JBSlider &JBSlider::removeDetent( const JDetent &d)
{
   BOOL rc = sendEvent( SLM_REMOVEDETENT, d.handle());
   if( !rc)
      pmError( 2, "SLM_REMOVEDETENT");
   return self;
}

ushort JBSlider::detentPosition( const JDetent &d) const
{
   JMR rc = sendEvent( SLM_QUERYDETENTPOS, d.handle());
   if( SLDERR_INVALID_PARAMETERS == rc.s1())
      pmError( 1, "SLM_QUERYDETENTPOS");
   return rc.s1();
}

// tick mark annotations
JBSlider &JBSlider::addTickText( ushort tick, char *label)
{
   BOOL rc = sendEvent( SLM_SETSCALETEXT, tick, label);
   if( !rc)
      pmError( 1, "SLM_SETSCALETEXT");
   return self;
}

JStr JBSlider::tickText( ushort tick) const
{
   JMR rc = sendEvent( SLM_QUERYSCALETEXT, JMP( tick, 0));
   if( SLDERR_INVALID_PARAMETERS == rc.s1())
      pmError( 1, "SLM_QUERYSCALETEXT 1");

   JStr str; str.setsize( rc.s1() + 1);
   rc = sendEvent( SLM_QUERYSCALETEXT, JMP( tick, rc.s1() + 1), str.buffer());
   if( SLDERR_INVALID_PARAMETERS == rc.s1())
      pmError( 1, "SLM_QUERYSCALETEXT 2");

   return str;
}

// loadsa metrics: helper funcs
JMR JBSlider::qsi( const JMP &mp1) const
{
   JMR rc = sendEvent( SLM_QUERYSLIDERINFO, mp1);
   if( SLDERR_INVALID_PARAMETERS == (long)rc)
      pmError( 3, "SLM_QUERYSLIDERINFO");
   return rc;
}

void JBSlider::ssi( const JMP &mp1, ulong mp2) const
{
   BOOL rc = sendEvent( SLM_SETSLIDERINFO, mp1, mp2);
   if( !rc)
      pmError( 3, "SLM_SETSLIDERINFO");
}

// shaft size
JSize JBSlider::shaftSize() const
{
   JMR rc = qsi( JMP() /* SMA_SHAFTDIMENSIONS */ );
   return JSize( rc.s1(), rc.s2());
}

JBSlider &JBSlider::setShaftBreadth( ulong b)
{
   ssi( JMP() /* SMA_SHAFTDIMENSIONS */ , b);
   return self;
}

// arm position
ushort JBSlider::armPosInPels() const
{
   return qsi( JMP( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE));
}

ushort JBSlider::armPosInTicks() const
{
   return qsi( JMP( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE));
}

JBSlider &JBSlider::moveArmInPels( ushort pels)
{
   ssi( JMP( SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), pels);
   return self;
}

JBSlider &JBSlider::moveArmInTicks( ushort tick)
{
   ssi( JMP( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE), tick);
   return self;
}

// shaft position
JPoint JBSlider::shaftPosition() const
{
   JMR rc = qsi( SMA_SHAFTPOSITION);
   return JPoint( rc.s1(), rc.s2());
}

JBSlider &JBSlider::moveShaftTo( const JPoint &pos)
{
   ssi( SMA_SHAFTPOSITION, JMP( pos.x, pos.y));
   return self;
}

// arm size
JSize JBSlider::armSize() const
{
   JMR rc = qsi( SMA_SLIDERARMDIMENSIONS);
   return JSize( rc.s1(), rc.s2());
}

JBSlider &JBSlider::setArmSize( const JSize &s)
{
   ssi( SMA_SLIDERARMDIMENSIONS, JMP( s.x, s.y));
   return self;
}

// Bit more important, tick size
ushort JBSlider::tickLength( ushort tick) const
{
   JMR rc = sendEvent( SLM_QUERYTICKSIZE, tick);
   if( SLDERR_INVALID_PARAMETERS == rc.s1())
      pmError( 3, "SMA_QUERYTICKSIZE");
   return rc.s1();
}

JBSlider &JBSlider::setTickLength( ushort tick, ushort size)
{
   BOOL rc = sendEvent( SLM_SETTICKSIZE, JMP( tick, size));
   if( !rc)
      pmError( 3, "SLM_SETTICKSIZE");
   return self;
}

JBSlider &JBSlider::setAllTicks( ushort size)
{
   BOOL rc = sendEvent( SLM_SETTICKSIZE, JMP( SMA_SETALLTICKS, size));
   if( !rc)
      pmError( 3, "SLM_SETTICKSIZE");
   return self;
}

// and irrelevent again
JPoint JBSlider::tickPosition( ushort tick) const
{
   JMR rc = sendEvent( SLM_QUERYTICKPOS, tick);
   if( !rc.s1() || !rc.s2())  // seems a bizarre way of indicating an error...
      pmError( 3, "SLM_QUERYTICKPOS");

   return JPoint( rc.s1(), rc.s2());
}

// Event handling
BOOL JBSlider::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case SLN_CHANGE:
         return changed();
      case SLN_SLIDERTRACK:
         return tracking();
      case SLN_KILLFOCUS:
         return lostFocus();
      case SLN_SETFOCUS:
         return gainedFocus();
   }
   return false;
}

// Vertical slider ------------------------------------------------------------
JVertSlider::JVertSlider( JWindow *w, const JPoint &p, const JSize &s,
                          ulong id, ushort ticks, ulong style)
             : JBSlider( w, p, s, id, style | SLS_VERTICAL, ticks, 0)
{}

// Horizontal slider ----------------------------------------------------------
JHorzSlider::JHorzSlider( JWindow *w, const JPoint &p, const JSize &s,
                          ulong id, ushort ticks, ulong style)
           : JBSlider( w, p, s, id, style | SLS_HORIZONTAL, ticks, 0)
{}

// Progress bar ---------------------------------------------------------------
// hrm, this isn't really right...
JProgressBar::JProgressBar( JWindow *w, const JPoint &p, const JSize &s,
                            ulong id, ushort ticks, ulong style)
             : JBSlider( w, p, s, id, style | SLS_HORIZONTAL, ticks, 0)
{}
