/*
 * JSpinBut.cpp
 *
 * Spinbutton control classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JSpinBut.hpp"
#include "JWindow.hpp"
#include "JCtlH.hpp"
#include "JIntern.hpp"

#define spinError( s) jlib_throw( new JException( err_spin, JExn::base, s))

// common spinbutton styles
const ulong JSpinButton::allowNumbers = SPBS_NUMERICONLY;
const ulong JSpinButton::readOnly     = SPBS_READONLY;
const ulong JSpinButton::alignLeft    = SPBS_JUSTLEFT;
const ulong JSpinButton::alignRight   = SPBS_JUSTRIGHT;
const ulong JSpinButton::alignMiddle  = SPBS_JUSTCENTER;
const ulong JSpinButton::noBorder     = SPBS_NOBORDER;
const ulong JSpinButton::fastSpin     = SPBS_FASTSPIN;

// common spinbutton behaviour -----------------------------------------------
JSpinButton::JSpinButton( JWindow *w, const JPoint &pos, const JSize &size,
                          ulong Id, ulong style) : JControl()
{
   assertParms( w, "JSpinButton::JSpinButton");

   JCreateWBlock b( w->handle(), WC_SPINBUTTON, "", JWindow::visible | style,
                    pos, size, w->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
}

JSpinButton &JSpinButton::spinUp( ulong d)
{
   BOOL rc = sendEvent( SPBM_SPINUP, d);
   if( !rc)
      spinError( "spin up");
   return self;
}

JSpinButton &JSpinButton::spinDown( ulong d)
{
   BOOL rc = sendEvent( SPBM_SPINDOWN, d);
   if( !rc)
      spinError( "spin down");
   return self;
}

JSpinButton &JSpinButton::setTextLimit( uchar sz)
{
   BOOL rc = sendEvent( SPBM_SETTEXTLIMIT, sz);
   if( !rc)
      spinError( "set the text limit");
   return self;
}

// anyone know how you 'liberate' a slave spinbutton ?
// [ - Master, master, the slaves are revolting!
//   - They've always been revolting!]
JSpinButton &JSpinButton::setMaster( JSpinButton *s)
{
   BOOL rc = sendEvent( SPBM_SETMASTER, s->handle());
   if( !rc)
      spinError( "set the master");
   return self;
}

// event handling... I seem to recall that bizarre things happen wrt notifies
// when you use master & servants, but can't quite remember what goes crazy
BOOL JSpinButton::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      case SPBN_UPARROW:
         return spanUp( e.data());
      case SPBN_DOWNARROW:
         return spanDown( e.data());
      case SPBN_SETFOCUS:
         return gainedFocus();
      case SPBN_KILLFOCUS:
         return lostFocus();
      case SPBN_ENDSPIN:
         return spinEnded();
      case SPBN_CHANGE:
         return changed();
   }
   return false;
}

// numeric spinbutton --------------------------------------------------------
const ulong JNumericSpinButton::padWithZeros = SPBS_PADWITHZEROS;
const ulong JNumericSpinButton::normal       = JSpinButton::alignRight;

JNumericSpinButton::JNumericSpinButton( JWindow *parent, const JPoint &pos,
                                        const JSize &s, ulong id, ulong style)
                   : JSpinButton( parent, pos, s, id, style | SPBS_NUMERICONLY | SPBS_MASTER)
{}

JNumericSpinButton::JNumericSpinButton( JWindow *p, JSpinButton *master,
                                        const JPoint &pos, const JSize &s,
                                        ulong id, ulong style)
                   : JSpinButton( p, pos, s, id, style | SPBS_NUMERICONLY)
{
   setMaster( master);
}
// range of allowed values
JNumericSpinButton &JNumericSpinButton::setRange( const JRange &r, BOOL upd)
{
   ulong msg = upd ? SPBM_SETLIMITS : SPBM_OVERRIDESETLIMITS;
   BOOL rc = sendEvent( msg, r.high, r.low);
   if( !rc)
      spinError( "set the limits");
   return self;
}

JRange JNumericSpinButton::range() const
{
   JRange rnge( 0, 0);
   BOOL   rc = sendEvent( SPBM_QUERYLIMITS, &rnge.high, &rnge.low);
   if( !rc)
      spinError( "get the limits");
   return rnge;
}

// current value
JNumericSpinButton &JNumericSpinButton::setValue( long val)
{
   if( ! range().contains( val))
      jlib_throw( new JBoundsError( err_bounds));
   else {

      BOOL rc = sendEvent( SPBM_SETCURRENTVALUE, val);
      if( !rc)
         spinError( "set the value");
   }

   return self;
}

long JNumericSpinButton::value() const
{
   long result;
   sendEvent( SPBM_QUERYVALUE, &result, JMP( 0, SPBQ_ALWAYSUPDATE));
   // I assume this error test is commented out for a good reason...
// if( !rc)
//    spinError( "get the value");
   return result;
}

// text spinbutton -----------------------------------------------------------
const ulong JTextSpinButton::allowAll = SPBS_ALLCHARACTERS;
const ulong JTextSpinButton::normal   = SPBS_JUSTRIGHT | SPBS_ALLCHARACTERS;

JTextSpinButton::JTextSpinButton( JWindow *parent, const JPoint &pos,
                                  const JSize &s, ulong id, ulong style)
                : JSpinButton( parent, pos, s, id, style)
{}

JTextSpinButton::JTextSpinButton( JWindow *parent, JSpinButton *master,
                                  const JPoint &pos, const JSize &sz,
                                  ulong id, ulong style)
                : JSpinButton( parent, pos, sz, id, style)
{
   setMaster( master);
}

JTextSpinButton &JTextSpinButton::setRange( const JSequence<JStr> &array)
{
   int    sz  = array.elements();
   const char **arr = new const char* [ sz];

   for( int i = 0; i < sz; i++)
      arr[ i] = *(array[ i]);

   setRange( arr, sz);

   delete [] arr;
   return self;
}

JTextSpinButton &JTextSpinButton::setRange( const char **array, ulong size)
{
   BOOL rc = sendEvent( SPBM_SETARRAY, array, size);
   if( !rc)
      spinError( "set the array");
   return self;
}

ulong JTextSpinButton::valueIndex() const
{
   ulong index;
   BOOL  rc = sendEvent( SPBM_QUERYVALUE, &index, JMP( 0, SPBQ_UPDATEIFVALID));
   if( !rc)
      spinError( "get the index");
   return index;
}

JStr JTextSpinButton::value() const
{
   JStr ret; ret.setsize( 255);
   BOOL rc = sendEvent( SPBM_QUERYVALUE, ret.buffer(), JMP( 255, SPBQ_UPDATEIFVALID));
   if( !rc)
      spinError( "get the text");
   return ret;
}

JTextSpinButton &JTextSpinButton::setValueIndex( ulong ndex)
{
   BOOL rc = sendEvent( SPBM_SETCURRENTVALUE, ndex);
   if( !rc)
      spinError( "set the index");
   return self;
}
