/*
 * JSplit.cpp
 *
 * Split window control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOS2.h"
#include "JPM.hpp"
#include "JIntern.hpp"
#include "JCtlH.hpp"
#include "JSplit.hpp"

#include <JSystem.hpp>

// This is a quick port to JLib of a control I wrote ages ago for a C
// project.  Thus it's a regular window class, registered in JIntern.
// Thus this file is `split' into two parts: the C implementation of the
// class at the bottom, and the C++ interface at the top.

// Gubbins -------------------------------------------------------------------
#define SPM_SETLEFTWINDOW  (WM_USER + 1)
#define SPM_SETRIGHTWINDOW (WM_USER + 2)
#define SPM_SETPERCENTAGE  (WM_USER + 3)
#define SPM_SETDIRECTION   (WM_USER + 4)
#define SPM_GETSPLITDATA   (WM_USER + 5)

#define SPL_CHANGED        18

typedef struct _SPLITDATA {
   LONG     lSplitPercentage;
   LONG     lSplitMinPercentage;
   LONG     lSplitMaxPercentage;
   HWND     hwndLeftClient;
   HWND     hwndRightClient;
   LONG     lSplitBarWidth;
   BOOL     isSplit;
   LONG     lSplitBarPos;
   LONG     lSplitBarMinPos;
   LONG     lSplitBarMaxPos;
   BOOL     leftRight;
} SPLITDATA, *PSPLITDATA;

const ulong JSplitWindow::normal           = 0;
const ulong JSplitWindow::fixedSeparator   = 1;
const ulong JSplitWindow::noDeleteChildren = 2;

// C++ interface -------------------------------------------------------------
JSplitWindow::JSplitWindow( JWindow *parent, const JPoint &pos,
                            const JPoint &size, ulong Id, ulong style)
{
   assertParms( parent, "JSplit::ctor");

   JCreateWBlock b( parent->handle(), WC_SPLIT, 0,
                    style | JWindow::visible, pos, size,
                    parent->handle(), HWND_TOP, Id, 0, 0);

   setHwnd( JInternal::window::create( &b));
}

JSplitWindow::~JSplitWindow()
{
   if( hasStyle( noDeleteChildren)) {
      JWindow *p( parent());
      PSPLITDATA pSplit((PSPLITDATA) sendEvent( SPM_GETSPLITDATA)());
      if( pSplit->hwndLeftClient)
         WinSetParent( pSplit->hwndLeftClient, p->handle(), true);
      if( pSplit->hwndRightClient)
         WinSetParent( pSplit->hwndRightClient, p->handle(), true);
   }
}

JSplitWindow &JSplitWindow::setLeftWindow( JWindow *w)
{
   w->setParent( this);
   sendEvent( SPM_SETLEFTWINDOW, w->handle());
   return self;
}

JSplitWindow &JSplitWindow::setRightWindow( JWindow *w)
{
   w->setParent( this);
   sendEvent( SPM_SETRIGHTWINDOW, w->handle());
   return self;
}

JSplitWindow &JSplitWindow::setOrientation( JSplitWindow::orientation o)
{
   sendEvent( SPM_SETDIRECTION, (o == JSplitWindow::leftRight));
   return self;
}

JSplitWindow &JSplitWindow::setSplitPercentage( ushort pc)
{
   sendEvent( SPM_SETPERCENTAGE, pc);
   return self;
}

BOOL JSplitWindow::event( const JCtlEvent &e)
{
   BOOL rc = false;

   if( e.notify() == SPL_CHANGED)
      rc = separatorChanged( e.data());

   return rc;
}

// C bits --------------------------------------------------------------------
#define SPLIT_100P 10000L

static HPOINTER hptrSplitH;
static HPOINTER hptrSplitV;
static HPOINTER hptrArrow;

static void SplitSize( PSPLITDATA, HWND hwnd);
static void SplitDefaultCalc( PSPLITDATA pSplit, LONG cx, LONG cy);
static void SplitBorder( HPS hps, PRECTL pRcl);

#define WMU_SETFOCUS WM_USER

MRESULT EXPENTRY fnwpSplit( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   PSPLITDATA pSplit = (PSPLITDATA) WinQueryWindowPtr( hwnd, QWL_USER);
   MRESULT mRC = NULL;

   switch (msg) {
      case WM_PAINT:
      {
         HPS   hps;
         RECTL rclWindow;
         RECTL rcl;

         hps = WinBeginPaint( hwnd, NULLHANDLE, &rclWindow);
         WinQueryWindowRect( hwnd, &rclWindow);

         if( pSplit->leftRight) {
            /* draw border round lh window */
            rcl.xLeft = rclWindow.xLeft;
            rcl.yBottom = rclWindow.yBottom;
            if (pSplit->isSplit)
               rcl.xRight = pSplit->lSplitBarPos;
            else
               rcl.xRight = rclWindow.xRight;
            rcl.yTop = rclWindow.yTop;
            SplitBorder( hps, &rcl);

            /* we don't have to be a split window - degenerate to bevel :-) */
            if( pSplit->isSplit) {
               rcl.xLeft = rcl.xRight;
               rcl.xRight = rcl.xLeft+pSplit->lSplitBarWidth;
               WinFillRect( hps, &rcl, SYSCLR_DIALOGBACKGROUND);
               /* rh window border */
               rcl.xLeft = rcl.xRight;
               rcl.xRight = rclWindow.xRight;
               SplitBorder( hps, &rcl);
            }
         } else {
            /* split window is top-to-bottom */
            rcl.xLeft = rclWindow.xLeft;
            rcl.yTop = rclWindow.yTop;
            if (pSplit->isSplit)
               rcl.yBottom = pSplit->lSplitBarPos;
            else
               rcl.yBottom = rclWindow.yBottom;
            rcl.xRight = rclWindow.xRight;
            SplitBorder( hps, &rcl);
            if( pSplit->isSplit) {
               rcl.yTop = rcl.yBottom;
               rcl.yBottom = rcl.yTop-pSplit->lSplitBarWidth;
               WinFillRect( hps, &rcl, CLR_PALEGRAY);
               rcl.yTop = rcl.yBottom;
               rcl.yBottom = rclWindow.yBottom;
               SplitBorder( hps, &rcl);
            }
         }
         WinEndPaint( hps);
         break;
      }
      case WM_HELP:
         mRC = WinSendMsg( WinQueryWindow( hwnd, QW_OWNER), msg, mp1, mp2);
         break;
      case WM_CONTEXTMENU:
      /* cmenu on border or splitbar should be sent up to make ui nicer */
      /* (yeah, hack for containers :-)                                 */
      {
         HWND hwndFocus = WinQueryFocus( HWND_DESKTOP);
         if( hwndFocus && (hwndFocus == pSplit->hwndLeftClient ||
                           hwndFocus == pSplit->hwndRightClient))
            mRC = WinSendMsg( hwndFocus, msg, mp1, mp2);
         break;
      }
      case WM_DESTROY:
         free( pSplit);
         mRC = WinDefWindowProc( hwnd, msg, mp1, mp2);
         break;
      case WM_CREATE:
         pSplit = (PSPLITDATA) calloc( sizeof( SPLITDATA), 1);
         WinSetWindowPtr( hwnd, QWL_USER, pSplit);
         if( !hptrSplitH) {
            /* first run, init statics */
            hptrSplitH = WinQuerySysPointer( HWND_DESKTOP, SPTR_SIZEWE, FALSE);
            hptrSplitV = WinQuerySysPointer( HWND_DESKTOP, SPTR_SIZENS, FALSE);
            hptrArrow = WinQuerySysPointer( HWND_DESKTOP, SPTR_ARROW, FALSE);
         }
         pSplit->lSplitBarWidth = WinQuerySysValue( HWND_DESKTOP, SV_CXSIZEBORDER);
         pSplit->lSplitPercentage = (50L * SPLIT_100P) / 100L;
         pSplit->lSplitMinPercentage = (5L * SPLIT_100P) / 100L;
         pSplit->lSplitMaxPercentage = (95L * SPLIT_100P) / 100L;
         pSplit->leftRight = TRUE;
         mRC = WinDefWindowProc( hwnd, msg, mp1, mp2);
         break;
      case SPM_GETSPLITDATA:
         mRC = pSplit;
         break;
      case SPM_SETLEFTWINDOW:
         pSplit->hwndLeftClient = HWNDFROMMP( mp1);
         pSplit->isSplit = (pSplit->hwndLeftClient && pSplit->hwndRightClient);
         SplitSize( pSplit, hwnd);
         break;
      case SPM_SETRIGHTWINDOW:
         pSplit->hwndRightClient = HWNDFROMMP( mp1);
         pSplit->isSplit = (pSplit->hwndRightClient && pSplit->hwndLeftClient);
         SplitSize( pSplit, hwnd);
         break;
      case SPM_SETPERCENTAGE:
      {
         LONG p = (SHORT1FROMMP( mp1) * SPLIT_100P) / 100L;
         if( p < pSplit->lSplitMinPercentage) p = pSplit->lSplitMinPercentage;
         if( p > pSplit->lSplitMaxPercentage) p = pSplit->lSplitMaxPercentage;
         pSplit->lSplitPercentage = p;
         SplitSize( pSplit, hwnd);
         break;
      }
      case SPM_SETDIRECTION:
         pSplit->leftRight = (BOOL) mp1;
         SplitSize( pSplit, hwnd);
         break;
      case WM_WINDOWPOSCHANGED:
      {
         PSWP swp = ((PSWP) mp1) + 1;
         if( swp->fl & SWP_SIZE)
            SplitSize( pSplit, hwnd);
//          SplitDefaultCalc( pSplit, swp->cx, swp->cy);
         break;
      }
      case WM_FOCUSCHANGE:
         if( SHORT1FROMMP( mp2) == TRUE && pSplit->hwndLeftClient)
            WinPostMsg( hwnd, WMU_SETFOCUS, 0, 0);
         break;
      case WMU_SETFOCUS:
         if( pSplit->isSplit && rand() % 2) // this is truly ugly!
            WinSetFocus( HWND_DESKTOP, pSplit->hwndRightClient);
         else
            WinSetFocus( HWND_DESKTOP, pSplit->hwndLeftClient);
         break;
      case WM_SIZE:
      {
         /* control has been sized, need to resize split windows       */
         /* the 1s and 2s come from the borders, which are width 1 pel */
         if( pSplit->leftRight) {
            if( !pSplit->isSplit) {
               /* maxiumum of one window */
               HWND hwndOnly = pSplit->hwndLeftClient ? pSplit->hwndLeftClient:
                                                        pSplit->hwndRightClient;
               if( hwndOnly)
//                WinSetWindowPos( hwndOnly, 0, 2, 2,
//                                 SHORT1FROMMP( mp2) - 4,
//                                 SHORT2FROMMP( mp2) - 4,
                  WinSetWindowPos( hwndOnly, 0, 1, 1,
                                   SHORT1FROMMP( mp2) - 2,
                                   SHORT2FROMMP( mp2) - 2,
                                   SWP_MOVE | SWP_SIZE);
            } else {
               /* both windows */
//             WinSetWindowPos( pSplit->hwndLeftClient, 0, 2, 2,
//                              pSplit->lSplitBarPos - 4,
//                              SHORT2FROMMP( mp2) - 4,
               WinSetWindowPos( pSplit->hwndLeftClient, 0, 1, 1,
                                pSplit->lSplitBarPos - 2,
                                SHORT2FROMMP( mp2) - 2,
                                SWP_MOVE | SWP_SIZE);
               WinSetWindowPos( pSplit->hwndRightClient, 0,
//                              pSplit->lSplitBarPos + pSplit->lSplitBarWidth + 2,
//                              2,
//                              SHORT1FROMMP( mp2) - pSplit->lSplitBarPos -
//                                                   pSplit->lSplitBarWidth - 4,
//                              SHORT2FROMMP( mp2) - 4,
                                pSplit->lSplitBarPos + pSplit->lSplitBarWidth + 1,
                                1,
                                SHORT1FROMMP( mp2) - pSplit->lSplitBarPos -
                                                     pSplit->lSplitBarWidth - 2,
                                SHORT2FROMMP( mp2) - 2,
                                SWP_MOVE | SWP_SIZE);
            }
         } else {
            /* up-down case [wbn to unify these code-paths...] */
            if ( !pSplit->isSplit) {
               HWND hwndOnly = pSplit->hwndLeftClient ? pSplit->hwndLeftClient:
                                                        pSplit->hwndRightClient;
               if (hwndOnly)
//                WinSetWindowPos( hwndOnly, 0, 2, 2,
//                                 SHORT1FROMMP( mp2) - 4,
//                                 SHORT2FROMMP( mp2) - 4,
                  WinSetWindowPos( hwndOnly, 0, 1, 1,
                                   SHORT1FROMMP( mp2) - 2,
                                   SHORT2FROMMP( mp2) - 2,
                                   SWP_MOVE | SWP_SIZE);
            } else {
               /* one up, one down */
               WinSetWindowPos( pSplit->hwndLeftClient, 0,
//                              2, pSplit->lSplitBarPos + 2,
//                              SHORT1FROMMP( mp2) - 4,
//                              SHORT2FROMMP( mp2) - pSplit->lSplitBarPos - 4,
                                1, pSplit->lSplitBarPos + 1,
                                SHORT1FROMMP( mp2) - 2,
                                SHORT2FROMMP( mp2) - pSplit->lSplitBarPos - 2,
                                SWP_MOVE | SWP_SIZE);
               WinSetWindowPos( pSplit->hwndRightClient, 0,
//                              2, 2,
//                              SHORT1FROMMP( mp2) - 4,
                                1, 1,
                                SHORT1FROMMP( mp2) - 2,
//                              pSplit->lSplitBarPos - pSplit->lSplitBarWidth - 4,
                                pSplit->lSplitBarPos - pSplit->lSplitBarWidth - 2,
                                SWP_MOVE | SWP_SIZE);
            }
         }
         break;
      }
      case WM_MOUSEMOVE:
         /* alter pointer to 'manipuble' over splitbar (if present) */
         // add check for 'fixed' style
         if( pSplit && pSplit->isSplit &&
             !(WinQueryWindowULong( hwnd, QWL_STYLE) & 0x01) ) {
            short x = SHORT1FROMMP( mp1), y = SHORT2FROMMP( mp1);
            if( (pSplit->leftRight && (x >= pSplit->lSplitBarPos &&
                                       x < (pSplit->lSplitBarPos + pSplit->lSplitBarWidth))) ||
               (!pSplit->leftRight && (y >= (pSplit->lSplitBarPos - pSplit->lSplitBarWidth) &&
                                       y < pSplit->lSplitBarPos)))
               WinSetPointer( HWND_DESKTOP, pSplit->leftRight ? hptrSplitH : hptrSplitV);
            else
               WinSetPointer( HWND_DESKTOP, hptrArrow);
         }
         break;
      case WM_BUTTON1DOWN:
         /* catch splitbar drags */
         // add check for 'fixed' style
         if( pSplit && pSplit->isSplit &&
             !(WinQueryWindowULong( hwnd, QWL_STYLE) & 0x01) ) {
            short x = SHORT1FROMMP( mp1), y = SHORT2FROMMP( mp1);
            if ((pSplit->leftRight && (x >= pSplit->lSplitBarPos &&
                                       x < (pSplit->lSplitBarPos + pSplit->lSplitBarWidth))) ||
               (!pSplit->leftRight && (y >= (pSplit->lSplitBarPos - pSplit->lSplitBarWidth) &&
                                       y < pSplit->lSplitBarPos)))
            {
               HPS       hps;
               TRACKINFO trackInfo;
               LONG      lWindowWidth;
               LONG      lWindowHeight;

               hps = WinGetPS( hwnd);
               /* set up always stuff for trackbar */
               WinQueryWindowRect( hwnd, &trackInfo.rclTrack);
               lWindowWidth = trackInfo.rclTrack.xRight -
                              trackInfo.rclTrack.xLeft;
               lWindowHeight = trackInfo.rclTrack.yTop -
                               trackInfo.rclTrack.yBottom;
               trackInfo.cxGrid = trackInfo.cyGrid = 1;
               trackInfo.cxKeyboard = trackInfo.cyKeyboard = 4;
               trackInfo.fs = TF_MOVE | TF_ALLINBOUNDARY;
               trackInfo.cxBorder = pSplit->lSplitBarWidth / 2;
               trackInfo.cyBorder = trackInfo.cxBorder;

               if( pSplit->leftRight) {
                  trackInfo.rclTrack.xLeft = pSplit->lSplitBarPos;
                  trackInfo.rclTrack.xRight = pSplit->lSplitBarPos +
                                              pSplit->lSplitBarWidth;
                  trackInfo.ptlMinTrackSize.x = trackInfo.ptlMinTrackSize.y = 0;
                  trackInfo.ptlMaxTrackSize.x = pSplit->lSplitBarWidth;
                  trackInfo.ptlMaxTrackSize.y = trackInfo.rclTrack.yTop;
                  trackInfo.rclBoundary = trackInfo.rclTrack;
                  trackInfo.rclBoundary.xLeft = pSplit->lSplitBarMinPos;
                  trackInfo.rclBoundary.xRight = pSplit->lSplitBarMaxPos;
               } else {
                  trackInfo.rclTrack.yTop = pSplit->lSplitBarPos;
                  trackInfo.rclTrack.yBottom = pSplit->lSplitBarPos -
                                               pSplit->lSplitBarWidth;
                  trackInfo.ptlMinTrackSize.x = trackInfo.ptlMinTrackSize.y = 0;
                  trackInfo.ptlMaxTrackSize.y = pSplit->lSplitBarWidth;
                  trackInfo.ptlMaxTrackSize.x = trackInfo.rclTrack.xRight;
                  trackInfo.rclBoundary = trackInfo.rclTrack;
                  trackInfo.rclBoundary.yTop = pSplit->lSplitBarMinPos;
                  trackInfo.rclBoundary.yBottom = pSplit->lSplitBarMaxPos;
               }
               if( WinTrackRect( hwnd, NULLHANDLE, &trackInfo)) {
                  LONG old = pSplit->lSplitPercentage;
                  if( pSplit->leftRight)
                     pSplit->lSplitPercentage = (SPLIT_100P * trackInfo.rclTrack.xLeft + lWindowWidth - 1) /
                                                lWindowWidth;
                  else {
                     pSplit->lSplitPercentage = (SPLIT_100P * trackInfo.rclTrack.yTop + lWindowHeight - 1) /
                                                lWindowHeight;
                     pSplit->lSplitPercentage = SPLIT_100P - pSplit->lSplitPercentage;
                  }
                  WinReleasePS( hps);
                  SplitSize( pSplit, hwnd);
                  if( pSplit->lSplitPercentage != old)
                     /* if we changed, send a wm_control notify */
                      WinSendMsg( WinQueryWindow( hwnd, QW_OWNER), WM_CONTROL,
                                  MPFROM2SHORT( WinQueryWindowUShort( hwnd, QWS_ID),
                                                SPL_CHANGED),
                                  MPFROMSHORT( (pSplit->lSplitPercentage*100L)/SPLIT_100P));

               } else
                  WinReleasePS( hps);
               break;
            }
         }
         break;
      default:
         mRC = WinDefWindowProc( hwnd, msg, mp1, mp2);
         break;
   }

   return mRC;
}

void SplitSize( PSPLITDATA pSplit, HWND hwnd)
{
   RECTL rcl;
   WinQueryWindowRect( hwnd, &rcl);
   SplitDefaultCalc( pSplit, rcl.xRight, rcl.yTop);
   WinSendMsg( hwnd, WM_SIZE, MPFROM2SHORT( rcl.xRight, rcl.yTop),
               MPFROM2SHORT( rcl.xRight, rcl.yTop));
   WinInvalidateRect( hwnd, NULL, FALSE);
}

void SplitDefaultCalc( PSPLITDATA pSplit, LONG cx, LONG cy)
{
   if( pSplit->leftRight) {
      pSplit->lSplitBarPos = (pSplit->lSplitPercentage*cx) /
                              SPLIT_100P;
      pSplit->lSplitBarMinPos = (pSplit->lSplitMinPercentage*cx) /
                                 SPLIT_100P;
      pSplit->lSplitBarMaxPos = (pSplit->lSplitMaxPercentage*cx) /
                                 SPLIT_100P;
   } else {
      pSplit->lSplitBarPos = ((SPLIT_100P-pSplit->lSplitPercentage)*cy)/
                               SPLIT_100P;
      pSplit->lSplitBarMinPos = ((SPLIT_100P-pSplit->lSplitMinPercentage)*cy) /
                                  SPLIT_100P;
      pSplit->lSplitBarMaxPos = ((SPLIT_100P-pSplit->lSplitMaxPercentage)*cy) /
                                  SPLIT_100P;
   }
}

void SplitBorder( HPS hps, PRECTL pRcl)
{
   POINTL ptl;

   /* draw a normal bevelin round the very outside */
   WinDrawBorder( hps, pRcl, 1, 1, 0, 0, 0x800);

   /* then inset that with a hilight to make white backgrounds look ok */
/* ptl.x = pRcl->xLeft + 1;
   ptl.y = pRcl->yBottom + 1;
   GpiSetColor( hps, CLR_PALEGRAY);
   GpiMove( hps, (PPOINTL)&ptl);
   ptl.x = pRcl->xRight - 2;
   GpiLine( hps, &ptl);
   ptl.y = pRcl->yTop - 2;
   GpiLine( hps, &ptl);
   GpiSetColor( hps, CLR_BLACK);
   ptl.x = pRcl->xLeft + 1;
   GpiLine( hps, &ptl);
   ptl.y = pRcl->yBottom + 1;
   GpiLine( hps, &ptl); */
}
