/*
 * JStr.cpp
 *
 * Small string class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JStr.hpp"
#include "JResID.hpp"
#include "JPM.hpp"
#include "JBuffer.hpp"

#include <string.h>
#include <stdarg.h>
#include <stdio.h>

// take care to ensure that the length member is the length of the string
// plus one for the null

// Reference counting class, providing us with copy on write
// deep copier
JStringRef::JStringRef( const JStringRef &r)
{
   string = strdup( r.string);
   refs = 1;
}

// ctor
JStringRef::JStringRef( const char *s)
{
   string = strdup( s?s:"");
   refs = 1;
}

// when we get delete'd, free the memory
JStringRef::~JStringRef()
{
   free( string);
}

JStr::JStr() : pRef( new JStringRef( ""))
{}

JStr::JStr( const char *c) : pRef( new JStringRef( c))
{}


JVStr::JVStr( const char *c, ...) : JStr()
{
   va_list argPtr;
   char    buff[ 2048]; // !! hmm

   assertParms( c, "JVStr::JVStr");

   va_start( argPtr, c);
   vsprintf( buff, c, argPtr);
   va_end( argPtr);

   delete pRef;
   pRef = new JStringRef( buff);
}

JStr::JStr( const JResID &id, const JModule &mod)
{
   char str[ 257]; // max length is 256 from PM
   long l = WinLoadString( JPM::current()->hab(),
                           mod,
                           id.value(),
                           257,
                           str );

   if( l) pRef = new JStringRef( str);
   else resError( "string");
}

JStr::JStr( const JStr &s)
{
   // do a shallow copy
   pRef = s.pRef;
   pRef->refs++;
}

JStr::~JStr()
{
   // decrement usage count; if zero free the string
   if( 0 == --pRef->refs)
      delete pRef;
}

void JStr::cow() // Copy On Write, of course...
{
   // if someone else is using this string, make a private copy
   if( pRef->refs > 1) {
      --pRef->refs;
      pRef = new JStringRef( *pRef);
   }
}

// Sets the size of the buffer
JStr &JStr::setsize( long l)
{
   cow();
   pRef->string = (char *) realloc( pRef->string, l);
   return self;
}

char *JStr::buffer()
{
   cow();
   return pRef->string;
}

JStr::operator const char * () const
{
   return pRef->string;
}

JStr::operator char * ()
{
   cow();
   return pRef->string;
}

ulong JStr::length( void) const
{
   return strlen( pRef->string) + 1;
}

long JStr::asInt() const
{
   return atol( pRef->string);
}

double JStr::asDouble() const
{
   return atof( pRef->string);
}

JStr &JStr::add( const char *s)
{
   assertParms( s, "JStr::add");
   cow();

   long newlen = strlen( s) + length();
   pRef->string = (char *) realloc( pRef->string, newlen);
   strcat( pRef->string, s);
   return self;
}

JStr &JStr::add( char s)
{
   cow();

   long len = length() + 1;
   pRef->string = (char *) realloc( pRef->string, len);
   pRef->string[ len - 2] = s;
   pRef->string[ len - 1] = 0;
   return self;
}

JStr &JStr::reverse( void)
{
   cow();
   strrev( pRef->string);
   return self;
}

JStr JStr::operator + ( const char *op2)
{
   // shallow copy this string
   JStr result( self);
   // this will force a deep copy and an append
   result.add( op2);
   return result;
}

JStr &JStr::operator += (const JStr &op2)
{
   add( op2);
   return self;
}

JStr &JStr::operator = ( const char *op2)
{
   // lose our string
   if( 0 == --pRef->refs) {
      free( pRef->string);
      // do this 'cos it saves having to ctor/dtor a ref object
      pRef->string = strdup( op2?op2:"");
      pRef->refs = 1;
   } else
      pRef = new JStringRef( op2);

   return self;
}

JStr &JStr::operator = ( const JStr &op2)
{
   if( 0 == --pRef->refs)
      free( pRef->string);

   pRef = op2.pRef;
   pRef->refs++;
   return self;
}

// Comparison operators ------------------------------------------------------
int JStr::operator == ( const char *op2) const
{
   assertParms( op2, "JStr::op ==");
   return !strcmp( pRef->string, op2);
}

int JStr::operator != ( const char *op2) const
{
   assertParms( op2, "JStr::op !=");
   return !!strcmp( pRef->string, op2);
}

int JStr::operator > ( const char *op2) const
{
   assertParms( op2, "JStr::op >");
   return strcmp( pRef->string, op2) < 0;
}

int JStr::operator < ( const char *op2) const
{
   assertParms( op2, "JStr::op <");
   return strcmp( pRef->string, op2) > 0;
}

static char c = 'q';

// Getting at chars ----------------------------------------------------------
char &JStr::operator [] ( ulong l)
{
   if( l >= length()) {
      jlib_throw( new JBoundsError( err_bounds));
      return c;
   }

   cow();

   return pRef->string[ l];
}

// this returns a char ( not a ref-to-a-char) so can preserve wossname
char JStr::operator [] ( ulong l) const
{
   if( l >= length()) {
      jlib_throw( new JBoundsError( err_bounds));
      return c;
   }

   return pRef->string[ l];
}

BOOL JStr::contains( const char *needle) const
{
   return strstr( pRef->string, needle) != 0;
}

JStr JStr::substring( long start, long count) const
{
   long sl = length();
   if( start >= sl) return JStr();

   if( start + count >= sl) count = sl - start - 1;

   JStr sub;
   sub.setsize( count + 1);
   strncpy( sub, pRef->string + start, count);
   sub[ count] = '\0';

   return sub;
}

BOOL JStr::endsWith( const char *s2) const
{
   ulong s2l = strlen( s2);
   ulong ml = length() - 1;
   if( s2l > ml) return false;
   return !strcmp( s2, pRef->string + ml - s2l);
}

BOOL JStr::startsWith( const char *s2) const
{
   ulong s2l = strlen( s2);
   if( s2l >= length()) return false;
   return !strncmp( s2, pRef->string, s2l);
}

JStr JStr::left( int numchars) const
{
   return substring( 0, numchars);
}

JStr JStr::right( int numchars) const
{
   return substring( length() - 1 - numchars, numchars);
}

// vbuffer methods
void *JStr::pvAddr() const
{ return pRef->string; }

ulong JStr::cbSize() const
{ return length(); }

BOOL JStr::isValid() const
{ return length() > 1; }

// hashcode
ulong JStr::hashcode() const
{
#if 0
   // RMN says to take the first 8 bytes as a number, square that number,
   // and take the three middle bytes as the hashcode.

   // Well, time is short, so I propose to square the first 4 bytes, giving
   // 8 bytes, and take the middle 4.
   // This doesn't actually get the answer right - it seems to be off by 2,
   // but it's close enough.
   uchar input[ 4] = { 0, 0, 0, 0 };

   // if there are 4 bytes in the string, get them.
   ulong l = strlen( pRef->string);
   if( l > 3) memcpy( input, pRef->string, 4);
   else // right-align them
      memcpy( input + 4 - l, pRef->string, l);

   // input = a<<16 + b
   ushort a = (*(ulong*)input) >> 16;
   ushort b = (*(ulong*)input) & 0xffff;

   // input^2 = (a*a)<<32 + (2*a*b)<<16 + b*b
   ulong lt = a * a;
   ulong mt = a * b;   // delay double
   ulong rt = b * b;

   // potential for overflow in two places, from right to middle on
   // addition, and middle to left on the initial doubling
   BOOL rOver = false;
   BOOL mOver = false;

   if( mt & (1<<31)) mOver = true;
   mt <<= 1;

   // input^2 = rH << 32 + rL
   ulong rL = rt + (mt << 16);
   if( rL < rt) {
      rOver = true;
      rL = 0xffffffff;
   }

   ulong rH = lt + (mOver << 16) + (mt >> 16) + rOver;

   // result is middle 4 of those 8 bytes
   return ((rH & 0xffff) << 16) + (rL >> 16);
#else
   // have a much simpler & less skewed algorithm from that nice man Knuth
   ulong h = 0;

   for( char *s = pRef->string; *s; s++)
      h = (h >> 28) ^ (h << 4) ^ *s;

   return h;
#endif
}
