/*
 * JStream.cpp
 *
 * Base classes for stream hierarchy
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"
#include "JPM.hpp" // oops
#include "JStream.hpp"
#include <string.h>

// thrown when trying to read past the end of a file.
JEOF::JEOF() : JException( err_eof, JExn::eof)
{}

// thrown when you write to a pipe with no reader.
JBrokenPipe::JBrokenPipe() : JException( err_brokenpipe, JExn::brokenPipe)
{}

// incomplete write
JIncompleteWrite::JIncompleteWrite( ulong n)
                 : JException( err_badwrite, JExn::incompleteWrite),
                   cbActual( n)
{}

// the base stream class -----------------------------------------------------
void JStream::dup( ulong toDup)
{
   ulong rc = DosDupHandle( toDup, &hStream);
   dosError( rc, "DosDupHandle");
};

JStream &JStream::close()
{
   if( handle() == HBAD)
      jlib_throw( new JException( err_nostream, JExn::base));
   else {
      ulong rc = DosClose( hStream);
      dosError( rc, "DosClose");
      forget();
   }
   return self;
}

// class which provides read access to streams -------------------------------

// actual reading function
void JRStream::read( void *addr, ulong cb) const
{
   assertParms( addr, "JRStream::read");
   if( handle() == HBAD)
      jlib_throw( new JException( err_nostream, JExn::base));
   else {
      ulong rc, actual;

      rc = DosRead( hStream, addr, cb, &actual);
      if( rc == ERROR_PIPE_NOT_CONNECTED) jlib_throw( new JEOF);
      else if( rc != ERROR_MORE_DATA) // what is this about ?
         dosError( rc, "DosRead");
      if( !rc && actual != cb) jlib_throw( new JEOF);
   }
}

// reading base objects from the stream
char JRStream::readc() const
{
   char res;
   read( &res, sizeof res);
   return res;
}

long JRStream::readl() const
{
   long res;
   read( &res, sizeof res);
   return res;
}

// slightly more complicated; hopefully this will turn out to be buffered by
// os2, so it won't be as painfully slow as you might think...
JStr JRStream::readLine() const
{
   JStr res;

#ifdef REALLY_THROW_EXCEPTIONS
   try {
#else
   JPM *pm( JPM::current());
#endif

      for(;;) {
         char c;
         read( &c, sizeof c);
#ifndef REALLY_THROW_EXCEPTIONS
         if( pm->type == JExn::eof) {
            jlib_catch();
            break;
         }
#endif
         if( c < 32) break;
         res.add( c);
      }
#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JEOF *eof) {
   }
#endif

   return res;
}

// static read-only stream
JRStream cin( 0);

// class which provides write access to streams ------------------------------
void JWStream::write( void *addr, ulong cb) const
{
   assertParms( addr, "JWStream::write");
   if( handle() == HBAD)
      jlib_throw( new JException( err_nostream, JExn::base));
   else {
      ulong rc, actual;

      rc = DosWrite( hStream, addr, cb, &actual);

      // order important here...
      if( rc == ERROR_BROKEN_PIPE)
         jlib_throw( new JBrokenPipe);
      else {
         dosError( rc, "DosWrite");
         if( !rc && actual != cb)
            jlib_throw( new JIncompleteWrite( actual));
      }
   }
}

// writing base objects to the stream; slightly more simple because we already
// know how big they are.
JWStream &JWStream::writeLine( const char *s)
{
   write( (void *)s, strlen( s) + 1);
   return self;
}

// static writable streams...not const, mind you.
JWStream cout( 1);
JWStream cerr( 2);
