/*
 * JSystem.cpp
 *
 * Interface to system-y things
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JSystem.hpp"
#include "JCoord.hpp"
#include "JPM.hpp"
#include "JWindow.hpp"
#include "JBitmap.hpp"
#include "JDC.hpp"
#include "JPSpace.hpp"
#include "JGPrim.hpp"
#include "JProfile.hpp"

ulong JSystem::query( JSystem::value v)
{
   ulong rc = WinQuerySysValue( JWindow::theDesktopWindow, v);
   return rc;
}

void JSystem::lockup()
{
   BOOL rc = WinLockupSystem( JPM::current()->hab());
   if( !rc)
      pmError( 3, "WinLockupSystem");
}

BOOL JSystem::unlock( const char *passwd)
{
   return WinUnlockSystem( JPM::current()->hab(), (PSZ)passwd);
}

JSize &JSystem::screen()
{
   static JSize sz( JSystem::query( JSystem::deskWidth),
                    JSystem::query( JSystem::deskHeight));

   return sz;
}

JBitmap JSystem::capture( const JRect &a)
{
   JRect area( a);
   if( area == JRect()) area = JRect( 0, 0, JSystem::screen());

   JMemoryDC dc;
   JPSpace   memPs( dc);
   JWindowPS deskPS( WinGetScreenPS( HWND_DESKTOP));
   JBitmap   bmp( area.size(), deskPS.dc().colourDepth());

   memPs.select( bmp);
   memPs.render( JBmpImage( deskPS, area));
   memPs.deselectBitmap();

   return bmp;
}

void JSystem::beep( ulong p, ulong d)
{
   DosBeep( p, d);
}

void JSystem::alarm( JSystem::alarm_type t)
{
   WinAlarm( HWND_DESKTOP, t);
}

BOOL JSystem::isToggled( JKeyEvent::vkeys v)
{
   long lks = WinGetKeyState( HWND_DESKTOP, v);
   return lks & 1;
}

BOOL JSystem::isDown( JKeyEvent::vkeys v)
{
   long lks = WinGetKeyState( HWND_DESKTOP, v);
   return lks & 0x8000;
}

// version things
static ulong qsv( ulong index)
{
   ulong rc;
   DosQuerySysInfo( index, index, &rc, sizeof rc);
   return rc;
}

ulong JSystem::getMinorOSVersion()
{
   return qsv( QSV_VERSION_MINOR);
}

ulong JSystem::getMajorOSVersion()
{
   return qsv( QSV_VERSION_MAJOR);
}

BOOL JSystem::isWarp4()
{ return getMajorOSVersion() == 20 && getMinorOSVersion() == 40; }

// fonts from the ini
static JStr getIniFont( const char *name, const char *def)
{
   JStr ret;

#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
      ret = JProfile::userINI.readString( name, "PM_SystemFonts");
      if( !ret.isValid()) ret = def;

#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JNotFoundError *e) {
#else
   if( JPM::current()->type == JExn::notFound) {
      jlib_catch();
#endif
      ret = def;
   }

   return ret;
}

JStr JSystem::getGeneralFont()
{ return getIniFont( "WindowText", "8.Helv"); }

JStr JSystem::getTitlebarFont()
{ return getIniFont( "WindowTitles", "10.Helv"); }

JStr JSystem::getMenuFont()
{ return getIniFont( "Menus", "10.Helv"); }

JStr JSystem::getIconFont() // for (eg) container objects
{ return getIniFont( "IconText", "8.Helv"); }
