/*
 * JVSet.cpp
 *
 * Value-set GUI component
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "JOs2.h"
#include "JVSet.hpp"
#include "JCoord.hpp"
#include "JStr.hpp"
#include "JIcon.hpp"
#include "JColour.hpp"
#include "JHelp.hpp"
#include "JBitmap.hpp"
#include "JDrag.hpp"
#include "JDragSet.hpp"
#include "JDrgItem.hpp"
#include "JCtlH.hpp"
#include "JMParam.hpp"
#include "JIntern.hpp"

#include <string.h>

// style ---------------------------------------------------------------------
const ulong JValueset::normal       = 0x080 | 0x020;
const ulong JValueset::border       = 0x020;
const ulong JValueset::itemBorder   = 0x040;
const ulong JValueset::scaleBitmaps = 0x080;
const ulong JValueset::custom       = 0x200;

// ctors ---------------------------------------------------------------------
JValueset::JValueset( ulong hwnd) : JControl( hwnd)
{ go(); }

JValueset::JValueset( JWindow *pw, ulong id) : JControl( pw, id)
{ go(); }

JValueset::JValueset( JWindow *parent, const JPoint &pos, const JPoint &size,
                      ulong Id, const JSize &celSize, itemType i, ulong style)
          : JControl()
{
   assertParms( parent, "JValueset::Jvalueset");

   VSCDATA vscData = { sizeof( VSCDATA), celSize.y, celSize.x };

   JCreateWBlock b( parent->handle(), WC_VALUESET, 0,
                    style | (ulong) i | JWindow::visible, pos, size,
                    parent->handle(), HWND_TOP, Id, &vscData, 0);

   setHwnd( JInternal::window::create( &b));
   go();
}

void JValueset::go()
{
   JSize cs = cells();

   int i, j;
   for( i = 1; i <= cs.x; i++)
      for( j = 1; j <= cs.y; j++)
         sendEvent( VM_SETITEMATTR, JMP( i, j),
                                    JMP( VIA_DRAGGABLE | VIA_DROPONABLE, true));
}

// cel type ------------------------------------------------------------------
JValueset::itemType JValueset::typeOf( const JPoint &cel) const
{
   short rc = sendEvent( VM_QUERYITEMATTR, JMP( cel.y, cel.x));
   if( rc == VSERR_INVALID_PARAMETERS)
      pmError( 1, "VM_QUERYITEMATTR");
   return (JValueset::itemType) ( rc & 0xf);
}

ulong JValueset::datum( const JPoint &cel) const
{
   ulong rc = sendEvent( VM_QUERYITEM, JMP( cel.y, cel.x));
   if( rc == (ulong)VSERR_INVALID_PARAMETERS)
      pmError( 3, "VM_QUERYITEM");
   return rc;
}

void JValueset::datam( const JPoint &c, ushort fs, ulong d) const
{
   BOOL rc = sendEvent( VM_SETITEMATTR, JMP( c.y, c.x), JMP( fs, true));
   if( !rc)
      pmError( 3, "VM_SETITEMATTR");
   else {
      rc = sendEvent( VM_SETITEM, JMP( c.y, c.x), d);
      if( !rc)
         pmError( 3, "VM_SETITEM");
   }
}

// cel contents --------------------------------------------------------------
#define badcel jlib_throw( new JException( err_valueset, JExn::base))
JColour JValueset::asColour( const JPoint &cel) const
{
   if( typeOf( cel) != JValueset::colour)
      badcel;

   return JColour( datum( cel));
}

JIcon JValueset::asIcon( const JPoint &cel) const
{
   if( typeOf( cel) != JValueset::icon)
      badcel;

   return JIcon( datum( cel));
}

JBitmap JValueset::asBitmap( const JPoint &cel) const
{
   if( typeOf( cel) != JValueset::bitmap)
      badcel;

   return JBitmap( datum( cel));
}

JStr JValueset::asStr( const JPoint &cel) const
{
   if( typeOf( cel) != JValueset::text)
      badcel;

   JStr string; string.setsize( 256);
   VSTEXT vsText = { string.buffer(), 255 };
   long rc = sendEvent( VM_QUERYITEM, JMP( cel.y, cel.x), &vsText);
   if( rc == VSERR_INVALID_PARAMETERS)
      pmError( 3, "VM_QUERYITEM");
   return string;
}

JValueset &JValueset::setBitmap( const JPoint &cel, const JBitmap &b)
{
   datam( cel, VIA_BITMAP, b.handle());
   return self;
}

JValueset &JValueset::setIcon( const JPoint &cel, const JIcon &i)
{
   datam( cel, VIA_ICON, i.handle());
   return self;
}

JValueset &JValueset::setColour( const JPoint &cel, const JColour &c)
{
   datam( cel, VIA_RGB, c.asULong());
   return self;
}

JValueset &JValueset::setStr( const JPoint &cel, char *txt)
{
   datam( cel, VIA_TEXT, (ulong) txt);
   return self;
}

// random attributes ---------------------------------------------------------
JValueset &JValueset::hilight( const JPoint &cel)
{
   BOOL rc = sendEvent( VM_SELECTITEM, JMP( cel.y, cel.x));
   if( !rc)
      pmError( 2, "VM_SELECTITEM");
   return self;
}

JPoint JValueset::chosen() const
{
   JMR rc = sendEvent( VM_QUERYSELECTEDITEM);
   return JPoint( rc.s2(), rc.s1());
}

JValueset &JValueset::enable( const JPoint &cel, BOOL doit)
{
   long rc =  sendEvent( VM_SETITEMATTR, JMP( cel.y, cel.x),
                                         JMP( VIA_DISABLED, !doit));
   if( rc == VSERR_INVALID_PARAMETERS)
      pmError( 2, "VM_SETITEMATTR");
   return self;
}

JValueset &JValueset::makeCustom( const JPoint &cel)
{
   long rc = sendEvent( VM_SETITEMATTR, JMP( cel.y, cel.x),
                                        JMP( VIA_OWNERDRAW, true));
   if( rc == VSERR_INVALID_PARAMETERS)
      pmError( 2, "VM_SETITEMATTR");
   return self;
}

// dimensions; oddness cos of weird pm-ness
JSize JValueset::cells()
{
   PVSCDATA pdata = (PVSCDATA) malloc( sizeof( VSCDATA));
   memset( pdata, 0, sizeof( VSCDATA));
   PWNDPARAMS pwp = (PWNDPARAMS) malloc( sizeof( WNDPARAMS));
   memset( pwp, 0, sizeof( WNDPARAMS));

   pwp->fsStatus = WPM_CTLDATA; pwp->pCtlData = pdata;
   BOOL rc = sendEvent( WM_QUERYWINDOWPARAMS, pwp);
   if( !rc) {
      pmError( 1, "WM_QWP");
      return JSize();
   }

   JSize sz( pdata->usColumnCount, pdata->usRowCount);

   free( pwp); free( pdata);

   return sz;
}

JValueset &JValueset::setSize( const JSize &s)
{
   VSCDATA vscdata = { sizeof( VSCDATA), s.y, s.x };
   WNDPARAMS wp = { WPM_CTLDATA, 0, 0, 0, 0, sizeof( VSCDATA), &vscdata };
   BOOL rc = sendEvent( WM_SETWINDOWPARAMS, &wp);
   if( !rc)
      pmError( 2, "WM_SETWNDP");
   else go();
   return self;
}

// event handling ------------------------------------------------------------
BOOL JValueset::event( const JCtlEvent &e)
{
   void *v = (void *) e.data();

   switch( e.notify()) {
      case VN_DRAGOVER:
      {
         PVSDRAGINFO pThing = (PVSDRAGINFO) v;
         JDragSet    dragset( pThing->pDragInfo);

         JDropOk isOk = isDropOk( JPoint( pThing->usColumn, pThing->usRow),
                                  dragset);

         e.setrc( JMR( (ushort) isOk.state, (ushort) isOk.op));
         return true;
      }
      case VN_DRAGLEAVE:
         dragLeave( JPoint( SHORT1FROMMP( v), SHORT2FROMMP( v)));
         return true;
      case VN_DROP:
      {
         PVSDRAGINFO pThing = (PVSDRAGINFO) v;
         JDragSet    dragset( pThing->pDragInfo);

         drop( JPoint( pThing->usColumn, pThing->usRow), dragset);
         return true;
      }
      case VN_DROPHELP:
         if( HelpManager.helpModel() != JHelpManager::none) {
            PVSDRAGINFO pThing = (PVSDRAGINFO) v;
            JDragSet    dragset( pThing->pDragInfo);

            JHelpID h = dropHelp( JPoint( pThing->usColumn, pThing->usRow),
                                  dragset);
            if( h.ID())
               HelpManager.showHelp( h);
         }
         return true;
      case VN_HELP:
         if( HelpManager.helpModel() != JHelpManager::none) {
            JHelpID h = help();
            if( h.ID())
               HelpManager.showHelp( h);
            else
               HelpManager.showHelp( JHelpID( ID()));
         }
         return true;
      case VN_INITDRAG:
      {
         PVSDRAGINIT pThing = (PVSDRAGINIT) v;
         JStdDragSet dragset( this, 1);

         JDragItem *d = initDrag( JPoint( pThing->usColumn, pThing->usRow));
         if( d) {
            d->setOffset( pThing->cx, pThing->cy);
            dragset.addDragItem( *d);
            delete d;
            dragset.drag();
         }
         return true;
      }
      case VN_ENTER:
         return opened( JPoint( SHORT2FROMMP( v), SHORT1FROMMP( v)));
      case VN_SELECT:
         return selected( JPoint( SHORT2FROMMP( v), SHORT1FROMMP( v)));
      case VN_SETFOCUS:
         return gainedFocus();
      case VN_KILLFOCUS:
         return lostFocus();
   }
   return false;
}
