/*
 * JViewer.cpp
 *
 * Sizing control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JIntern.hpp"
#include "JColour.hpp"
#include "JSystem.hpp"
#include "JMParam.hpp"

#include "JViewer.hpp"

#include "JCanvas.hpp"
#include "JScrlbar.hpp"

#include "JPosnH.hpp"
#include "JScrollH.hpp"
#include "JKeyH.hpp"

// arbitrary window ids for the children -------------------------------------
static const ulong id_scanvas_vscroll = 0x84;
static const ulong id_scanvas_hscroll = 0x85;
static const ulong id_scanvas_square  = 0x86;

// this is the window class used for the small square ------------------------
class JViewerCanvas : public JCanvas
{
 public:
   JViewerCanvas( JWindow *parent) : JCanvas( parent, JPoint(), JPoint(),
                                              id_scanvas_square) {}
   BOOL paint( JPaintEvent &e)
   {
      e.ps().fillRect( e.updateRect(), JColour::dialogBackground);
      return true;
   }
};

// this is the scanvas' private data area, containing objects for the three --
// child windows - 2 scrollbars and a canvas
struct JViewerData
{
   JScrollbar    *vert;
   JScrollbar    *horz;
   JWindow       *viewee;
   JViewerCanvas *smallSquare;
   JSize          maxSize;
   JSize          cvSize;
   BOOL           scroll; // are they visible?
   JViewerData( JWindow *viewE)
       : vert(0), horz(0), viewee(viewE), smallSquare(0), scroll(0)
   {
      maxSize = viewee->size();
   }
  ~JViewerData() {
      delete vert;
      delete horz;
      delete smallSquare;
   }
};

// Style bits ----------------------------------------------------------------
const ulong JViewer::dynamic       = 0x01;
const ulong JViewer::destroyViewee = 0x02;
const ulong JViewer::normal        = 0x03;

// this is the size handler for the viewer -----------------------------------

class JViewerSizeH : public JPositionHandler
{
 public:
   BOOL postReposition( JPostPositionEvent &e)
   {
      JSWP *oldSwp = e.getOldSWP();
      if( oldSwp->flags() & JSWP::resize) {
         JViewer *sc( (JViewer *) e.window());
         sc->adjustSize( e.getNewSWP()->size(), false, oldSwp->height());
         return true;
      }
      return false;
   }
};

static JViewerSizeH _JLibViewerSH;

// this is the vertical scroll handler for the top window --------------------
class JViewerVScrollH : public JScrollHandler
{
 public:
   JViewerVScrollH() : JScrollHandler( id_scanvas_vscroll) {}
   BOOL lineScroll( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( e.command() == JScrollEvent::lineUp)
         wnd->scrollWindow( 0, -16);
      else
         wnd->scrollWindow( 0, 16);
      return true;
   }
   BOOL pageScroll( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      long sz = wnd->data->cvSize.y;
      if( e.command() == JScrollEvent::pageUp)
         wnd->scrollWindow( 0, -sz);
      else
         wnd->scrollWindow( 0, sz);
      return true;
   }
   BOOL dragEnded( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( wnd->hasStyle( JViewer::dynamic))
         return false;
      JPoint offset = wnd->offset();
      wnd->scrollWindow( 0, e.position() - offset.y);
      wnd->invalidate(); // anyone know why we have to do this?
      return true;
   }
   BOOL tracking( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( ! wnd->hasStyle( JViewer::dynamic))
         return false;
      JPoint offset = wnd->offset();
      wnd->scrollWindow( 0, e.position() - offset.y);
      return true;
   }
};

static JViewerVScrollH _JLibViewerVSH;

// this is the horizontal scroll handler for the top window ------------------
class JViewerHScrollH : public JScrollHandler
{
 public:
   JViewerHScrollH() : JScrollHandler( id_scanvas_hscroll) {}
   BOOL lineScroll( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( e.command() == JScrollEvent::lineLeft)
         wnd->scrollWindow( 16, 0);
      else
         wnd->scrollWindow( -16, 0);
      return true;
   }
   BOOL pageScroll( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      long sz = wnd->data->cvSize.x;
      if( e.command() == JScrollEvent::pageLeft)
         wnd->scrollWindow( sz, 0);
      else
         wnd->scrollWindow( -sz, 0);
      return true;
   }
   BOOL tracking( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( ! wnd->hasStyle( JViewer::dynamic))
         return false;
      JPoint offset = wnd->offset();
      wnd->scrollWindow( offset.x - e.position(), 0);
      return true;
   }
   BOOL dragEnded( const JScrollEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( wnd->hasStyle( JViewer::dynamic))
         return false;
      JPoint offset = wnd->offset();
      wnd->scrollWindow( offset.x - e.position(), 0);
      wnd->invalidate(); // anyone know why we have to do this?
      return true;
   }
};

static JViewerHScrollH _JLibViewerHSH;

// this is the keypress handler for the viewer -------------------------------
class JViewerKeyH : public JKeyHandler
{
 public:
   JViewerKeyH() : JKeyHandler() {}
   BOOL keypress( const JKeyEvent &e)
   {
      JViewer *wnd = (JViewer *) e.window();
      if( e.isVk() && !e.isKeyUp() && wnd->data->scroll) {
         switch( e.vk()) {
            case JKeyEvent::up:
               if( !e.isClear()) break;
               wnd->sendEvent( WM_VSCROLL, id_scanvas_vscroll,
                               JMP( 0, SB_LINEUP));
               e.setrc( true);
               return true;
            case JKeyEvent::down:
               if( !e.isClear()) break;
               wnd->sendEvent( WM_VSCROLL, id_scanvas_vscroll,
                               JMP( 0, SB_LINEDOWN));
               e.setrc( true);
               return true;
            case JKeyEvent::right:
               if( !e.isClear()) break;
               wnd->sendEvent( WM_HSCROLL, id_scanvas_hscroll,
                               JMP( 0, SB_LINERIGHT));
               e.setrc( true);
               return true;
            case JKeyEvent::left:
               if( !e.isClear()) break;
               wnd->sendEvent( WM_HSCROLL, id_scanvas_hscroll,
                               JMP( 0, SB_LINELEFT));
               e.setrc( true);
               return true;
            case JKeyEvent::pageUp:
            {
               if( e.hasShift() || e.hasAlt()) break;
               ushort  id = e.hasCtrl() ? id_scanvas_hscroll : id_scanvas_vscroll;
               ulong  msg = e.hasCtrl() ? WM_HSCROLL : WM_VSCROLL;
               wnd->sendEvent( msg, id, JMP( 0, SB_PAGEUP));
               e.setrc( true);
               return true;
            }
            case JKeyEvent::pageDown:
            {
               if( e.hasShift() || e.hasAlt()) break;
               ushort id = e.hasCtrl() ? id_scanvas_hscroll : id_scanvas_vscroll;
               ulong  msg = e.hasCtrl() ? WM_HSCROLL : WM_VSCROLL;
               wnd->sendEvent( msg, id, JMP( 0, SB_PAGEDOWN));
               e.setrc( true);
               return true;
            }
            case JKeyEvent::home:
            {
               if( e.hasShift() || e.hasAlt()) break;
               ushort id = e.hasCtrl() ? id_scanvas_hscroll : id_scanvas_vscroll;
               ushort not = wnd->hasStyle( JViewer::dynamic) ?
                            SB_SLIDERTRACK : SB_SLIDERPOSITION ;
               ulong  msg = e.hasCtrl() ? WM_HSCROLL : WM_VSCROLL;
               wnd->sendEvent( msg, id, JMP( 0, not));
               e.setrc( true);
               return true;
            }
            case JKeyEvent::end:
            {
               if( e.hasShift() || e.hasAlt()) break;
               ushort id = e.hasCtrl() ? id_scanvas_hscroll : id_scanvas_vscroll;
               ushort not = wnd->hasStyle( JViewer::dynamic) ?
                            SB_SLIDERTRACK : SB_SLIDERPOSITION ;
               ulong  msg = e.hasCtrl() ? WM_HSCROLL : WM_VSCROLL;
               // this might be a bit dodgy...
               JScrollbar *bar = (JScrollbar*) wnd->childWithID( id);
               wnd->sendEvent( msg, id, JMP( bar->range().high, not));
               e.setrc( true);
               return true;
            }
            default:
               break;
         }
      }
      return false;
   }
};

static JViewerKeyH _JLibViewerKH;

// private setup method, called by ctors -------------------------------------
void JViewer::setup( JWindow *w, JWindow *v, const JPoint &p,
                     const JSize &s, ulong i, ulong st)
{
   // Create top control window.
   JCreateWBlock b( w->handle(), JInternal::window::name(), 0,
                    JWindow::visible | st | JWindow::syncpaint | JControl::tabstop,
                    p, s, w->handle(), HWND_TOP, i, 0, 0);

   setHwnd( JInternal::window::create( &b));

   // start creating and setting up children
   data = new JViewerData( v);

   // get the viewee sorted out
   data->viewee->setParent( this);
   data->viewee->setOwner( this);
   data->viewee->setStyle( JWindow::clipSiblings);

   // create canvas to cover gap between scrollbars
   data->smallSquare = new JViewerCanvas( this);
   data->smallSquare->hide();

   // horizontal scrollbar
   data->horz = new JScrollbar( this, JPoint(), JPoint(),
                                id_scanvas_hscroll, JControl::tabstop |
                                JScrollbar::horizontal | JScrollbar::dynamic);
   data->horz->hide();

   // vertical scrollbar
   data->vert = new JScrollbar( this, JPoint(), JPoint(),
                                id_scanvas_vscroll, JControl::tabstop |
                                JScrollbar::vertical | JScrollbar::dynamic);
   data->vert->hide();

   // attach size handler ( to resize controls and manipulate scrollbars)
   attach( &_JLibViewerSH);

   // attach scroll handlers ( to scroll the window)
   attach( &_JLibViewerVSH);
   attach( &_JLibViewerHSH);

   // attach keypress handler ( to handle keystrokes)
   attach( &_JLibViewerKH);

   // call adjustSize function to position & set controls
   adjustSize( s == JSize() ? data->maxSize : s, true);
}

// Ctors / dtor --------------------------------------------------------------
JViewer::JViewer( JWindow *parent, JWindow *viewee,
                  const JPoint &pos, const JSize &size,
                  ulong Id, ulong style) : JControl()
{
   assertParms( parent, "JSCanvas::JSCanvas");

   // okay, parameters valid. Get going.
   setup( parent, viewee, pos, size, Id, style);
}

JViewer::JViewer( JWindow *parent, ulong Id, JWindow *viewee, ulong style)
{
   assertParms( parent, "JSCanvas::ct");

   // find out about the guy we're gonna replace
   JWindow *ctrl = parent->childWithID( Id);
   JSize sz = ctrl->size();
   JPoint wh = ctrl->origin();
   // then grease him
   ctrl->destroy();
   // and set up ourselves
   setup( parent, viewee, wh, sz, Id, style);
}

JViewer::~JViewer()
{
   if( !hasStyle( JViewer::destroyViewee))
      data->viewee->setParent( &JWindow::theObjectWindow);
   delete data;
}

// Access to positions -------------------------------------------------------
JPoint JViewer::offset() const
{
   return JPoint( data->horz->position(), data->vert->position());
}

JRect JViewer::view() const
{
   // get a vector from the view to the viewee 0,0
   JPoint ogin = orig();
   // reverse it
   ogin *= -1;
   // get the view size
   JSize szView = size();
   if( data->scroll) {
      JSize szScroll( JSystem::query( JSystem::scrollWidth) + 1,
                      JSystem::query( JSystem::scrollHeight) );
      szView -= szScroll;
      ogin.y += szScroll.y;
   }
   return JRect( ogin, szView);
}

JPoint JViewer::orig() const
{
   JPoint tlhc = offset();
   JPoint ogin( 0 - tlhc.x, - ( data->maxSize.y - tlhc.y - size().y));
   return ogin;
}

JViewer &JViewer::scrollTo( const JPoint &p)
{
   assertParms( !( p > data->maxSize), "JViewer::scrollTo");
   JSize canvas = data->cvSize;
   JPoint to( p);

   if( p.x > data->maxSize.x - canvas.x)
      to.x = data->maxSize.x - canvas.x;
   if( p.y > data->maxSize.y - canvas.y)
      to.y = data->maxSize.y - canvas.y;
   to.y = data->maxSize.y - to.y;

   ushort notify = hasStyle( JViewer::dynamic) ?
                             SB_SLIDERTRACK : SB_SLIDERPOSITION ;

   sendEvent( WM_HSCROLL, id_scanvas_hscroll, JMP( to.x, notify));
   sendEvent( WM_VSCROLL, id_scanvas_vscroll, JMP( to.y, notify));

   return self;
}

// Max size ------------------------------------------------------------------
JSize   &JViewer::getVieweeSize() const
{
   return data->maxSize;
}

JViewer &JViewer::setVieweeSize( const JSize &s)
{
   assertParms( size() < s, "JViewer::setMaximumSize"); // fix
   data->maxSize = s;
   data->viewee->resize( s);
   JSize sz( size());
   adjustSize( sz, false, sz.y);
   return self;
}

// Get the viewee ------------------------------------------------------------
JWindow *JViewer::viewee() const
{
   return data->viewee;
}

// called when the size changes. Repositions everything. Complex. ------------
void JViewer::adjustSize( const JSize &s, BOOL first, long oldHeight)
{
   // Here's the plan:
   //  - what's the size of the window?           yP , xP
   //  - what's the maximum size of the canvas?   yL , xL
   //  - if yP < yL or xP < xL
   //    - size the canvas to xP - width of bar, yP - height of bar
   //    - display, size and set both scrollbars
   //  - else
   //    - size the canvas to xP, yP
   //    - hide the scrollbars

   if( s > data->maxSize)
      // umm... this ought not to happen, but the task manager does weird
      // things when you try to cascade or tile.
      return;

   if( s == data->maxSize) {
      // move the viewee to 0, 0
      data->viewee->moveTo( JPoint());

      // set the visible size in our data area
      data->cvSize = s;

      // hide the scrollbars etc. and set to 0
      if( data->scroll) {
         data->vert->hide();
         data->vert->moveSliderTo( 0);
         data->horz->hide();
         data->horz->moveSliderTo( 0);
         data->smallSquare->hide();
         data->scroll = false;
      }
   } else {
      // size of view must be less than that of viewee.

      // get size of scrollbars from system
      JSize scroll( JSystem::query( JSystem::scrollHeight),
                    JSystem::query( JSystem::scrollWidth));

      // get the new size of the view, update the data area, and remember
      // it - it's useful later.
      JSize szView( s - scroll);
      if( szView.x < 0) szView.x = 0;
      if( szView.y < 0) szView.y = 0;
      data->cvSize = szView;

      // position scrollbars etc.
      data->horz->reshape( JPoint(), JSize( szView.x, scroll.y));
      data->vert->reshape( JPoint( szView.x, scroll.y),
                           JSize( scroll.x, szView.y));
      data->smallSquare->reshape( JPoint( szView.x, 0), scroll);

      // Set the scrollbar ranges.
      // First get the old position of the viewer. Now, if we can keep this
      // position the same ( new size + old position < max size) this is fine,
      // but if not we move the position back so that we're flush with an edge.
      JPoint oldPos;
      JPoint newPos;

      if( !first) // offset() undefined on first invocation of this method
         oldPos = offset();

      // vertically...
      if( szView.y != oldHeight) {
         if( szView.y + oldPos.y > data->maxSize.y)
            newPos.y = data->maxSize.y - szView.y;
         else
            newPos.y = oldPos.y;
      }

      data->vert->setRange( JRange( 0, data->maxSize.y - szView.y),
                            newPos.y);
      data->vert->setSliderSize( data->maxSize.y, szView.y);

      // ...and horizontally.
      if( szView.x + oldPos.x > data->maxSize.x)
         newPos.x = data->maxSize.x - szView.x;
      else
         newPos.x = oldPos.x;

      data->horz->setRange( JRange( 0, data->maxSize.x - szView.x),
                            newPos.x);
      data->horz->setSliderSize( data->maxSize.x, szView.x);

      // update the viewee's position; note swp will filter unnecessary stuff
      data->viewee->moveTo( orig());

      // display scrollbars etc.
      if( !data->scroll) {
         data->horz->show();
         data->vert->show();
         data->smallSquare->show();
         data->scroll = true;
      }
   }
}

// Do some scrolling ---------------------------------------------------------
void JViewer::scrollWindow( long dx, long dy)
{
   // get the current position
   JPoint pos = offset();
   // get size of the view
   JSize sz = data->cvSize;

   // adjust dx and set scrollbar
   if( dx) {
      if( (pos.x + sz.x - dx) > data->maxSize.x)
         dx = pos.x - data->maxSize.x;
      else if( (pos.x - dx) < 0)
         dx = pos.x;
      data->horz->moveSliderTo( pos.x - dx);

      // move viewee
      data->viewee->moveTo( orig());
   }

   // adjust dy
   if( dy) {
      if( (pos.y + sz.y + dy) > data->maxSize.y)
         dy = data->maxSize.y - pos.y;
      else if( (pos.y + dy) < 0)
         dy = 0 - pos.y;
      data->vert->moveSliderTo( pos.y + dy);

      // move viewee
      data->viewee->moveTo( orig());
   }
}
