/*
 * JWindow.cpp
 *
 * Basic window class implementation; has window-ness
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JWindow.hpp"
#include "JHandler.hpp"
#include "JEvent.hpp"
#include "JControl.hpp"
#include "JCoord.hpp"
#include "JCtlH.hpp"
#include "JPM.hpp"
#include "JColour.hpp"
#include "JStr.hpp"
#include "JMParam.hpp"
#include "JSWP.hpp"
#include "JPointer.hpp"
#include "JSystem.hpp"
#include "JMsgH.hpp"
#include "JPSpace.hpp"
#include "JFont.hpp"
#include "JIntern.hpp"

#include "JBag.hpp"
#include "JSafeHT.hpp"
#include "JEvSem.hpp"
#include "JThread.hpp"

#include "JTrace.hpp"

#pragma priority( -999)

// exceptions ----------------------------------------------------------------
JActionForbidden::JActionForbidden()
   : JException( err_wact_forbidden, JExn::base) {}
JDuplicateWindow::JDuplicateWindow()
   : JException( err_win_duplicate, JExn::base) {}

// Window timer structures, handler ------------------------------------------
class JTimerHandler : public JMsgHandler<WM_TIMER>
{
 public:
   BOOL gotMsg( JWindow *w, JMP mp1, JMP mp2, JMR &rc);
};

static JTimerHandler windowtimerhandler;

// Private, per-window data area ---------------------------------------------
typedef JBag<JHandler *>                 JHandlers;
typedef JHashtable<JControl *,ushort>    JCtls;
typedef JHashtable<JWindowTimer*,ushort> JTimers;

// one of these per window object
struct JRealWindowData : public JWindow::Data
{
   // ulong  hwnd;
   // ushort id;
   JHandlers handlers;      // handlers attached to window
   JCtls     controls;      // 'adopted' children, get own wm_controls
   JTimers   timers;        // timers attached to the window
   ulong     tmrCount;      // next id for timers
   PFNWP     defwp;         // set when we get subclassed
   ulong     status;        // flagword, bits as below

   JRealWindowData() : tmrCount( 0), defwp(0), status( 0)
   { hwnd = 0; id = 0; }
};

#define RDATA ((JRealWindowData *) wdata)

// flags for realwindowdata.status
static const ulong autodelete  = 0x01; // kill pm window on dtor
static const ulong subclassed  = 0x02; // need to put FNWP back on dtor
static const ulong fostered    = 0x04; // have attached adoption handlers
static const ulong timered     = 0x08; // have attached timer handler
static const ulong secondclass = 0x10; // made by JWinPtr, some ops forbidden

// Window management ---------------------------------------------------------
// 1) We have a table of hwnds vs. JWindows which the user's created.
//
// 2) We have another table for those JWindows which have been created by
// JWindowPtr -- these are `second-class citizens' as the user may create a
// window using their handle at a later date.  We try to prune this table.
//
// 3) Because of linkage, we have one C window procedure external to all
// classes,but through which all messages pass.

// 1)
// the point of subclassing is to delete any jwindows left at the end
class JWindowList : public JSafeHTable<JWindow*,HWND>
{
 public:
   JWindowList( int size) : JSafeHTable<JWindow*,HWND>( size) {}
  ~JWindowList()
   {
      int sz = elements();
      if( sz != 0)
         trace << JVStr( "Warning: you have %d undeleted windows!",sz).buffer();
   }
};

static JWindowList windowList( 43); // arbitrary size...

// 2)
// the point of subclassing is to try and clean up the list
class JOtherWindowList : public JSafeHTable<JWindow*,HWND>,
                         public JThread, public JEventSem
{
   JTimer timer;
   BOOL   die;

 public:
   JOtherWindowList( int size) : JSafeHTable<JWindow*,HWND>( size),
                                 JEventSem( true),
                                 timer( 0), die( false)
   {
      timer = postEvery( 30 * 1000);  // 30s
      start();
   }

  ~JOtherWindowList()
   {
      die = true;
      timer.stop();
      post();

      cursor c( self);

      // clean up anything we created

      for(;;) {
         c.toFirst();
         if( !c.isValid()) break;
         delete *c.current(); // ~jwindow removes this from the list
      }
   }

   // periodically check for dead or promoted windows
   ulong operator()()
   {
      for(;!die;) {
         JEventSem::wait();
         if( die) break;
         reset();

         lock lk( this);
         // scan list
         cursor c( this);

         c.toFirst();
         while( c.isValid()) {
            if( !(*c.current())->isValid() ||
                windowList.contains( *c.currentKey())) {
                delete *c.current();
                c.toFirst();
            } else
               c.toNext();
         }
      }
      return 0;
   }
};

static JOtherWindowList otherWindowList( 23); // again, arbitrary

// virtual constructor for jwindows
// JWindowPtr -- the idea is use an existing window if we can; if not,
// create one & put it in the *other* window list.
JWindowPtr::JWindowPtr( ulong hwnd) : win( 0)
{
   JWindow **pWin;
#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
      pWin = windowList.get( hwnd);
#ifdef REALLY_THROW_EXCEPTIONS
      win = *pWin;
#else
      if( pWin) win = *pWin;
      else {
         jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JKeyNotFound *) {

      try {
#endif
         pWin = otherWindowList.get( hwnd);
#ifdef REALLY_THROW_EXCEPTIONS
         win = *pWin;
#else
         if( pWin) win = *pWin;
         else {
            jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
      } catch( JKeyNotFound *) {
#endif
         // a new window
         win = new JWindow( hwnd);
         win->makeSecondClass();
      }
   }
}

// 3)
// window procedure for all our windows
void * _System JLibFnWp( ulong hwnd, ulong msg, void *mp1, void *mp2)
{
   JWindowPtr ths( hwnd); // really, we want to complain if this creates...
   JEventData evt( ths, msg, mp1, mp2);

   if( !ths->preHandleEvents( &evt)) {

      JHandlers::cursor c( ((JRealWindowData*)(ths->wdata))->handlers);

      // go through the handler list offering the event
      for_cursor( c) {
         JHandler *h = *c.current();
         if( h->handle( &evt)) break;
      }

      // if we're off the end of the list, then no-one wanted the event;
      // give to def wproc
      if( !c.isValid()) {
         JRealWindowData *rwd = ((JRealWindowData*) (ths->wdata));
         if( rwd->defwp)
            evt.rc = (*rwd->defwp)( hwnd, msg, mp1, mp2);
         else
            evt.rc = WinDefWindowProc( hwnd, msg, mp1, mp2);
      }
   }

   if( WinIsWindow( JPM::current()->hab(), hwnd)) // window may have gone...
      ths->postHandleEvents( &evt);

   return evt.rc;
}

// Handlers we attach to windows in order to send things to children ----------

// 1) WM_CONTROL notifications
class JLoopyHandler : public JCtlHandler
{
 public:
   JLoopyHandler() : JCtlHandler( -1) {}
   BOOL control( const JCtlEvent &e)
   {
      BOOL handled = false;

      // look for control in list
#ifdef REALLY_THROW_EXCEPTIONS
      try {
#endif
         JRealWindowData *rwd = ((JRealWindowData*) (e.window()->wdata));
         JControl **ctrl = rwd->controls.get( e.id());
         // pass it the event
#ifdef REALLY_THROW_EXCEPTIONS
         handled = (*ctrl)->event( e);
#else
         if( ctrl) handled = (*ctrl)->event( e);
         else jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
      } catch( JKeyNotFound *) {
      }
#endif
      return handled;
   }
};

static JLoopyHandler _JLibLoopyH;

// 2) Ownerdraw things, see JOwnerDH.?pp
class JDrawLoopyHandler : public JHandler
{
 public:
   JDrawLoopyHandler() : JHandler() {}
   BOOL handle( JEventData *e)
   {
      BOOL handled = false;
      if( e->msg == WM_DRAWITEM || e->msg == WM_MEASUREITEM) {

         // look for control in list
#ifdef REALLY_THROW_EXCEPTIONS
         try {
#endif
            JRealWindowData *rwd = ((JRealWindowData*) (e->win->wdata));
            JControl **ctrl = rwd->controls.get( e->mp1.s1());
            // pass it the event
#ifdef REALLY_THROW_EXCEPTIONS
            e->rc = (*ctrl)->sendEvent( e->msg, e->mp1, e->mp2);
            handled = true;
#else
            if( ctrl) {
               e->rc = (*ctrl)->sendEvent( e->msg, e->mp1, e->mp2);
               handled = true;
            } else jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
         } catch( JKeyNotFound *) {
         }
#endif
      }
      return handled;
   }
};

static JDrawLoopyHandler drawloopyhandler;


// Definition of class JWindow -----------------------------------------------

// constructors
JWindow::JWindow() : wdata( new JRealWindowData())
{}

JWindow::JWindow( ulong h, BOOL autod, BOOL sub)
        : wdata( new JRealWindowData())
{
   assertParms( h, "JWindow::JWindow#1");

   setHwnd( h);
   if( autod) RDATA->status |= autodelete;
   if( sub) subclass();
}

JWindow::JWindow( const JPoint &pt, BOOL autodel, BOOL sub)
        : wdata( new JRealWindowData())
{
   HWND hwnd = WinWindowFromPoint( JWindow::theDesktopWindow, pt, true);
   assertParms( hwnd, "JWindow::JWindow#2");
   setHwnd( hwnd);
   if( autodel) RDATA->status |= autodelete;
   if( sub) subclass();
}

// this is the only place JRealWindowDatas get made
JWindow &JWindow::setHwnd( ulong hwnd)
{
   assertParms( hwnd, "JWindow::setHwnd");
   if( checkFirstClass()) {
      // first look in the list to see if we already exist
      if( windowList.contains( hwnd)) {
         // we do; this is bad
         jlib_throw( new JDuplicateWindow);
      } else {
         // good, new window..
         wdata->hwnd = hwnd;
         wdata->id = WinQueryWindowUShort( hwnd, QWS_ID);
         windowList.put( hwnd, this);
      }
   }
   return self;
}

// subclassed-ness; ie. allow our handler scheme to work
JWindow &JWindow::subclass()
{
   if( checkFirstClass()) {
      // check not already subclassed
      if( !(RDATA->status & subclassed)) {

         RDATA->defwp = WinSubclassWindow( handle(), JLibFnWp);
         if( !RDATA->defwp)
            pmError( 4, "WinSubclassWindow");

         RDATA->status |= subclassed;
      }
   }
   return self;
}

JWindow &JWindow::setautodelete( BOOL s)
{
   if( checkFirstClass()) {
      if( s) RDATA->status |= autodelete;
      else  RDATA->status &= ~autodelete;
   }
   return self;
}

BOOL JWindow::checkFirstClass()
{
   BOOL ok = true;
   if( RDATA && (RDATA->status & secondclass)) {
      ok = false;
      jlib_throw( new JActionForbidden);
   }
   return ok;
}

void JWindow::makeSecondClass()
{
   RDATA->status |= secondclass;
   windowList.remove( handle());
   otherWindowList.put( handle(), this);
}

JWindow::~JWindow()
{
   // We should clean up, kill the pm window if we're allowed to,
   // and forget all about it.

   // take off the loopy ones if attached..
   if( RDATA->status & fostered) {
      detach( &_JLibLoopyH);
      detach( &drawloopyhandler);
   }

   // ..timer too..
   if( RDATA->status & timered)
      detach( &windowtimerhandler);

   // ..destroy the PM window (bad for dialog controls)
   if( (RDATA->status & autodelete) && isValid())
      destroy();

   if( RDATA->status & subclassed) {
      if( !(RDATA->status & autodelete) && isValid()) {
         // take off our subclass, put the original one back
         if( !WinSubclassWindow( handle(), RDATA->defwp))
            pmError( 2, "WinSubclassWindow");
      }
   }

   // remove from list
   if( RDATA->status & secondclass)
      otherWindowList.remove( handle());
   else
      windowList.remove( handle());

   // goodnight!
   delete RDATA;
}

// misc. things --------------------------------------------------------------
BOOL JWindow::isValid() const
{
   HWND hwnd = handle();
   return ( hwnd == HWND_DESKTOP || hwnd == HWND_OBJECT) ||
            WinIsWindow( JPM::current()->hab(), hwnd);
}

// adopted controls get wm_controls via the loopyhandlers
JWindow &JWindow::adopt( JControl *c)
{
   assertParms( c, "JWindow::adopt");

   if( checkFirstClass()) {

      // check control's not already there.. (!! sorry?)
      if( !RDATA->controls.contains( c->ID())) {

         // add control to list
         RDATA->controls.put( c->ID(), c);

         // attach the loopy handler if we haven't yet done so
         if( !( RDATA->status & fostered)) {
            attach( &_JLibLoopyH);
            attach( &drawloopyhandler);
            RDATA->status |= fostered;
         }
      }
   }

   return self;
}

void JWindow::destroy()
{
   JInternal::window::destroy( handle());
}

JWindow &JWindow::setID( ushort Id)
{
   if( !WinSetWindowUShort( handle(), QWS_ID, Id))
      pmError( 3, "WinSetWindowUShort");
   RDATA->id = Id;
   return self;
}

// event handlers ------------------------------------------------------------
JWindow &JWindow::attach( JHandler *h)
{
   if( checkFirstClass()) {
      // note this doesn't test for subclassed-ness; JControl does.
      RDATA->handlers.add( h);
   }
   return self;
}

JWindow &JWindow::detach( JHandler *h)
{
#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
     RDATA->handlers.remove( h);
#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JElementNotFound *) {
   }
#else
   jlib_catch();
#endif
   return self;
}

// focus, enabled-ness, visibility -------------------------------------------
JWindow &JWindow::enable( BOOL f)
{
   if( !WinEnableWindow( handle(), f))
      pmError( 1, "WinEnableWindow");
   return self;
}

BOOL JWindow::isEnabled() const
{
   return WinIsWindowEnabled( handle());
}

JWindow &JWindow::show( BOOL f)
{
   if( !WinShowWindow( handle(), f))
      pmError( 2, "WinShowWindow");
   return self;
}

BOOL JWindow::isVisible() const
{
   return WinIsWindowVisible( handle());
}

BOOL JWindow::isShowing() const
{
   return WinIsWindowShowing( handle());
}

JWindow &JWindow::activate()
{
   BOOL rc = WinSetFocus( theDesktopWindow, handle());
   if( !rc)
      pmError( 2, "WinSetFocus");
   return self;
}

BOOL JWindow::hasFocus() const
{
   return( handle() == WinQueryFocus( JWindow::theDesktopWindow));
}

// window positioning --------------------------------------------------------
JSWP JWindow::position() const
{
   JSWP swp;
   if( !WinQueryWindowPos( handle(), swp))
      pmError( 4, "WinQueryWindowPos");
   return swp;
}

JWindow &JWindow::setPosition( const JSWP &swp)
{
   if( !WinSetWindowPos( handle(), swp.hwndBehind(), swp.pos().x,
                         swp.pos().y, swp.size().x, swp.size().y,
                         swp.flags()))
      pmError( 3, "WinSetWindowPos");

   return self;
}

JWindow &JWindow::moveOverCursor()
{
   JSize  sz = size();
   JSize  szScreen = JSystem::screen();
   JPoint newPos;

   // first center beneath mouse
   newPos = JPointer::pos() - sz / 2;

   // then move so the entire window is visible, aligning along the top-right
   // ie. if window is > than screen, align TRH corner
   if( newPos.x + sz.x > szScreen.x) newPos.x = szScreen.x - sz.x;
   else if( newPos.x < 0) newPos.x = 0;

   if( newPos.y + sz.y > szScreen.y) newPos.y = szScreen.y - sz.y;
   else if( newPos.y < 0) newPos.y = 0;

   moveTo( newPos);
   return self;
}

JWindow &JWindow::centreOver( JWindow *papa)
{
   JSize  szNicole = size();
   JSize  szPapa   = papa->size();
   JSize  szScreen = JSystem::screen();
   JPoint posPapa  = papa->origin();
   JPoint posNicole;

   posNicole = posPapa + ( szPapa - szNicole) / 2;

   // check we're on-screen
   if( posNicole.x + szNicole.x > szScreen.x)
      posNicole.x = szScreen.x - szNicole.x;
   else if( posNicole.x < 0) posNicole.x = 0;

   if( posNicole.y + szNicole.y > szScreen.y)
      posNicole.y = szScreen.y - szNicole.y;
   else if( posNicole.x < 0) posNicole.x = 0;

   moveTo( posNicole);
   return self;
}

// z-order
JWindow &JWindow::toTop()
{
   JSWP swp( JSWP::zorder);
   swp.hwndBehind() = HWND_TOP;
   return setPosition( swp);
}

JWindow &JWindow::toBottom()
{
   JSWP swp( JSWP::zorder);
   swp.hwndBehind() = HWND_BOTTOM;
   return setPosition( swp);
}

// Relationships with other windows ------------------------------------------

JWindow *JWindow::owner() const
{
   return JWindowPtr( WinQueryWindow( handle(), QW_OWNER));
}

JWindow &JWindow::setOwner( JWindow *w)
{
   assertParms( w, "JWindow:;setOwner");
   if( !WinSetOwner( handle(), w->handle()))
      pmError( 3, "WinSetOwner");

   return self;
}

JWindow *JWindow::parent() const
{
   return JWindowPtr( WinQueryWindow( handle(), QW_PARENT));
}

JWindow &JWindow::setParent( JWindow *w)
{
   assertParms( w, "JWindow:;setParent");
   if( !WinSetParent( handle(), w->handle(), true))
      pmError( 2, "WinSetParent");

   return self;
}

JWindow *JWindow::childWithID( ushort id)
{
   ulong h;

   h = WinWindowFromID( handle(), id);
   if( !h)
      pmError( 3, "WinWindowFromID");

   return h ? (JWindow *) JWindowPtr( h) : 0;
}

// The 'text' in a window ----------------------------------------------------

JWindow &JWindow::setText( const char *txt)
{
   if( !WinSetWindowText( handle(), txt))
      pmError( 1, "WinSetWindowText");

   return self;
}

JStr JWindow::getText() const
{
   JStr string;

   long l = WinQueryWindowTextLength( handle());
   string.setsize( l + 1);
   WinQueryWindowText( handle(), l + 1, string);

   return string;
}

// static data members
JWindow JWindow::theDesktopWindow( HWND_DESKTOP);
JWindow JWindow::theObjectWindow( HWND_OBJECT);

// What the window looks like ------------------------------------------------

// font
JWindow &JWindow::setFont( const JFont &f, ulong pointsize)
{
   return setFontNameSize( JVStr( "%d.%s", pointsize, f.face().buffer()));
}

JWindow &JWindow::setFontNameSize( const char *ppfont)
{
   if( !WinSetPresParam( handle(), PP_FONTNAMESIZE,
                         strlen( ppfont) + 1, (void*) ppfont))
      pmError( 1, "WinSetPresParam");
   return self;
}

JWindow &JWindow::removeFont()
{
   if( !WinRemovePresParam( handle(), PP_FONTNAMESIZE))
      pmError( 4, "WinRemovePresParam");
   return self;
}

JStr JWindow::getFontNameSize()
{
   char buf[ 128];

   if( !WinQueryPresParam( handle(), PP_FONTNAMESIZE, 0, 0, 128, &buf, 0))
      pmError( 3, "WinQueryPresParam");
   return JStr( buf);
}

// colour
JWindow &JWindow::setColour( JPP::colour param, const JColour &color)
{
   if( !WinSetPresParam( handle(), param, sizeof( RGB2), color))
      pmError( 3, "WinSetPresParam");
   return self;
}

JColour JWindow::getColour( JPP::colour param, BOOL pure)
{
   RGB2  rgb;
   ulong found;

   if( !WinQueryPresParam( handle(), param, 0, &found, sizeof( RGB2), &rgb,
                           pure ? QPF_PURERGBCOLOR : 0))
      pmError( 3, "WinQueryPresParam");

   if( found != param)
      notFound( "presentation parameter");
   return JColour( rgb.bRed, rgb.bGreen, rgb.bBlue);
}

// window style
JWindow &JWindow::setStyle( ulong flags, BOOL or)
{
   if( or)
      flags |= WinQueryWindowULong( handle(), QWL_STYLE);
   if( !WinSetWindowULong( handle(), QWL_STYLE, flags))
      pmError( 2, "WinSetWindowULong");
   return self;
}

BOOL JWindow::hasStyle( ulong flags) const
{
   if( ( flags & WinQueryWindowULong( handle(), QWL_STYLE)) == flags)
      return true;
   else return false;
}

const unsigned long JWindow::normal = 0;
const unsigned long JWindow::syncpaint = WS_SYNCPAINT;
const unsigned long JWindow::animate = WS_ANIMATE;
const unsigned long JWindow::clipChildren = WS_CLIPCHILDREN;
const unsigned long JWindow::clipSiblings = WS_CLIPSIBLINGS;
const unsigned long JWindow::clipToParent = WS_PARENTCLIP;
const unsigned long JWindow::saveBits = WS_SAVEBITS;
const unsigned long JWindow::visible = WS_VISIBLE;
const unsigned long JWindow::minimized = WS_MINIMIZED;

// pointer capture -----------------------------------------------------------
JWindow &JWindow::capturePointer()
{
   if( !WinSetCapture( JWindow::theDesktopWindow, handle()))
      pmError( 2, "WinSetCapture");
   return self;
}

JWindow &JWindow::releasePointer()
{
   if( !WinSetCapture( JWindow::theDesktopWindow, NULLHANDLE))
      pmError( 3, "WinSetCapture");
   return self;
}

BOOL JWindow::hasPointer()
{
   return ( handle() == WinQueryCapture( JWindow::theDesktopWindow));
}

// coordinate conversion -----------------------------------------------------
JPoint JWindow::fromDesktopCoords( const JPoint &from)
{
   return JWindow::convertPos( JWindow::theDesktopWindow, self, from);
}

JPoint JWindow::toDesktopCoords( const JPoint &from)
{
   return JWindow::convertPos( self, JWindow::theDesktopWindow, from);
}

// (static)
JPoint JWindow::convertPos( JWindow &from, JWindow &to, const JPoint &cpt)
{
   JPoint pt = cpt;
   if( !WinMapWindowPoints( from, to, pt, 1))
      pmError( 1, "WinMapWindowPoints");
   return pt;
}

// message processing --------------------------------------------------------
JMR JWindow::defaultAction( const JEvent &e)
{
   return JMR( (*RDATA->defwp)( handle(), e.data()->msg,
                                e.data()->mp1, e.data()->mp2));
}

JMR JWindow::defaultAction( ulong msg, const JMP &mp1, const JMP &mp2)
{
   return JMR( (*RDATA->defwp)( handle(), msg, mp1, mp2));
}

JMR JWindow::sendEvent( const JEvent &e) const
{
   return JWindow::SendEvent( handle(), e.data()->msg,
                              e.data()->mp1, e.data()->mp2);
}

JWindow &JWindow::postEvent( const JEvent &e)
{
   JWindow::PostEvent( handle(), e.data()->msg, e.data()->mp1, e.data()->mp2);
   return self;
}

JMR JWindow::sendEvent( ulong msg, const JMP &mp1, const JMP &mp2) const
{
   return JWindow::SendEvent( handle(), msg, mp1, mp2);
}

JWindow &JWindow::postEvent( ulong msg, const JMP &mp1, const JMP &mp2)
{
   JWindow::PostEvent( handle(), msg, mp1, mp2);
   return self;
}

// (static)
JMR JWindow::SendEvent( ulong hwnd, ulong msg, const JMP &mp1, const JMP &mp2)
{
   return WinSendMsg( hwnd, msg, mp1, mp2);
}

// (static)
void JWindow::PostEvent( ulong hwnd, ulong msg, const JMP &mp1, const JMP &mp2)
{
   if( !WinPostMsg( hwnd, msg, mp1, mp2))
      pmError( 1, "WinPostMsg");
}

// painting support ----------------------------------------------------------
// (static)
JWindowPS JWindow::getDesktopPS()
{
   return JWindowPS( WinGetScreenPS( JWindow::theDesktopWindow));
}

JWindow &JWindow::invalidate( const JRect &rcl)
{
   if( !WinInvalidateRect( handle(),
                           rcl == JRect() ? NULL : (PRECTL) &rcl,
                           true)) pmError( 1, "WinInvalidateRect");
   return self;
}

JWindow &JWindow::invalidate( ulong region)
{
   if( !WinInvalidateRegion( handle(), region, true))
      pmError( 2, "WinInvalidateRegion");
   return self;
}

JWindow &JWindow::validate( const JRect &rcl)
{
   if( !WinValidateRect( handle(), (RECTL *) &rcl, true))
      pmError( 3, "WinValidateRect");
   return self;
}

JWindow &JWindow::validate( ulong region)
{
   if( !WinValidateRegion( handle(), region, true))
      pmError( 1, "WinValidateRegion");
   return self;
}

JRect JWindow::updateRect() const
{
   JRect rcl;
   if( !WinQueryUpdateRect( handle(), rcl))
      pmError( 1, "WinQueryUpdateRect");
   return rcl;
}

ulong JWindow::updateRegion() const
{
   ulong region;
   if( RGN_ERROR == WinQueryUpdateRegion( handle(), region))
      pmError( 3, "WinQueryUpdateRegion");
   return region;
}

// locking paint update
JWindow &JWindow::lock( BOOL l)
{
   BOOL rc = WinLockWindowUpdate( HWND_DESKTOP, l ? handle() : 0);
   if( !rc)
      pmError( 4, "WinLockWindowUpdate");
   return self;
}

// Window timers --------------------------------------------------------------
JWindow &JWindow::startTimer( JWindowTimer *timer, ulong delay)
{
   assertParms( timer, "JWindow::startTimer");

   if( checkFirstClass()) {

      if( !( RDATA->status & timered)) {
         attach( &windowtimerhandler);
         RDATA->status |= timered;
      }

      RDATA->timers.put( ++RDATA->tmrCount, timer);
      ulong rc = WinStartTimer( JPM::current()->hab(), handle(),
                                RDATA->tmrCount, delay);
      if( !rc)
         pmError( 2, "WinStartTimer");
   }

   return self;
}

JWindow &JWindow::resetTimer( JWindowTimer *timer, ulong delay)
{
   assertParms( timer, "JWindow::resetTimer");

   if( checkFirstClass()) {
      JTimers::cursor c( RDATA->timers);
   
      for_cursor( c)
         if( *c.current() == timer) break;
   
      if( c.isValid()) {
         ulong rc = WinStartTimer( JPM::current()->hab(), handle(),
                                   *c.currentKey(), delay);
         if( !rc)
            pmError( 2, "WinRestartTimer");
      }
   }

   return self;
}

JWindow &JWindow::stopTimer( JWindowTimer *timer)
{
   assertParms( timer, "JWindow::stopTimer");

   if( checkFirstClass()) {
   
      JTimers::cursor c( RDATA->timers);
   
      for_cursor( c)
         if( *c.current() == timer) break;
   
      if( c.isValid()) {
         BOOL rc = WinStopTimer( JPM::current()->hab(), handle(), *c.currentKey());
         if( !rc)
            pmError( 2, "WinStopTimer");
         RDATA->timers.remove( *c.currentKey());
      }
   }

   return self;
}

// wm_timer handler implementation
BOOL JTimerHandler::gotMsg( JWindow *w, JMP mp1, JMP, JMR &)
{
   ushort idTimer = mp1.s1();
   if( idTimer < TID_USERMAX) {
#ifdef REALLY_THROW_EXCEPTIONS
      try {
#endif
         JRealWindowData *rwd = (JRealWindowData*) (w->wdata);
         JWindowTimer **timer = rwd->timers.get( idTimer);
#ifdef REALLY_THROW_EXCEPTIONS
         (**timer)( w);
         return true;
#else
         if( timer) {
            (**timer)(w);
            return true;
         } else jlib_catch();
#endif
#ifdef REALLY_THROW_EXCEPTIONS
      } catch( JKeyNotFound *) {
      }
#endif
   }
   return false;
}
