/*
 * JBag.hpp
 *
 * Bag collection header file
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jbag_h
#define _jbag_h

#include "JLib.h"
#include "JColln.hpp"

// Basic bag - no order, element equality okay ------------------------------
template < class T>
class JBag : public JCollection
{
 protected:
   // as a bag, we don't care about the result here & allow duplicates
   virtual int compare( void *e1, void *e2, BOOL)
   { return !(realElement( *((T*)e1)) == realElement( *((T*)e2))); }

   void deleteElement( void *element) { delete (T*) element; }

 public:
   void add( const T &element)
        { JCollection::add( new T( element)); }
   void remove( const T &element)
        { JCollection::remove( (void*) &element); }
   BOOL contains( const T &element) const
        { return !!JCollection::find( (void*) &element); }

   typedef JIterator<T> iterator;

   void doForAll( const iterator &iter)
        { JCollection::doForAll( (const JVIterator &) iter); }

   typedef JConstIterator<T> const_iterator;                             

   void doForAll( const const_iterator &iter)
        { JCollection::doForAll( (const JVIterator &) iter); }

   typedef JCursor<T> cursor;

   void removeAt( const cursor &cursor)
        { JCollection::removeAt( (const JVCursor &) cursor); }

  ~JBag() { empty(); }
};

// Sorted bag - order defined, element equality okay ------------------------
template < class T>
class JSortedBag : public JBag<T>
{
   // still don't care about duplicates
   int compare( void *e1, void *e2, BOOL)
   {
      T &t1 = *(T*)e1; T &t2 = *(T*)e2;
      return realCompare( realElement( t1), realElement( t2));
   }

 public:

   typedef JIterator<T> iterator;
   typedef JConstIterator<T> const_iterator;
   typedef JCursor<T> cursor;
};

// Keyed bag - order defined, key equality okay -----------------------------
template < class T, class K>
class JKeyBag : public JKeyCollection
{
 protected:
   // sort by key, still a bag
   int compare( void *e1, void *e2, BOOL)
   {
      T &t1 = *(T*)e1; T &t2 = *(T*)e2;
      return realCompare( key( realElement( t1)), key( realElement( t2));
   }

   int compareK( void *el, void *keyy)
   {
      T &t = *(T*)el; K &k = *(K*)keyy;
      return realCompare( k, key( realElement( t)));
   }

   void deleteElement( void *element) { delete (T*) element; }

 public:
   void add( const T &element)
        { JCollection::add( new T( element)); }
   void removeElementWithKey( const K &k)
        { JKeyCollection::removeK( (void*) &k); }
   void removeElementsWithKey( const K &k)
        { JKeyCollection::removeAllK( (void*) &k); }
   BOOL containsElementWithKey( const K &k) const
        { return !!JKeyCollection::findK((void*) &k); }
   T   *elementWithKey( const K &k)
        { return (T*)JKeyCollection::withK( (void*) &k); }
   const T *elementWithKey( const K &k) const
        { return (T*)JKeyCollection::withK( (void*) &k); }

   typedef JCursor<T> cursor;

   void removeAt( const cursor &cursor)
        { JCollection::removeAt( (const JVCursor &) cursor); }

   typedef JIterator<T> iterator;

   void doForAll( const iterator &iter)
        { JCollection::doForAll( const JVIterator &) iter); }

   typedef JConstIterator<T> const_iterator;

   void doForAll( const const_iterator &iter)
        { JCollection::doForAll( const JVIterator &) iter); }

  ~JKeyBag() { empty(); }
};

#endif
