/*
 * JCnr.hpp
 *
 * Container control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jcnr_h
#define _jcnr_h

#include "JLib.h"
#include "JControl.hpp"
#include "JCoord.hpp"
#include "JStr.hpp"
#include "JHelp.hpp"
#include "JCnr2.hpp"

#include "JCnrRec.hpp"
#include "JCnrCol.hpp"

#include "JDrag.hpp"

class JWindow;
class JCtlEvent;

class JCnrIterator;
class JCnrRecord;
class JCnrFilter;
class JCnrSorter;
class JCnrColumn;
class JMenu;

class JContainer : public JControl
{
   friend class JCnrHandler;

   void setInfo( long fl, void *cnrInfo) const;
   void getInfo( void *cnrInfo) const;

   BOOL iterate( JCnrRecord *obj, JCnrIterator &i);
   void setup();

   // data
   JMenu *menuRec, *menuCnr;
   ushort flStatus, currentOp;
   JHelpID hlpID;

   // list of things that the last menu-operation was relavent to
   JCnrRecordList sourced;

 public:
   // style - surprisingly few entries here
   static const ulong
      readOnly,
      autoPosition,
      normal;

   // constructors
   JContainer( ulong hwnd);
   JContainer( JWindow *w, ulong id);
   JContainer( JWindow *parent, const JPoint &pos, const JSize &size,
               ulong id, ulong style = JContainer::normal,
               JCnr::selType sel = JCnr::extended);

   // How the container looks ------------------------------------------------

   // get/set current view
   JContainer      &setView( JCnr::view_type v);
   JCnr::view_type  view() const;

   // get/set title of container
   JContainer &addTitle( const char *text,
                         JCnr::title_align = JCnr::title_centre,
                         BOOL hasSeparator = true, BOOL readOnly = true);
   JContainer &setTitle( const char *text);
   JStr        title() const;
   // open the edit mle on the title.
   JContainer &editTitle();

   // custom background. Need to attach a JCnrBGPaintHandler.
   JContainer &setCustomBackground( BOOL set = true);
   // custom record/column. Do something with a JOwnerDrawHandler.
   JContainer &setCustomRecord( BOOL set = true);

   // set source emphasis to the container ( for context menu)
   JContainer &setSource( BOOL on = true);

   // inserting columns
   JContainer &insert( JCnrColumn *col, JCnr::rec_insert = JCnr::last);
   JContainer &insert( JCnrColumn *col, JCnrColumn *afterMe);
   JContainer &insert( JCnrColumnList &l, JCnr::rec_insert = JCnr::last);
   JContainer &insert( JCnrColumnList &l, JCnrColumn *afterMe);

   // querying columns
   JCnrColumn *firstColumn() const;
   JCnrColumn *lastColumn() const;
   JCnrColumnList &allColumns( JCnrColumnList &l) const;

   // scrolling split windows
   JContainer &scrollLeftWindow( long amt);
   JContainer &scrollRightWindow( long amt);

   // show/hide record text
   JContainer &showText( BOOL show = true);
   JContainer &hideText();

   JContainer &scroll( JCnr::scroll_dir dir, long distance);

   // tree view settings
   JContainer &showTreeLines( BOOL show = true);
   JContainer &hideTreeLines() { return showTreeLines( false); }
   JContainer &setTreeIndent( ulong pels);
   JContainer &setTreeLinesWidth( ulong pels);

   // context menus used - note the container doesn't own these
   JContainer &setRecordMenu( JMenu *menu);
   JContainer &setContainerMenu( JMenu *menu);

   // help id for the container. This page gets shown on drophelps or record
   // helps if either of those is undefined.
   JContainer &setHelpID( const JHelpID &i);
   JHelpID     helpID() const { return hlpID; }

   // records ----------------------------------------------------------------

   // initialise a record with this container's context
   JCnrRecord *initialise( JCnrRecord *rec) const;

   // counting records
   ulong records() const;

   // inserting a record
   JContainer &insert( JCnrRecord *rec, JCnr::rec_insert = JCnr::last);
   JContainer &insert( JCnrRecord *rec, JCnrRecord *afterMe);
   JContainer &insertAsChild( JCnrRecord *rec, JCnrRecord *parent);
   // inserting a list of records
   JContainer &insert( JCnrRecordList &list, JCnr::rec_insert = JCnr::last);
   JContainer &insert( JCnrRecordList &list, JCnrRecord *afterMe);
   JContainer &insertAsChildren( JCnrRecordList &list, JCnrRecord *parent);

   // removing records
   JContainer &remove( JCnrRecordList &rec, BOOL freeRecs = false);
   JContainer &empty( BOOL free = false);

   // get next record after this one (default null) with emphasis given
   // initialise()s the record
   JCnrRecord *nextRecord( JCnr::emphasis e, JCnrRecord *after = 0) const;

   // methods to produce lists of / single records
   // get all selected records
   JCnrRecordList &selectedRecords( JCnrRecordList &l) const;
   // get all records with source emphasis
   JCnrRecordList &sourcedRecords( JCnrRecordList &l) const;
   // get all records with picked emphasis
   JCnrRecordList &pickedRecords( JCnrRecordList &l) const;
   // get all open records
   JCnrRecordList &openRecords( JCnrRecordList &l) const;
   // get all filtered records
   JCnrRecordList &filteredRecords( JCnrRecordList &l) const;

   // get records which the last menu-operation acted on.
   // This is needed 'cos the cnr de-emphasises the records before you get a
   // commandevent.
   JCnrRecordList &subjectRecords() { return sourced; }

   // get all records
   JCnrRecordList &allRecords( JCnrRecordList &l);

   // call a given iterator for all records below 'top'.
   JContainer &runIterator( JCnrIterator &it, JCnrRecord *top = 0);

   // get first / last records in the container
   JCnrRecord *firstRecord() const;
   JCnrRecord *lastRecord() const;

   // prevent default dragndrop actions from happening
   JContainer &stopDrags();
   JContainer &stopDrops();
   JContainer &stopLazyDrags();

   // get all records with given string in
   JCnrRecordList &matchingRecords( JCnrRecordList &list, const char *string,
                                    BOOL start = false, BOOL ins = true);
   // get next record after this one with string in
   JCnrRecord *matchingRecord( const char *string, JCnrRecord *after = 0,
                               BOOL start = false, BOOL ins = true);

   // run a given filter
   JContainer &filter( JCnrFilter &filter);
   // unfilter
   JContainer &unfilter();

   // sort the container using a given method
   JContainer &sort( JCnrSorter &sorter);
   JContainer &sortByName();

   // arrange
   JContainer &arrange();

   // deltas
   JContainer &setDelta( long delta);
   JContainer &deltaOff() { return setDelta( 0); }
   ulong       delta() const;

   // event handling 
   BOOL event( const JCtlEvent &e);

   virtual BOOL lostFocus()                       { return false; }
   virtual BOOL gainedFocus()                     { return false; }
   virtual BOOL scrolled( long /*distance*/)      { return false; }
   virtual BOOL deltaReached( JCnr::deltaType)    { return false; }

   // called when the user presses f1 whilst a dragset is over the container.
   // as default, the container's helpID is returned
   virtual JHelpID dropHelp( JDragSet &) { return hlpID; }

   // return whether or not drop on cnr background is valid.
   // default rejects everything
   virtual JDropOk isDropOk( JDragSet &)
   { return JDropOk( JDrag::never, JDrag::link); }

   // handle a drop on the cnr background; the dragset will have been validated
   virtual void drop( JDragSet &)
   {}

   // this method gets called when a drag which originated from the container
   // terminates. The flag indicates whether the target was successful in
   // rendering the item.
   virtual void dragEnded( JCnrRecord *, JDrag::operation, BOOL /*success*/)
   {}

   // this method gets called when the delete key is pressed, or a record is
   // successfully dropped on the shredder.
   virtual JDrag::psr deleteRecord( JCnrRecord *)
   { return JDrag::doneIt; }
};

// The cnr iterator. This makes a pass thru' the tree in the cnr, in a
// tree-like fashion, from the bottom up. (Post-traversal, I think)
class JCnrIterator
{
 public:
   // return 'true' if you want to abort the operation
   virtual BOOL operator () ( JCnrRecord *rec) = 0;
};

// The cnr filter. This is used to decide which records are shown in the cnr
class JCnrFilter
{
 public:
   // return true to let the object be seen
   virtual BOOL operator () ( JCnrRecord *rec) = 0;
};

// The cnr sorter. Used to order records in the container
class JCnrSorter
{
 public:
   // return >0 : arg1 > arg2
   //         0 : arg1 == arg2
   //        <0 : arg1 < arg2
   virtual int operator () ( JCnrRecord *arg1, JCnrRecord *arg2) = 0;
};

#endif
