/*
 * JCnrRec.hpp
 *
 * Container record object
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jcnrrec_h
#define _jcnrrec_h

#include "JLib.h"
#include "JCoord.hpp"
#include "JCnr2.hpp"
#include "JIcon.hpp"
#include "JHelp.hpp"
#include "JSeq.hpp"

#include "JDrag.hpp"

class JMiniRCore;
class JDragItem;
class JDragSet;

#include <string.h>

// Normally one subclasses from here to provide a useful record class for
// your application.  Even though you then have access to the name & icon
// fields directly (inside core), you should still use the setText & setIcon
// methods to update them.

// How should you find offsets of fields you add for the details view?
// erm, well, 12 is good for the first thing in a class derived from
// JCnrRecord.  This is very ugly, I know.

// Note the bizarro syntax for creating records - you must use new, with
// an additional parameter for the container, eg:
//
// JContainer cnr;
// JCnrRecord *rec = new( cnr) JCnrRecord( ...)

class JCnrRecord
{
   friend class JContainer;

   // checks the object has a container-context; throws up if not...
   BOOL check() const;

 protected:
   // jlib owner cnr, record data
   JContainer *cnr;
   JMiniRCore *core;

   virtual ~JCnrRecord();

 public:
   // the text given is deep-copied and free'd... the icon isn't.
   JCnrRecord( const char *text, ulong iconHandle);

   // you *must* get objects of this type with new( cnrspec)
   void *operator new( size_t cbSize, const JContainer &c);
   void *operator new( size_t cbSize, JContainer *c);
   void operator delete( void *addr);

   // note that this is /not/ used for sorting by containers.
   int operator == ( const JCnrRecord &arg2) const;

   // remove the record from the current cnr. Free's if possible if so told.
   // when the record is free'd this object is deleted. Note that records
   // can be inserted into >1 cnrs. This is why, if told to free, the method
   // will return quietly if the obj is still inserted elsewhere.
   void remove( BOOL freeit = false);
   void free();

   // access to data members
   JStr        text() const;
   JCnrRecord &setText( const char *txt); // needs cnr context
   JIcon       icon() const { return JIcon( core->hIcon); }
   JCnrRecord &setIcon( ulong hIcon);   // needs cnr context
   JPoint      pos() const  { return core->position; }

   // record attribute setting & querying - require a cnr context
   BOOL        hasEmphasis( JCnr::emphasis e) const;
   JCnrRecord &setEmphasis( JCnr::emphasis e, BOOL set = true);
   JCnrRecord &removeEmphasis( JCnr::emphasis e)
   { return setEmphasis( e, false); }

   // text editable-ness
   JCnrRecord &setReadonly( BOOL f = true);
   BOOL isReadonly();

   // rects for the record - need a context
   JRect textRect() const;
   JRect iconRect() const;
   JRect recordRect() const;

   // manipulation of parents in tree view - require a cnr context
   JCnrRecord &collapse();
   JCnrRecord &expand();
   // re-parenting the record. 'rec'=null moves the rec to the root
   JCnrRecord &setParent( JCnrRecord *rec, BOOL moveSiblings = false);

   // open the edit mle for the object - requires a cnr context
   JCnrRecord &editText();

   // related records; need a context - things they return have context.
   JCnrRecord *next();
   JCnrRecord *previous();
   JCnrRecord *parent();
   JCnrRecord *firstChild();
   JCnrRecord *lastChild();

   // notifies - methods called when things happen to the cnr record
   // when a parent tree node gets manipulated
   virtual BOOL collapsed() { return false; }
   virtual BOOL expanded()  { return false; }

   // when the user changes the name of the record
   virtual BOOL renamed()   { return false; }

   // when the record is double-clicked. Return true if you don't open the rec
   // remember to keep a copy of 'this' somewhere to un-open it later...
   virtual BOOL opened()    { return false; }

   // called just before a context menu for the object gets opened; you can
   // call the cnr to change the menu if necessary. Return true to abort.
   virtual BOOL menu( JSequence<JCnrRecord *> &) { return false; }

   // dragndrop support - return a dragitem created with new.
   // the default drags a thing with an undefined rmf
   // Note that the cnr puts a ptr to the record in the ulItemID slot, after
   // this method gets called.
   virtual JDragItem *initDrag();

   // return a JDropOk object based on the dragset provided
   // the default returns never.
   virtual JDropOk isDropOk( JDragSet &)
   { return JDropOk( JDrag::no_location, JDrag::def); }

   // handle a set of (validated) things dropped on the object
   virtual void drop( JDragSet &) {}

   // called when a request to print the object comes in. The default aborts
   // the operation.
   virtual JDrag::psr print( void *printdest) { return JDrag::abort; }

   // called when a request for help for the object happens. The default
   // returns the container's helpid.
   virtual JHelpID helpID();

   // used for detail fields
   static long iconOffset;
   static long textOffset;
};

// thrown when an attempt is made to access a method of the cnr object which
// requires a container context when it has none.
struct JNoCnrContext : public JException
{
   JNoCnrContext();
};

typedef JSequence<JCnrRecord *> JCnrRecordList;

#endif
