/*
 * JColln.hpp
 *
 * Common header file for collections
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jcollection_h
#define _jcollection_h

#include "JLib.h"
#include "JExcept.hpp"

// Exception classes used by collections ------------------------------------
class JDuplicateElement : public JException
{ public: JDuplicateElement(); };
class JInvalidCursor : public JException
{ public: JInvalidCursor(); };
class JElementNotFound : public JException
{ public: JElementNotFound(); };
class JKeyNotFound : public JException
{ public: JKeyNotFound(); };
class JDuplicateKey : public JException
{ public: JDuplicateKey(); };

// Basic collection class from which others derive --------------------------
class JVIterator;
class JVCursor;
class TreeNode;

class JCollection
{
 protected:
   friend class JVCursor;
   friend class TreeNode;

   class node;
   node *root;

   void doRemove( TreeNode *node);

   ulong  cElements;

   void  add( void *element);
   void  remove( void *element);
   void  removeAt( const JVCursor &c);
   void  doForAll( const JVIterator &iterator);
   node *find( void *element) const;

   JCollection() : root( 0), cElements( 0) {}

   // compare: return 0 if equal, +1 if e1 < e2, -1 if e1 > e2
   // may throw an exception if `dupl' is set
   virtual int compare( void *e1, void *e2, BOOL dupl) = 0;

   virtual void deleteElement( void *element) = 0;

 public:
   // pay attention: subclasses must call 'empty' in their dtors.
   // we can't do it here 'cos vfn calls happen.
   virtual ~JCollection() {}

   ulong elements() const { return cElements; }

   void empty();

   struct node { void *element; };
};

// Base class for keyed collections -----------------------------------------
class JKeyCollection : public JCollection
{
   friend class TreeNode;

 protected:
   virtual int compareK( void *e, void *k) = 0;

   void  removeK( void *key);
   void  removeAllK( void *key);
   node *findK( void *key) const;
   void *withK( void *key) const;

   JKeyCollection() : JCollection() {}
};

// Iterator class, two-stage to allow base concreteness ---------------------
class JVIterator
{
   friend class TreeNode;

 protected:
   virtual void iaction( void *element) = 0;
};

template < class T>
class JIterator : public JVIterator
{
   void iaction( void *e) { action( *(T*)e); }
 public:
   virtual void action( T &t) = 0;
};

template<class T>
class JConstIterator : public JVIterator
{
   void iaction( void *e) { action( *((const T*)e)); }
 public:
   virtual void action( const T &t) const = 0;
};

// Cursor class, two stage to allow base to be concrete ---------------------
class JVCursor
{
   friend class JCollection;

   JCollection &cn;

 protected:
   JCollection::node *cnode;

 public:
   JVCursor( JCollection &c) : cn( c), cnode( 0) {}

   virtual void toFirst();
   virtual void toNext();
   virtual void toLast();
   virtual void toPrevious();
   virtual BOOL isValid() const { return !!cnode; }
};

#define for_cursor( c) for( c.toFirst(); c.isValid(); c.toNext())
#define for_cursor_r( c) for( c.toLast(); c.isValid(); c.toPrevious())

template < class T>
class JCursor : public JVCursor
{
 public:
   JCursor( JCollection &c) : JVCursor( c) {}

   T *current()
   { 
      T *t = 0;
      if( !isValid()) jlib_throw( new JInvalidCursor);
      else t = (T*)(cnode->element);
      return t; 
   }

   const T *current() const
   {
      const T *t = 0;
      if( !isValid()) jlib_throw( new JInvalidCursor);
      else t = (T*)(cnode->element);
      return t;
   }
};

// Element comparison functions ---------------------------------------------
// Use these to allow collections of pointers, so pointed-to objects get used
template < class T> const T &realElement( const T &t) { return t; }
template < class T> int realCompare( const T &t1, const T &t2)
{
   if( t1 == t2) return 0;
   if( t1 <  t2) return 1;
   return -1;
}

#endif
