/*
 * JCoord.hpp
 *
 * Basic point, rectangle etc. classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jcoord_h
#define _jcoord_h

#include "JLib.h"
#include <JExcept.hpp>
#include <math.h>
#undef min
#undef max

struct _POINTL;

/* NB it's important that there's no virtual-ness involved with these
   classes - they need to be binary compatible with their PM equivalents.
*/

#define DIVZ jlib_throw( new JException( "divz", JExn::base))

// vector-ish class ----------------------------------------------------------
class JOrderedPair
{
 public:
   long x;
   long y;

 public:
   JOrderedPair( long i = 0, long j = 0) : x( i), y( j) {}
   JOrderedPair( const JOrderedPair &p) : x( p.x), y( p.y) {}
   JOrderedPair &operator = ( const JOrderedPair &copy)
   { x = copy.x; y = copy.y; return self; }

   // Comparison. These operators refer to the vector's position, ie.
   // (4,5) == (5,4) is false
   // fwiw, this is product order.
   int operator == ( const JOrderedPair &op2) const
   { return x == op2.x && y == op2.y; }
   int operator != ( const JOrderedPair &op2) const
   { return !( x == op2.x && y == op2.y); }
   int operator >  ( const JOrderedPair &op2) const
   { return x > op2.x && y > op2.y; }
   int operator >= ( const JOrderedPair &op2) const
   { return x >= op2.x && y >= op2.y; }
   int operator <  ( const JOrderedPair &op2) const
   { return x < op2.x && y < op2.y; }
   int operator <= ( const JOrderedPair &op2) const
   { return x <= op2.x && y <= op2.y; }

   // vector things
   double magnitude() const
   { return sqrt( x * x + y * y); }
   double gradient() const
   { if( x == 0) jlib_throw( new JException( "divz", JExn::base)); return y / x; }
   long   dot( const JOrderedPair &op2) const // inner product
   { return x * op2.x + y * op2.y; }
   double angleBetween( const JOrderedPair &op2) const; // radians
   double projectionOnto( const JOrderedPair &op2) const;
   double distanceFrom( const JOrderedPair &op2) const
   { return sqrt( pow( x - op2.x, 2) + pow( y - op2.y, 2)); }
   // returns true if -ve or 0, false if +ve
   BOOL   crossWith( const JOrderedPair &op2) const
   { return ( (x * op2.y - y * op2.x) <= 0); }

   // math operations
   JOrderedPair & operator += ( const JOrderedPair &op2)
   { x += op2.x; y += op2.y; return self; }
   JOrderedPair & operator -= ( const JOrderedPair &op2)
   { x -= op2.x; y -= op2.y; return self; }
   JOrderedPair & operator *= ( long l)
   { x *= l; y *= l; return self; }
   JOrderedPair & operator /= ( long l)
   { if( l == 0) DIVZ; else x /= l; y /= l; return self; }
   JOrderedPair operator + ( const JOrderedPair &op2) const
   { return JOrderedPair( x + op2.x, y + op2.y); }
   JOrderedPair operator - ( const JOrderedPair &op2) const
   { return JOrderedPair( x - op2.x, y - op2.y); }
   JOrderedPair operator * ( long l) const
   { return JOrderedPair( x * l, y * l); }
   JOrderedPair operator / ( long l) const
   { if( !l) { DIVZ; return JOrderedPair(); } return JOrderedPair( x/l, y/l); }

   // PM compatibility
   operator _POINTL * () const;
   operator _POINTL & () const;
};

// the reason why we define things like this, as opposed to just typedef-ing
// JPoint & JSize to JOrderedPair is so you can say `class JPoint;' and not
// get abuse from the compiler.
class JPoint : public JOrderedPair
{
 public:
   JPoint( long i = 0, long j = 0) : JOrderedPair( i, j) {}
   JPoint( const JOrderedPair &copy) : JOrderedPair( copy) {}
};

class JSize : public JOrderedPair
{
 public:
   JSize( long i = 0, long j = 0) : JOrderedPair( i, j) {}
   JSize( const JOrderedPair &copy) : JOrderedPair( copy) {}
};

extern const JSize _autosize;

struct _RECTL;

// Rectangle class. A rectangle is defined by two points ---------------------
class JRect
{
 public:
   JPoint min;
   JPoint max;

 public:
   JRect( long x1 = 0, long y1 = 0, long x2 = 0, long y2 = 0)
      : min( x1, y1), max( x2, y2) {}
   JRect( const JPoint &p1, const JPoint &p2)
      : min( p1), max( p2) {}
   JRect( const JPoint &p1, const JSize &p2)
      : min( p1), max( p1 + p2) {}
   JRect( const JPoint &p1, long x2 = 0, long y2 = 0)
      : min( p1), max( x2, y2) {}
   JRect( long x1, long y1, const JPoint &p2)
      : min( x1, y1), max( p2) {}
   JRect( const JRect &r)
      : min( r.min), max( r.max) {}
   JRect( const _RECTL &);

   JRect &operator = ( const JRect &copy)
   { min = copy.min; max = copy.max; return self; }


   // access to members
   const JPoint bottomLeft() const  { return min; }
   const JPoint topLeft() const     { return JPoint( min.x, max.y); }
   const JPoint topRight() const    { return max; }
   const JPoint bottomRight() const { return JPoint( max.x, min.y); }
   JPoint &bottomLeft()             { return min; }
   JPoint &topRight()               { return max; }

   JSize size() const { return max - min; }

   // intersect if 1) contains at least one point of r
   //              2) has at least one point contained by r
   BOOL intersects( JRect &r) const;
   // equality == containment
   BOOL contains( const JRect &r) const;
   // on the edge == in
   BOOL contains( const JPoint &p) const;

   // comparison - position in space
   int operator == ( const JRect &r) const
   { return min == r.min && max == r.max; }
   int operator != ( const JRect &r) const
   { return min != r.min || max != r.max; }

   // comparison - size
   int operator >  ( const JSize &s) const
   { return size() > s; }
   int operator >= ( const JSize &s) const
   { return size() >= s; }
   int operator <  ( const JSize &s) const
   { return size() < s; }
   int operator <= ( const JSize &s) const
   { return size() <= s; }

   // translations
   JRect operator - ( const JPoint &p) const
   { return JRect( min - p, JPoint( max - p)); }
   JRect operator + ( const JPoint &p) const
   { return JRect( min + p, JPoint( max + p)); }
   JRect &operator += ( const JPoint &p)
   { min += p; max += p; return self; }
   JRect &operator -= ( const JPoint &p)
   { min -= p; max -= p; return self; }

   // PM compatibility
   operator _RECTL * () const;
   operator _RECTL & () const;
};

// Range class ---------------------------------------------------------------
struct JRange
{
   long low;
   long high;
   JRange( long l = 0, long h = 0) : low( l), high( h) {}
   BOOL contains( long s) const
   { return ((s >= low) && (s <= high)); }
};

#endif
