/*
 * JExcept.hpp
 *
 * Exception classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jexcept_h
#define _jexcept_h

#include "JLib.h"
#include "JStr.hpp"

// gcc doesn't support exceptions properly.  JLib was written for VACPP,
// which does.  What we do now is to generate exceptions and store them
// in the thread's PM.  There's a compile-time symbol 'REALLY_THROW' which
// makes routines throw exceptions (in addition to storing them).
//
// This means you should test JPM::current().exn for errors.
// I'm not happy about this, but it seems the best solution.
//
// Note that even if exceptions aren't thrown, their messages are still
// generated & sent to JDebug if appropriate.

// Exceptions are thrown a lot in JLib.
// class JException provides the facility of sending its string to the trace
// (see JTrace.hpp) to be displayed in JDebug, if the variable
// JException::debugEnabled is set to TRUE.

// this is an enumeration of exception types.  When an exception is generated,
// its type is stored along with it in the JPM, so you can cast as appropriate
// (if necessary -- only need to if extra info is supplied in the subclass)

struct JExn
{
   enum type {
      none,
      // principle exception types
      base,
      notFound,
      bounds,
      dos,
      pm,
      resource,
      params,
      // IO (streams)
      incompleteWrite,
      brokenPipe,
      eof,
      eaMismatch,
      sharing,
      // drag and drop
      nodrag,
      doubledrag,
      // general PM
      fontNotFound,
      cnrContext,
      // collections
      duplicate,
      badCursor,
      badKey,
      duplicateKey
   };
};

typedef JExn::type JExnType; // (cringe)

class JException
{
   JStr     text;
   JExnType type;

 protected:
   virtual void setText( const char *s);
   JException( JExnType t) : type( t) {}

 public:
   JException( char *tx, JExnType t) : type( t) { setText( tx); }
   JException( ulong id, JExnType t, char *p1 = 0, char *p2 = 0);

   JStr &report() { return text; }
   JExnType getType() const { return type; }

   // set this flag to pipe messages to JDebug
   static BOOL debugEnabled;
};

class JNotFoundError : public JException
{
 public:
   JNotFoundError( char *by) : JException( by, JExn::notFound) {}
   JNotFoundError( ulong id, char *p = 0) : JException( id, JExn::notFound, p) {}
};

class JBoundsError : public JException
{
 public:
   JBoundsError( char *by) : JException( by, JExn::bounds) {}
   JBoundsError( ulong id, char *p = 0) : JException( id, JExn::bounds, p) {}
};

// ct with return code from a Dos* api; appends 'returned x' to msg/text given
class JDosExcep : public JException
{
 protected:
   virtual void setText( const char *s);
   ulong rc;

 public:
   JDosExcep( ulong Rc, char *m) : JException( JExn::dos), rc( Rc) { setText( m); }
   JDosExcep( ulong rc, ulong id, char *p = 0);
   ulong error() { return rc; }
};

class JPMExcep : public JException
{
 protected:
   ulong code;
   virtual void setText( const char *s);

 public:
   JPMExcep( char *m) : JException( JExn::pm) { setText( m); }
   JPMExcep( ulong id, char *p = 0);
   short sev();
   short error();
};

// use this instead of 'throw' keyword
void jlib_setup_throw( JException *new_exn);
#ifdef REALLY_THROW_EXCEPTIONS
template <class T>
void jlib_throw( T *e) { jlib_setup_throw( e); throw e; }
#else
#define jlib_throw(e) jlib_setup_throw( e)
#endif
void jlib_catch(); // clear up current thread exception status

#endif
