/*
 * JGPrim.hpp
 *
 * Physical graphics primitives
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jgprim_h
#define _jgprim_h

#include "JLib.h"
#include "JStr.hpp"
#include "JCoord.hpp"
#include "JBuffer.hpp"
#include "JSeq.hpp"
#include "JPtr.hpp"

class JBPSpace;

// Base abstract class -------------------------------------------------------
class JGObject
{
 public:
   JGObject() {}
   virtual ~JGObject() {}
   virtual BOOL renderIn( JBPSpace &p) const = 0;
// virtual JBuffer orders() = 0;
};

#include "JGLPrim.hpp"

#ifndef NO_PRIMS

// Base objects --------------------------------------------------------------
// a line
class JLine : public JGObject
{
   JPoint pt;

 public:
   JLine( const JPoint &drawTo) : pt( drawTo) {}
   JLine( long x, long y) : pt( x, y) {}

   const JPoint &drawTo() const { return pt; }
   JPoint &drawTo()             { return pt; }

   BOOL renderIn( JBPSpace &p) const;
};

// change the graphics cursor position
class JMove : public JGObject
{
   JPoint pt;

 public:
   JMove( const JPoint &moveTo) : pt( moveTo) {}
   JMove( long x, long y) : pt( x, y) {}

   const JPoint &moveTo() const { return pt; }
   JPoint &moveTo()             { return pt; }

   BOOL renderIn( JBPSpace &p) const;
};

// a rectangle at the current position
class JBox : public JGObject
{
   JSize sz;
   long  opts;

 public:
   JBox( const JSize &size, BOOL outline = true, BOOL filled = false)
      : sz( size), opts( filled) { if( outline) opts |= 2; }
   JBox( long x, long y, BOOL outline = true, BOOL filled = false)
      : sz( x, y), opts( filled) { if( outline) opts |= 2; }

   const JSize &size() const { return sz; }
   JSize &size()             { return sz; }
   BOOL filled() const { return opts & 1; }
   BOOL outline() const { return opts & 2; }
   JBox &filled( BOOL f) { if( f) opts |= 1; else opts &= ~1; return *this; }
   JBox &outline( BOOL o) { if( o) opts |= 2; else opts &= ~2; return *this; }

   BOOL renderIn( JBPSpace &p) const;
};

typedef JSequence<JPoint> JPointList;
typedef JSequence< JPtr < JPointList > > JPolygonList;

// polygons are drawn filled, taking account of the current fill pattern
// the current position is taken as the first point of the first polygon
class JPolygons : public JGObject
{
   ulong *pgons;

 public:
   JPolygons( const JPolygonList &l, BOOL outline = true);
  ~JPolygons();

   const BOOL outline() const { return pgons[ 1]; }
   void outline( BOOL o) { pgons[ 1] = !!o; }

   BOOL renderIn( JBPSpace &p) const;
};

// Draw a sequence of lines, connected together
class JPolyLine : public JGObject
{
   ulong *points;

 public:
   JPolyLine( const JPointList &p);
  ~JPolyLine() { delete [] points; }

   BOOL renderIn( JBPSpace &p) const;
};

// Draw a sequence of lines: move, draw, move, draw...
class JPolyLines : public JGObject
{
   ulong *points;

 public:
   JPolyLines( const JPointList &p);
  ~JPolyLines() { delete [] points; }

   BOOL renderIn( JBPSpace &p) const;
};

// Markers - use JMarkerSettings to alter type/size/colours...
class JMarker : public JGObject
{
   JPoint pt;

 public:
   JMarker( const JPoint &p) : pt( p) {}
   JMarker( long x, long y) : pt( x, y) {}

   BOOL renderIn( JBPSpace &p) const;
};

class JMarkers : public JGObject
{
   ulong *points;

 public:
   JMarkers( const JPointList &p);
  ~JMarkers() { delete [] points; }

   BOOL renderIn( JBPSpace &p) const;
};

// Text in current font
class JText : public JGObject
{
   JStr  txt;
   ulong opts;

 public:
   JText( char *text, BOOL underline = false, BOOL strikeout = false);

   BOOL underline() const;
   BOOL strikeout() const;
   void underline( BOOL u);
   void strikeout( BOOL s);

   const JStr &text() const { return txt; }
   JStr &text() { return txt; }

   BOOL renderIn( JBPSpace &p) const;
};

// Retained label
class JLabel : public JGObject
{
   ulong l;

 public:
   JLabel( ulong labl) : l( labl) {}

   ulong label() const { return l; }
   void  label( ulong la) { l = la; }

   BOOL renderIn( JBPSpace &p) const;
};

class JComment : public JGObject
{
   JBuffer dat;

 public:
   JComment( const JBuffer &comment) : dat( comment) {}

   BOOL renderIn( JBPSpace &p) const;

   const JBuffer &data() const { return dat; }
   JBuffer &data() { return dat; }
};

// arcs are circle segments through three points - one is the current position
class JArc : public JGObject
{
   JPoint pts[ 2];

 public:
   JArc( const JPoint &mid, const JPoint &end)
      { pts[ 0] = mid; pts[ 1] = end; }

   BOOL renderIn( JBPSpace &p) const;

   const JPoint &middle() const { return pts[ 0]; }
   JPoint &middle() { return pts[ 0]; }
   const JPoint &end() const { return pts[ 1]; }
   JPoint &end() { return pts[ 1]; }
};

// JLineArc draws a line to the start of the arc, followed by the arc itself
// The point given is the centre of the fullarc which a section of is drawn (!)
// sweep & start are angles in degrees
class JLineArc : public JGObject
{
   JPoint pt;
   ulong rad, st, sw;

 public:
   JLineArc( const JPoint &p, ulong rd, ulong sweep, ulong start = 0)
      : pt( p), rad( rd), st( start), sw( sweep) {}
   BOOL renderIn( JBPSpace &p) const;

   const JPoint &center() const { return pt; }
   JPoint &center() { return pt; }
   ulong radius() const { return rad; }
   void radius( ulong r) { rad = r; }
   ulong sweep() const { return sw; }
   void sweep( ulong s) { sw = s; }
   ulong start() const { return st; }
   void start( ulong s) { st = s; }
};

// Not necessarily a circle if you alter the arc settings
class JCircle : public JGObject
{
   ulong rad, opts;

 public:
   JCircle( ulong radius, BOOL outline = true, BOOL filled = false)
      : rad( radius), opts( !!filled) { if( outline) opts |= 2; }

   BOOL renderIn( JBPSpace &p) const;

   ulong radius() const { return rad; }
   void radius( ulong r) { rad = r; }
   BOOL outline() const { return opts & 2; }
   void outline( BOOL o) { if( o) opts |= 2; else opts &= ~2; }
   BOOL filled() const { return opts & 1; }
   void filled( BOOL f) { if( f) opts |= 1; else opts &= ~1; }
};

// Leaves arcsettings unchanged
class JEllipse : public JGObject
{
   JArcSettings arcs;
   ulong        opts;

 public:
   JEllipse( ulong major, ulong minor, ulong angle = 0, BOOL outline = true,
             BOOL filled = false);
   JEllipse( const JEllipse &c) : arcs( c.arcSettings()), opts( c.filled()) {
      if( c.outline()) opts |= 2;
   }

   BOOL renderIn( JBPSpace &p) const;

   const JArcSettings &arcSettings() const { return arcs; }
   JArcSettings &arcSettings() { return arcs; }

   BOOL outline() const { return opts & 2; }
   void outline( BOOL o) { if( o) opts |= 2; else opts &= ~2; }
   BOOL filled() const { return opts & 1; }
   void filled( BOOL f) { if( f) opts |= 1; else opts &= ~1; }
};

// A fillet is a curve tangential to two lines. Give two points to draw a
// single fillet, any more to draw a 'polyfillet'
class JFillet : public JGObject
{
   ulong *points;

 public:
   JFillet( const JPointList &p);
  ~JFillet() { delete [] points; }

   BOOL renderIn( JBPSpace &p) const;
};

// Bezier curves; 4 control points per curve, but we use the current position
// to start with.  Only get C0 continuity.
class JSpline : public JGObject
{
   ulong *points;

 public:
   JSpline( JPointList &p);
  ~JSpline() { delete [] points; }

   BOOL renderIn( JBPSpace &p) const;
};

// blitting - from a ps with a bmp selected into it or a bmp in world coords
class JBmpImage : public JGObject
{
   ulong           hThing;
   JGfx::rop::type rop;
   BOOL            useWC;
   JPoint          pts[ 4];

 public:
   JBmpImage( const JBPSpace &ps, const JRect &source,
              const JRect &dest = JRect(),
              JGfx::rop::type t = JGfx::rop::normal);
   JBmpImage( const JBitmap &bmp, const JRect &source = JRect(),
              const JRect &dest = JRect(),
              JGfx::rop::type t = JGfx::rop::normal);

   BOOL renderIn( JBPSpace &ps) const;
};

// a chain of drawable objects
typedef JPtr<JGObject> JGPtr;

class JGList : public JGObject, public JSequence<JGPtr>
{
 public:
   JGList() {}

   BOOL renderIn( JBPSpace &p) const;
};

#endif

#endif
