/*
 * JHTable.hpp
 *
 * Straightforward hashtable collection header file
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jhtable_h
#define _jhtable_h

#include "JLib.h"
#include "JColln.hpp"                           // for exceptions, realElement

// How this works: T is the element type, K is the key type. class K must
// provide the equality operator.
// There must be a function with signature `ulong hash( const K &)' which
// provides the hashcode for the object.  Consult someone else for good
// ways of calculating hash values.  JCollection-like semantics for things
// in the table, ie. a copy is made on insertion and deleted on removal.
// Yes, there is a dichotomy here with the keyed collections.  This is so
// because of the sheer commonness of wanting to provide a map between an
// integer handle and a more complex object, where the handle is not naturally
// part of the object.  A default 'hash' for int-types is provided.
// You can use classes from JPtr.hpp if you want.

// Theoretical wibble: we use open hashing here, throwing linkedlists around;
// we double the size & rehash when the load factor gets to 80%

// This collection is not threadsafe.  For a threadsafe hashtable, see
// the file JSafeHT.hpp

template<class T, class K>
class JHashtable
{
 public:
   class link;

   // construct with initial no. of buckets
   JHashtable( ulong nBuckets = 23)
            : buckets( new link* [ nBuckets]),
              cBuckets( nBuckets),
              cElements( 0)
   {
      // get threshold; only do this the once.  WT-P says 80%, so there...
      threshold = (ulong) (.8 * (double)nBuckets);
      for( ulong i = 0; i < cBuckets; i++) buckets[ i] = 0;
   }

   // table is emptied at death
   virtual ~JHashtable()
   {
      empty();
      delete [] buckets;
   }


   // put something into the table, may throw JDuplicateKey
   JHashtable<T,K> &put( const K &k, const T &t)
   {
      if( linkForKey( k)) jlib_throw( new JDuplicateKey);
      else {
         link *old = buckets[ ndx];
         buckets[ ndx] = new link( new T( t), new K( k), old);
         cElements++;
         checkSize();
      }

      return *this;
   }

   // get a key's element from the table, may throw JKeyNotFound
   T *get( const K &k)
   {
      T *t = 0;
      link *l = linkForKey( k);
      if( l) t = l->element;
      else jlib_throw( new JKeyNotFound);
      return t;
   }

   const T *get( const K &k) const
   {
      T *t = 0;
      link *l = ((JHashtable<T,K>*)this)->linkForKey( k);
      if( l) t = l->element;
      else jlib_throw( new JKeyNotFound);
      return t;
   }

   // does the table contain a key ?
   BOOL contains( const K &k) const
   { return !!((JHashtable<T,K>*)this)->linkForKey( k); }

   // remove a key-element pair from the table, may throw JKeyNotFound
   void remove( const K &k)
   {
      link *l = linkForKey( k);
      if( !l) jlib_throw( new JKeyNotFound);
      else {
         link *c = buckets[ ndx];
         if( c == l) {
            buckets[ ndx] = l->next;
         } else {
            for(;;) {
               if( c->next == l) { c->next = l->next; break; }
               c = c->next;
            }
         }
         l->next = 0;
         delete l;
         cElements--;
      }
   }

   // remove everything from the table
   void empty()
   {
      for( ulong i = 0; i < cBuckets; i++) {
         delete buckets[ i]; buckets[ i] = 0;
      }
      cElements = 0;
   }

   // how many keys are there in the table ?
   ulong elements() const
   { return cElements; }

   // cursor class for elements/key enumeration
   class cursor
   {
      link            *c_l;
      ulong            c_b;
      JHashtable<T,K> &tbl;

      void nextBk()
      {
         for( ; c_b < tbl.cBuckets; c_b++) {
            c_l = tbl.buckets[ c_b];
            if( c_l) return;
         }
         c_l = 0;
      }

    public:
      // cursor is not set to first record at creation...
      cursor( const JHashtable<T,K> &ht)
      : c_l( 0), c_b( 0), tbl( (JHashtable<T,K> &)ht)
      {}

      cursor( JHashtable<T,K> *ht) : c_l( 0), c_b( 0), tbl( *ht)
      {}

      // set the cursor to the first element
      void toFirst()
      { c_b = 0; nextBk(); }

      // is the cursor valid ?
      BOOL isValid() const
      { return c_l != 0; }

      // set the cursor to the next element, may throw JInvalidCursor
      void toNext()
      {
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else {
            if( c_l->next) c_l = c_l->next;
            else { c_b++; nextBk(); }
         }
      }

      // get the current element, may throw JInvalidCursor
      T *current()
      {
         T *t = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else t = c_l->element;
         return t;
      }

      const T *current() const
      {
         T *t = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else t = c_l->element;
         return t;
      }

      // get the current key, may throw JInvalidCursor
      K *currentKey()
      {
         K *k = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else k = c_l->key;
         return k;
      }

      const K *currentKey() const
      {
         K *k = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else k = c_l->key;
         return k;
      }
   };
   friend class cursor;

   // iterators
   struct iterator
   {
      virtual void action( T &t) = 0;
   };

   struct const_iterator
   {
      virtual void action( const T &t) const = 0;
   };

   // run an iterator over the contents of the table
   void doForAll( iterator &iter)
   {
      for( ulong i = 0; i < cBuckets; i++)
         for( link *lc = buckets[ i]; lc; lc = lc->next)
            iter.action( *(lc->element));
   }

   void doForAll( const const_iterator &iter)
   {
      for( ulong i = 0; i < cBuckets; i++)
         for( link *lc = buckets[ i]; lc; lc = lc->next)
            iter.action( *(lc->element));
   }

 private:
   link **buckets;
   ulong    cBuckets;
   ulong    cElements;
   ulong    threshold;
   ulong  ndx;

   link *linkForKey( const K &k)
   {
      ulong hcode = hash( realElement( k));
      ndx = hcode % cBuckets;
      link *lc = buckets[ ndx];
      while( lc) {
         if( realElement( k) == *(lc->key)) return lc;
         lc = lc->next;
      }
      return 0;
   }

   void checkSize()
   {
      if( cElements > threshold) {
         // plan is to double the size; first save what we've got
         // (bit crap)
         T **ts = new T* [ cElements];
         K **ks = new K* [ cElements];
         ulong  i = 0;
         cursor c( *this);
         for_cursor( c) {
            ts[ i] = c.current();
            ks[ i] = c.currentKey();
            i++;
         }

         // reset storage
         delete [] buckets;
         cBuckets *= 2;
         threshold *= 2; // hmm, rounding error might accumulate badly...
         buckets = new link* [ cBuckets];
         for( i = 0; i < cBuckets; i++) buckets[ i] = 0;

         // populate new table
         ulong j = cElements; cElements = 0;
         for( i = 0; i < j; i++)
            put( *(ks[ i]), *(ts[ i]));
      }
   }

   struct link
   {
      link *next;
      T    *element;
      K    *key;
      link( T *e=0, K *k=0, link *l=0) : next(l), element(e), key(k) {}
     ~link() { delete next; delete element; delete key; }
   };
};

inline ulong hash( const ulong &u) { return u; }

#endif
