/*
 * JListBox.hpp
 *
 * Base Listbox control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jlbox_h
#define _jlbox_h

#include "JLib.h"
#include "JControl.hpp"
#include "JCoord.hpp"
#include "JStr.hpp"
#include "JSeq.hpp"

class JWindow;
class JCtlEvent;

class JListBox : public JControl
{
   BOOL check( ushort i) const;

 protected:
   JListBox() : JControl() {}

 public:
   // style
   static const unsigned long
      normal,
      horzScroll,
      multiSelection,
      noAdjustPos;

   // standard constructors
   JListBox( ulong hwnd) : JControl( hwnd) {}
   JListBox( JWindow *parent, ulong id) : JControl( parent, id) {}
   JListBox( JWindow *parent, const JPoint &pos, const JSize &size,
             ulong id, ulong style = JListBox::normal);

   // listbox item class
   class item
   {
      JListBox *lb;
      ushort    ndex;
      JStr     *txt;
      ulong     hndle;

    public:
      // wrap existing item
      item( JListBox *lb, ushort index);
      // create an item for insertion
      item( const char *text = "", ulong handle = 0);
      item( const item &copy);
     ~item();

      // access to item's string
      JStr   text() const;
      item  &setText( const char *s);

      // access to item's handle
      ulong  handle() const;
      item  &setHandle( ulong ul);

      // access to control (may be 0)
      JListBox *getListBox() const { return lb; }

      // remove item from listbox
      item  &remove();

      // make the item topmost in the listbox
      item  &scrollTo();

      // get the item's zero-based index
      ushort index() const { return ndex; }

      // select the item
      item  &select( BOOL select = true);
   };

   // remove every item from the listbox
   JListBox &empty();

   // count of entries in the listbox
   ushort    items() const;

   // access to top entry, scrolling
   ushort    topEntry() const;
   JListBox &scrollTo( ushort index);

   // remove an item from the listbox
   JListBox &deleteItem( ushort index);

   // get / set current selection
   ushort    choice() const;
   JListBox &select( ushort index, BOOL select = true);
   JListBox &deselect( ushort index) { return select( index, false); }

   // add item[s] to the listbox
   static const unsigned short
      addAtEnd,
      addSortedUp,
      addSortedDown;

   JListBox &insert( const item &itm, short type = addAtEnd, BOOL select = false);
   JListBox &insert( JSequence<JListBox::item> &list, short type = addAtEnd);

   // find an item with text like this in the listbox
   ushort    find( const char *text, BOOL mustStartWith = false) const;

   enum cursorType {
      selection,
      all
   };

   // a cursor for looking thru' a lbox or just those selected elements
   class cursor
   {
      JListBox::cursorType type;
      JListBox::item      *curr;
      JListBox            *lb;
      ushort               last;
      void                 check();

    public:
      cursor( JListBox *lb, JListBox::cursorType = JListBox::selection);

      cursor &toFirst();
      cursor &toNext();
      BOOL    isValid() const;

      JListBox::item *current() const;
      JListBox::item *operator -> ();
   };

   // event handling
   virtual BOOL event( const JCtlEvent &e);

   virtual BOOL lostFocus()                          { return false; }
   virtual BOOL gainedFocus()                        { return false; }
   virtual BOOL scrolled()      /* horizontally */   { return false; }
   // just a selection
   virtual BOOL selected( JListBox::cursor &)        { return false; }
   // dbl-click or enter
   virtual BOOL choice( JListBox::cursor &)          { return false; }

};

typedef JSequence<JListBox::item> JLBoxItemList;

#endif
