/*
 * JMTable.hpp
 *
 * A memotable collection
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

// This is a memo table.  A cache.  Works like a hash table using closed
// hashing (no linked lists), but eventually handles collisions by replacing
// something else.  This means it's not a place for holding things, but
// storing results of a more complex function; dynamic programming.
//
// Two parameters define the table's behaviour: bucket number and the probe
// distance.  The memo table does not resize, it just drops more information.
// The probe distance determines how far the table looks for a space when
// there's a collision before discarding something (we use linear probing).
// This number should be small, ideally at least an order of magnitude less
// than the bucket number.
//
// T is the element type (output), K is the key (input). Need hash function
// as in JHashTbl.hpp and realElement function from JColln.hpp.
// Elements aren't copied into the table; keys must support null constructors
// and an assignment operator.
//
// As an experiment, there's a symbol JLIB_NO_EXN which, if defined, will
// alter the behaviour of the methods to not throw exceptions, but show
// errors using other means.  This may provide efficiency gains.

#ifndef _jmtable_h
#define _jmtable_h

#include "JLib.h"
#include "JColln.hpp"

template<class T, class K>
class JMemoTable
{
   struct Bucket
   {
      T *element;
      K  key;     // this key is only valid if the element pointer is !null
   };

   Bucket *buckets;
   int     cBuckets;
   int     dProbe;

 public:
   JMemoTable( int size = 43, int distProbe = 4)
           : cBuckets( size), dProbe( distProbe), buckets( new Bucket [ size])
   { empty(); }

  ~JMemoTable() { delete[] buckets; }

   // empty the table
   void empty() {
      for( int i = 0; i < cBuckets; i++) {
         buckets[ i].element = 0;
      }
   }

   #define MD(i) ((i) % cBuckets)

   // put something into the table
#ifndef JLIB_NO_EXN
   // may throw JDuplicateKey
   void put( const K &k, const T *element)
#else
   // returns true on success, false if the key's already present
   BOOL put( const K &k, T *element)
#endif
   {
      int ndx = MD( hash( realElement( k)));
      int i;

      for( i = 0; i < dProbe; i++) {
         Bucket &bkt = buckets[ MD( ndx + i)];
         if( !bkt.element) break; // found a spot
         if( bkt.key == k)
#ifndef JLIB_NO_EXN
            throw JDuplicateKey();
#else
            return false;
#endif
      }

      if( i == dProbe) i = 0; // replace

      Bucket &bkt = buckets[ MD( i + ndx)];
      bkt.key = k;
      bkt.element = element;
#ifdef JLIB_NO_EXN
      return true;
#endif
   }

   // get something out of the table, may throw JKeyNotFound, or, if
   // JLIB_NO_EXN is set, returns null on not-founds.
   T *get( const K &key)
   {
      int ndx = MD( hash( realElement( key)));
      int i;

      for( i = 0; i < dProbe; i++) {
         Bucket &bkt = buckets[ MD( ndx + i)];
         if( !bkt.element) break;
         if( bkt.key == key) return bkt.element;
      }

#ifndef JLIB_NO_EXN
      throw JKeyNotFound();
#else
      return 0;
#endif
   }

   // not sure about the rest of this, may be overkill; the point is to
   // define a slightly nicer interface that encapsulates how this kind
   // of thing is usually used.
   struct function
   {
      virtual T *operator () ( const K &k) = 0;
   };

   T *get( const K &key, function &fn)
   {
#ifndef JLIB_NO_EXN
      try {
         return get( key);
      } catch( JKeyNotFound &) {
         T *t = fn( key);
         put( key, t);
         return t;
      }
#else
      T *t = get( key);
      if( !t) {
         t = fn( key);
         put( key, t);
      }
      return t;
#endif
   }
};

#endif
