/*
 * JMLE.hpp
 *
 * Multi-line entry field control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jmle_h
#define _jmle_h

#include "JLib.h"
#include "JControl.hpp"
#include "JCoord.hpp"
#include "JStr.hpp"

class JWindow;
class JCtlEvent;

class JMLE : public JControl
{
 public:

   // style
   static const unsigned long
      border,
      readonly,
      wordwrap,
      horzScroll,
      vertScroll,
      scrollBars,
      ignoreTab,
      noUndo,
      normal;

   enum margin {
      left = 1,
      bottom,
      right,
      top
   };

   JMLE( ulong hwnd);
   JMLE( JWindow *w, ulong id);
   JMLE( JWindow *parent, const JPoint &pos, const JSize &size,
         ulong id, ulong style = JMLE::normal);

   // clipboard support and selection
   JMLE &clear();
   JMLE &cut();
   JMLE &copy();
   JMLE &paste();
   JMLE &setSelection( long start, long count);
   JStr  getSelection() const;

   // cursor control
   JMLE &moveCursorTo( long offset);
   long  cursorPos() const;

   // line interface
   long  lines() const;
   JMLE &scrollToLine( long line);
   long  lineLength( long line) const;
   JStr  line( long line) const;
   JMLE &addLine( const char *text);

   // various text setting
   long textLength() const;
   virtual JStr getText() const;
   virtual JMLE &setText( const char *text);
   JMLE &insert( const char *text);
   JMLE &empty();

   // import/export interface; mle needs various cr/lf to look nice
   JStr  export() const;
   JMLE &import( const char *s);

   // screen update
   JMLE &enableRefresh( BOOL on = true);
   JMLE &disableRefresh() { return enableRefresh( false); }

   // various flags - readonly...
   JMLE &setReadonly( BOOL set);
   BOOL  isReadonly() const;
   // ...tabstop...
   JMLE &setTabStop( long pixels);
   long  getTabStop() const;
   // ...wrap...
   JMLE &setWrap( BOOL on = true);
   BOOL  hasWrap() const;
   // ...changed indicator...
   BOOL  hasChanged() const;
   JMLE &setChanged( BOOL changed = true);
   // ...maximum field length
   JMLE &setMaxLength( long count);
   long  getMaxLength() const;

   // undo management
   BOOL  canUndo() const;
   JMLE &resetUndo();
   JMLE &undo();

   // search strings
   BOOL find( const char *string, BOOL caseMatch = false, long start = 0) const;
   BOOL replace( const char *find, const char *replace,
                 BOOL caseMatch = false, long start = 0) const;

   // event handling
   BOOL event( const JCtlEvent &e);
   virtual BOOL changed()                       { return false; }
   virtual BOOL lostFocus()                     { return false; }
   virtual BOOL gainedFocus()                   { return false; }
   virtual BOOL scrolled( ulong /*direction*/)  { return false; }
   virtual BOOL continueSearch( long /*last*/)  { return false; }
   virtual BOOL borderCrossed( JMLE::margin, ulong /*msg*/, long /*offset*/)
                { return false; }
};

#endif
