/*
 * JNoteBk.hpp
 *
 * Notebook control and page object
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jnotebk_h
#define _jnotebk_h

#include "JLib.h"
#include "JControl.hpp"
#include "JCoord.hpp"
#include "JResID.hpp"
#include "JModule.hpp"
#include "JHelp.hpp"
#include "JStr.hpp"
#include "JBitmap.hpp"

class JWindow;
class JCtlEvent;

class JPage;

// Notebooks are a lot less hassle in Warp4, and a lot of the gubbins here
// to do with tab-sizing isn't needed.
//
// For maximum compatability with Warp3, your notebooks as defined in .dlg
// files should be similar in layout to the Warp4 ones.
//
// Example of how to set up a notebook control with a page:
//
// class SettingsDlg : public JDialog
// {
//    JNotebook  book;
//    JDialog   *page1;
//
//    SettingsDlg(...) : JDialog( ....), book( this, id_book)
//    {
//       page1 = new JDialog( &book, id_dlg_page1, this);
//       book.addSection( page1, "Tab text", "Page 1 of 1");
//       ...
//    }
// ...

class JNotebook : public JControl
{
   JHelpID hID;
   void    setup();
   JSize   tabSz;
   void    adjustTabs( const char *s);
   friend class JPage;

 public:
   // ctors/dtor
   JNotebook( ulong hwnd);
   JNotebook( JWindow *parent, ulong id);
   JNotebook( JWindow *parent, const JPoint &pos, const JSize &size, ulong id,
              ulong style = 0);
   virtual ~JNotebook();

   // manipulating pages & sections; a section is a page with a tab.
   // first 2 methods append page to notebook
   JPage addSection( JWindow *w, const char *tabtext = "",
                     const char *statusField = "") const;
   JPage addPage( JWindow *w, const char *statusField = "") const;
   JPage addSectionAfter( JWindow *w, const JPage &pg,
                          const char *tabtext = "",
                          const char *field = "") const;
   JPage addPageAfter( JWindow *w, const JPage &pg,
                       const char *field = "") const;

   JNotebook &removePage( const JPage &p);

   // query number of pages
   int pages() const;

   // fiddling with tabs
   JNotebook &refreshTabs();
   JNotebook &setTabSize( const JSize &s);

   // setting/querying visible page
   JNotebook &turnToPage( const JPage &page);
   JNotebook &turnToTop();
   JPage      topPage() const;

   // query rect of page window
   JRect pageRect() const;

   // Help id; used for dynamic help model. Default is window id.
   JNotebook &setHelpID( const JHelpID &i);
   JHelpID    helpID() const;

   // page cursor thing
   class cursor
   {
      JPage     *pg;
      JNotebook *book;

    public:
      cursor( JNotebook *bk);
     ~cursor();

      cursor &toFirst();
      cursor &toNext();
      BOOL    isValid() const;

      JPage *current() const;
      JPage *operator -> () const;
   };

   // event handling
   BOOL event( const JCtlEvent &e);

   // return true to cancel the pending page turning
   virtual BOOL aboutToShowPage( const JPage &/*current*/, const JPage &/*new_pg*/)
   { return false; }

   virtual BOOL justShownPage( const JPage &/*new*/, const JPage &/*old_pg*/)
   { return false; }

   virtual BOOL pageSizeChanged( const JRect &/*newRect*/)
   { return false; }
};

// Page class; used to represent a page of a notebook ------------------------
class JPage
{
   ulong      pageID;
   JNotebook *book;
   friend class JNotebook;
   friend class JNotebook::cursor;

 public:
   JPage();
   JPage( const JPage &copy);
   JPage( const JNotebook *nb, ulong pID);
   JPage( JNotebook *nb, ulong pID);
   JPage &operator = ( const JPage &op2);

   // status text
   JStr   statusText() const;
   JPage &setStatusText( const char *s);

   // page data - a 4-byte space for your edification
   ulong  data() const;
   JPage &setData( ulong l);

   // tab text
   JStr   tabText() const;
   JPage &setTabText( const char *s);

   // tab bitmaps
   JBitmap tabBitmap() const;  // throws if nothing there
   JPage  &setTabBitmap( const JBitmap &b);

   // page window object; throws if not there
   JWindow *window() const;
};

#endif
