/*
 * JPSpace.hpp
 *
 * Presentation space classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jpspace_h
#define _jpspace_h

#include "JLib.h"
#include "JGfx.hpp"
#include "JWindow.hpp"

class JFont;
class JIcon;
class JRect;
class JSize;
class JPoint;
class JColour;
class JPSData;
class JBitmap;
class JGObject;
class JGSettings;
class JReadOnlyDC;
class JArcSettings;
class JPSBoundsData;
class JLineSettings;
class JImageSettings;
class JDeviceContext;
class JMarkerSettings;

class JSavedPS
{
   ulong hsps;

 public:
   JSavedPS( ulong h) : hsps( h) {}
   ulong handle() const { return hsps; }
};

class JBPSpace
{
 protected:
   // various 'set-id' bits
   friend class JMarkerSettings;
   friend class JFillSettings;
   friend class JTextSettings;
   JPSData *data;
   ulong    openFont( const JFont &f) const;
   ulong    newID() const;

   ulong hps;

   void setup();

   JBPSpace();

 public:
   JBPSpace( ulong wrapper);
   virtual ~JBPSpace();

   // Getting at the PM handle ------------------------------
   ulong handle() const     { return hps; }
   operator ulong () const  { return hps; }

   // Quick 'n dirty ----------------------------------------
   JBPSpace &fillRect( const JRect &rcl, const JColour &c);
   JBPSpace &invertRect( const JRect &rcl);
   JBPSpace &drawBitmapAt(const JBitmap &bmp, const JPoint &position);
   JBPSpace &drawBitmapScaled( const JBitmap &bmp, const JRect &destination);
   JBPSpace &drawIcon( const JIcon &ic, const JPoint &pt,
                       JGfx::icon::type t = JGfx::icon::normal);

   // DC association ----------------------------------------
   JBPSpace &disassociate();
   JBPSpace &associate( const JDeviceContext &dc);

   JReadOnlyDC dc() const;

   // clear ps ----------------------------------------------
   JBPSpace &clear();

   // saving and restoring ps -------------------------------
   JSavedPS save() const;
   JBPSpace &restore( const JSavedPS &p);

   JGfx::units::type units() const;
   JBPSpace &setUnits( JGfx::units::type);

   // reset - attributes, clear retaineds, delete fonts -----
   JBPSpace &reset( BOOL segs = false, BOOL complete = false);

   // selecting & deselecting bitmaps -----------------------
   JBPSpace &select( const JBitmap &bmp);
   JBPSpace &deselectBitmap();

   // Primitive attributes ----------------------------------
   enum undoT {
      on,
      off
   };

   undoT     undoState() const;
   JBPSpace &setUndo( undoT s = on);
   JBPSpace &undo( ulong howMany = 1);

   JGSettings settings() const;

   JLineSettings lineSettings() const;
   JLineSettings defLineSettings() const;

   JImageSettings imageSettings() const;
   JImageSettings defImageSettings() const;

   JMarkerSettings markerSettings() const;
   JMarkerSettings defMarkerSettings() const;

   JFillSettings fillSettings() const;
   JFillSettings defFillSettings() const;

   JTextSettings textSettings() const;
   JTextSettings defTextSettings() const;

   JArcSettings arcSettings() const;
   JArcSettings defArcSettings() const;

   // Quick ops.
   JBPSpace &setColour( const JColour &c);
   JBPSpace &setMix( JGfx::fgMix::type);

   // Rendering primitives ----------------------------------
   JBPSpace &render( const JGObject &g);

   // current state -----------------------------------------
   JFont  font() const;
   JPoint position() const;
};

// use to accumulate boundary data of drawing operations
class JPSBoundsData
{
   JBPSpace &ps;

 public:
   JPSBoundsData( JBPSpace &ps);
  ~JPSBoundsData();
   JPSBoundsData &reset();
   JRect bounds() const;
};

// actual classes which are in general use follow...

// A micro-cached PS linked to a window; needed for bizarre reasons
class JWindowPS : public JBPSpace
{
 public:
   JWindowPS( ulong wrapper = 0) : JBPSpace( wrapper) {}
   JWindowPS( JWindow *w);
   virtual ~JWindowPS();
};

class JClippedPS : public JBPSpace
{
 public:
   JClippedPS( JWindow *w, JWindow::clipType);
  ~JClippedPS();
};

class JDragPS : public JBPSpace
{
 public:
   JDragPS( JWindow *w);
  ~JDragPS();
};

// A general-use micro PS for any use you like
class JPSpace : public JBPSpace
{
 public:
   JPSpace( const JDeviceContext &dc, JGfx::units::type u = JGfx::units::pels);
  ~JPSpace();
};

// A normal PS, again general use
class JNormalPS : public JBPSpace
{
 public:
   JNormalPS( const JDeviceContext &d, JGfx::units::type u = JGfx::units::pels);
  ~JNormalPS();
};

#endif
