/*
 * JPtr.hpp
 *
 * Definition for smart pointer classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jptr_hpp
#define _jptr_hpp

#include "JExcept.hpp"

class JNullPointer : public JException { public: JNullPointer(); };

#define nullref jlib_throw( new JNullPointer);

// simple pointer object ----------------------------------------------------
template <class T>
class auto_ptr
{
   T    *obj;
   BOOL  owns;

 public:
   // init, term
   auto_ptr( T *t = 0) : obj( t), owns( true) {}
   auto_ptr( const auto_ptr<T> &copy) : obj( copy.obj), owns( true)
   { ((auto_ptr<T> &)copy).owns = false; }
   virtual ~auto_ptr() { if( owns) delete obj; }

   // assignment
   auto_ptr<T> &operator = ( const auto_ptr<T> &ass)
   { if( owns) delete obj; obj = ass.obj; ((auto_ptr<T> &)ass).owns = false;
     return *this; }

   // implicit casts
   operator T *() { return obj; }
   operator const T *() const { return obj; }

   // explicit cast
   T *operator()() { return obj; }

   // smart pointerness
   T *operator -> () { if( obj) return obj; nullref; return 0; }
   const T *operator -> () const { if( obj) return obj; nullref; return 0; }

   friend const T &realElement( const auto_ptr<T> &p) { return *p; }
};

// pointer wrapper for collections ------------------------------------------
template <class T>
class coll_ptr
{
   T *obj;

 public:
   coll_ptr( T *t = 0) : obj( t) {}
   coll_ptr<T> &operator = ( const coll_ptr<T> &ass) { obj = ass; }
   operator T* () { return obj; }
   T *operator -> () { if( obj) return obj; nullref; return 0; }

   friend const T &realElement( const coll_ptr<T> &p) { return *p; }
};

// reference-counted pointer object -----------------------------------------
template <class T>
class JPtr
{
   struct Ref
   {
      T *obj;
      ulong rc;
      Ref( T *t) : obj( t), rc( 1) {}
     ~Ref() { delete obj; }
   } *ref;

 public:
   // construct from c pointer
   JPtr( T *t = 0) : ref( 0) { if( t) ref = new Ref( t); }

   // copy constructor
   JPtr( const JPtr<T> &copy) : ref( copy.ref) {
      if( ref) ref->rc++;
   }

   // dtor
   virtual ~JPtr() {
      if( ref && !--ref->rc) delete ref;
   }

   // assignment
   JPtr<T> &operator = ( const JPtr<T> &assign) {
      if( ref && !--ref->rc) delete ref;
      ref = assign.ref;
      ref->rc++;
      return *this;
   }

   JPtr<T> &operator = ( T *assign) {
      if( ref && !--ref->rc) delete ref;
      ref = new Ref( assign);
      return *this;
   }

   // access
   operator T * ()   { if( ref) return ref->obj; return 0; }
   operator T * () const { if( ref) return ref->obj; return 0; }
   T *operator -> () { if( ref) return ref->obj; nullref; return 0; }
   const T *operator -> () const { if( ref) return ref->obj; nullref; return 0; }

   // for collections
   friend const T &realElement( const JPtr<T> &p) { return *p; }
};

#endif
