/*
 * JSeq.hpp
 *
 * Sequence collection class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jseq_h
#define _jseq_h

#include "JLib.h"
#include "JColln.hpp"

#include <string.h>

#define INVALID_INDEX ((ulong)-1)

// A sequence is arbitrarily ordered (==vector) -----------------------------
template < class T>
class JSequence
{
   JSequence( const JSequence<T> &) {}
   JSequence<T> &operator = ( const JSequence<T> &) { return self; }

 protected:
   T     **vector;
   ulong used, have;

   BOOL check( ulong pos) const {
      BOOL rc = true;
      if( pos >= used) {
         rc = false; jlib_throw( new JBoundsError( "element"));
      }
      return rc;
   }

   // grow array if necessary (never shrink it!)
   void ensure( int delta = 10) {
      if( used == have) {
         T **vec2 = new T* [ have + delta];
         memcpy( vec2, vector, have * sizeof(T*));
         delete [] vector;
         vector = vec2;
         have += delta;
      }
   }

 public:
   JSequence( ulong sz = 0) : vector( 0), used( 0), have( 0)
   {
      if( sz) ensure( sz);
   }

   virtual ~JSequence() { empty(); }

   // requested `position's must be >= 0 && < elements()
   int  elements() const { return used; }

   void empty() {
      for( ulong i = 0; i < used; i++) delete vector[ i];
      delete [] vector; vector = 0;
      used = have = 0;
   }

   // add an element at the end of the sequence...
   void append( const T &el) {
      ensure();
      vector[ used] = new T( el);
      used++;
   }

   // ...at the start...
   void prepend( const T &el) {
      ensure();
      for( ulong i = used; i > 0; i--) vector[ i] = vector[ i - 1];
      vector[ 0] = new T( el);
      used++;
   }

   // ...before a given position...
   void insertBefore( const T &el, ulong position)
   {
      if( check( position)) {
         ensure();
         for( ulong i = used; i > position; i--) vector[ i] = vector[ i - 1];
         vector[ position] = new T( el);
         used++;
      }
   }

   // ...and after a given position.
   void insertAfter( const T &el, ulong position)
   {
      if( check( position)) {
         ensure();
         for( ulong i = used; i > position + 1; i--) vector[ i] = vector[ i - 1];
         vector[ position + 1] = new T( el);
         used++;
      }
   }

   // look at an element in the sequence
   T *elementAt( ulong position)
   {
      T *t = 0;
      if( check( position)) t = vector[ position];
      return t;
   }

   T *operator [] ( ulong position)
   { return elementAt( position); }

   const T *elementAt( ulong position) const
   {
      const T *t = 0;
      if( check( position)) t = vector[ position];
      return t;
   }

   const T *operator [] ( ulong position) const
   { return elementAt( position); }

   // remove an element at a position...
   void removeAt( ulong position)
   {
      if( check( position)) {
         delete vector[ position];
         used--;
         for( ulong i = position; i < used; i++) vector[ i] = vector[ i + 1];
      }
   }

   // ...or at a cursor
   class cursor
   {
      friend class JSequence<T>;

      ulong         index;
      const JSequence<T> &seq;

    public:
      cursor( const JSequence<T> &s) : index( INVALID_INDEX), seq( s) {}

      BOOL isValid() const { return index != INVALID_INDEX; }

      void toFirst() { index = seq.used ? 0 : INVALID_INDEX; }

      void toNext() {
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else {
            if( index == seq.used - 1) index = INVALID_INDEX;
            else index++;
         }
      }

      void toLast() { index = seq.used ? seq.used-1 : INVALID_INDEX; }

      void toPrevious() {
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else {
            if( index == 0) index = INVALID_INDEX;
            else index--;
         }
      }

      T *current()
      {
         T *t = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor());
         else t = seq.vector[ index];
         return t;
      }

      const T *current() const
      {
         T *t = 0;
         if( !isValid()) jlib_throw( new JInvalidCursor);
         else t = seq.vector[ index];
         return t;
      }
   };

   friend class cursor;

   void removeAt( const cursor &c) { removeAt( c.index); }

   // run an iterator
   class iterator
   {
    public:
      virtual void action( T &t) = 0;
   };

   class const_iterator
   {
    public:
      virtual void action( const T &t) const = 0;
   };

   void doForAll( iterator &iter) {
      for( ulong i = 0; i < used; i++) iter.action( *vector[ i]);
   }

   void doForAll( const const_iterator &iter) {
      for( ulong i = 0; i < used; i++) iter.action( *vector[ i]);
   }
};

#endif
