/*
 * JSlider.hpp
 *
 * GUI Sliders
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

/* Classes defined here: JBSlider, an abstract-ish class from which are 
   derived JVertSlider, JHorzSlider, JProgressBar
*/

#ifndef _jslider_h
#define _jslider_h

#include "JLib.h"
#include "JControl.hpp"

class JSize;
class JPoint;
class JWindow;

// A detent - used to define a position somewhere
class JDetent
{
   ulong hDetent;

 public:
   JDetent( ulong h)    { hDetent = h; }

   ulong handle() const { return hDetent; }
};

// Base slider class
class JBSlider : public JControl
{
   JMR qsi( const JMP &mp1) const;
   void ssi( const JMP &mp1, ulong mp2) const;

 protected:
   JBSlider( JWindow *parent, const JPoint &pos, const JSize &size,
             ulong id, ulong style, ushort ticks, ushort spacing);
 public:
   JBSlider( ulong hwnd) : JControl( hwnd) {}
   JBSlider( JWindow *w, ulong id) : JControl( w, id) {}

   // detents
   JDetent   addDetent( ushort pelOffset) const;
   JDetent   addDetentAtArm() const;
   JBSlider &removeDetent( const JDetent &d);
   ushort    detentPosition( const JDetent &d) const;

   // tick mark annotations
   JBSlider &addTickText( ushort tick, char *label);
   JStr      tickText( ushort tick) const;

   // metrics: shaft size...
   JSize     shaftSize() const;
   JBSlider &setShaftBreadth( ulong b);

   // ...position of the slider arm...
   ushort    armPosInPels() const;
   ushort    armPosInTicks() const;
   JBSlider &moveArmInPels( ushort pels);
   JBSlider &moveArmInTicks( ushort tick);

   // ...position of shaft...
   JPoint    shaftPosition() const;
   JBSlider &moveShaftTo( const JPoint &pos); 

   // ...arm size...
   JSize     armSize() const;
   JBSlider &setArmSize( const JSize &s);

   // ...tick size...
   ushort    tickLength( ushort tick) const;
   JBSlider &setTickLength( ushort tick, ushort size);
   JBSlider &setAllTicks( ushort size);

   // ...tick position.
   JPoint tickPosition( ushort tick) const;

   // event handling
   BOOL event( const JCtlEvent &e);
   virtual BOOL changed()     { return false; }
   virtual BOOL tracking()    { return false; }
   virtual BOOL lostFocus()   { return false; }
   virtual BOOL gainedFocus() { return false; }

   static const ulong
      snapToTick,
      ownerDraw;
};

// vertical slider
class JVertSlider : public JBSlider
{
 public: 
   JVertSlider( ulong hwnd) : JBSlider( hwnd) {}
   JVertSlider( JWindow *w, ulong id) : JBSlider( w, id) {}

   static const ulong
      normal,
      sliderInMiddle,
      sliderOnLeft,
      sliderOnRight,
      scaleOnRight,
      scaleOnLeft,
      homeAtBottom,
      homeAtTop,
      buttonsAtTop,
      buttonsAtBottom;

    JVertSlider( JWindow *w, const JPoint &p, const JSize &s, ulong id,
                 ushort ticks, ulong style = JVertSlider::normal);
};

// Horizontal slider
class JHorzSlider : public JBSlider
{
 public: 
   JHorzSlider( ulong hwnd) : JBSlider( hwnd) {}
   JHorzSlider( JWindow *w, ulong id) : JBSlider( w, id) {}

   static const ulong
      normal,
      sliderInMiddle,
      sliderAtTop,
      sliderAtBottom,
      scaleAtTop,
      scaleAtBottom,
      homeOnLeft,
      homeOnRight,
      buttonsOnLeft,
      buttonsOnRight;

   JHorzSlider( JWindow *w, const JPoint &p, const JSize &s, ulong id,
                ushort ticks, ulong style = JHorzSlider::normal);
};

// Progressbar : horizontal, readonly.
class JProgressBar : public JBSlider
{
 public:
   JProgressBar( ulong hwnd) : JBSlider( hwnd) {}
   JProgressBar( JWindow *w, ulong id) : JBSlider( w, id) {}

   static const ulong
      normal;

    JProgressBar( JWindow *w, const JPoint &p, const JSize &s, ulong id,
                  ushort ticks = 100, ulong style = JProgressBar::normal);
};

#endif
