/*
 * JStr.hpp
 *
 * Small string class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jstr_h
#define _jstr_h

#include "JLib.h"
#include "JBuffer.hpp"
#include "JModule.hpp"

// JStr is referenced counted, providing copy on write.  This means that it
// is very cheap to return instances by value from functions - the string is
// not copied by this operation, but the pointer to it shared.

class JResID;
class JStringRef;

class JStr : public JVBuffer
{
   void cow();

 protected:
   JStringRef *pRef;

 public:
   JStr();  // empty string
   JStr( const JStr &copy);
   JStr( const char *s);
   // load from a stringtable
   JStr( const JResID &id, const JModule &mod = JModule::theResMod);
  ~JStr();

   BOOL isValid() const;

   // operators defined like this basically to avoid major heartache
   // when attempting to compare two pointers.
   int operator == ( const char *op2) const;
   int operator != ( const char *op2) const;
   int operator < ( const char *op2) const;
   int operator > ( const char *op2) const;

   // implicit & explicit access to contents
   char *buffer();

   operator const char * () const;
   operator char * ();

   // size of buffer; strlen + 1
   ulong length() const;
   JStr &setsize( long l);

   long asInt() const;
   double asDouble() const;

   JStr &add( const char *s);
   JStr &add( char c);
   JStr &reverse();

   JStr  operator + ( const char *op2);
   JStr &operator = ( const char *op2);
   JStr &operator = ( const JStr &op2);
   JStr &operator += (const JStr &op2);

   char &operator [] ( ulong l);
   char  operator [] ( ulong l) const;

   BOOL endsWith( const char *s2) const;
   BOOL startsWith( const char *s2) const;

   BOOL contains( const char *needle) const;
   // 0-based
   JStr substring( long start, long count) const;

   JStr left( int numchars) const;
   JStr right( int numchars) const;

   // vbuffer methods
   void *pvAddr() const;
   ulong cbSize() const;

   // hashcode
   ulong hashcode() const;
};

// a string constuctable from a printf-like thing
class JVStr : public JStr
{
 public:
   JVStr( const char *fmt, ...);
};

inline ulong hash( const JStr &str) { return str.hashcode(); }

// this is here purely as a debugging aid!
struct JStringRef
{
   char *string;
   ulong refs;
   JStringRef( const JStringRef &r);
   JStringRef( const char *s);
  ~JStringRef();
};

#endif
