/*
 * JStream.hpp
 *
 * Base classes for stream hierarchy
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jstream_h
#define _jstream_h

#include "JLib.h"
#include "JBuffer.hpp"
#include "JStr.hpp"
#include "JExcept.hpp"

#define HBAD ((ulong)-1)

// base stream handle manager class ------------------------------------------
class JStream
{
   friend class JPipe;

 protected:
   ulong hStream;
   void dup( ulong toDup);
   // protected constructor; only accessable to descendants
   JStream() : hStream( HBAD) {}
   JStream( ulong wrapme) : hStream( wrapme) {}

 public:
   // DupHandle: create new
   JStream( const JStream &copy) : hStream( HBAD) { dup( copy.hStream); }
   // DupHandle: alias
   JStream &operator = ( const JStream &dupe)
      { dup( dupe.hStream); return self; }

   virtual ~JStream() { if( hStream > 2 && hStream != HBAD) close(); }

   // this also does a forget()...
   JStream &close();
   // stops hStream from being Close'd in 'dtor
   JStream &forget() { hStream = HBAD; return self; }

   ulong handle() const { return hStream; }

   enum sharing {
      denyRead  = 0x10,
      denyWrite = 0x20,
      denyAll   = 0x30,
      denyNone  = 0x40
   };
};

// a stream from which you can read data -------------------------------------
class JRStream : public virtual JStream
{
   friend class JPipe;
   void read( void *addr, ulong sz) const;

 protected:
   JRStream() : JStream() {}

 public:
   JRStream( ulong wrapme) : JStream( wrapme) {}
   JRStream( const JRStream &copy) : JStream( HBAD) { dup( copy.hStream); }
   JRStream & operator = ( const JRStream &dupe)
      { dup( dupe.hStream); return self; }
   virtual ~JRStream() {}

   // reading things
   char readc() const;
   long readl() const;
   virtual JStr readLine() const;
   void read( const JVBuffer &buffer) const
      { read( buffer.pvAddr(), buffer.cbSize()); }
};

extern JRStream cin;

// thrown when reading past end of file
struct JEOF : public JException { JEOF(); };

// a stream to which you can write data --------------------------------------
class JWStream : public virtual JStream
{
   friend class JPipe;
   void write( void *addr, ulong sz) const;

 protected:
   JWStream() : JStream() {}

 public:
   JWStream( ulong wrapme) : JStream( wrapme) {}
   JWStream( const JWStream &copy) : JStream( HBAD) { dup( copy.hStream); }
   JWStream & operator = ( const JWStream &dupe)
      { dup( dupe.hStream); return self; }
   virtual ~JWStream() {}

   // writing things
   JWStream &writec( const char c)
      { write( (void *) &c, sizeof c); return self; }
   JWStream &writel( const long l)
      { write( (void *) &l, sizeof l); return self; }
   JWStream &writeLine( const char *s);
   JWStream &write( const JVBuffer &buffer)
      { write( (void *) buffer.pvAddr(), buffer.cbSize()); return self; }
};

extern JWStream cout;
extern JWStream cerr;

// thrown when writing to a broken pipe
// ( yes, it's in the wrong place, but i can't be bothered to copy write
// into the npipe object; if a write to a non-pipe stream returns
// ERROR_PIPE_BROKEN, then this is the least of our worries...
struct JBrokenPipe : public JException
{
   JBrokenPipe();
};

// thrown when a write operation fails ( full diskette, non-blocking pipes)
class JIncompleteWrite : public JException
{
   ulong cbActual;

 public:
   JIncompleteWrite( ulong act);
   ulong actual() const { return cbActual; }
};

#endif
