/*
 * JThread.hpp
 *
 * Definition for thread class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jthread_h
#define _jthread_h

#include "JLib.h"
#include "JEvSem.hpp"
#include "JHandler.hpp"
#include "JEvent.hpp"

class JWindow;

/* Thread callbacks ----------------------------------------------------------
   Threads can be made to look like function calls. When 'start'ed, you may
   supply a JWindow * and|or an 'id'. If you do at least the former, the
   window will be posted an event with id JThread::overMsgID. mp1 contains the
   rc from your thread and mp2 contains the id supplied earlier.
   Catch this in JWindow::handleEvents() or use the handler.

   Note that the thread that gets started ALREADY has a JPM object running.
   ------------------------------------------------------------------------ */

// thread class; derive off here for own threads, overloading the operator()
// method.  Calling start() will begin a new thread in that method.
class JThread
{
 protected:
   ulong tid;

 public:
   JThread() : tid( 0) {}
   void start( JWindow *c = 0, ulong id = 0, ulong p = 0, ulong d = 0);
   void wait();
   void kill();
   void suspend();
   void resume();

   virtual ulong operator () () = 0;

   // pause current thread
   static void sleep( ulong ulMilliseconds);

   static const ulong overMsgID;
};

class JThreadEvent : public JEvent
{
 public:
   JThreadEvent( JEventData *e) : JEvent( e) {}
   ulong rc() const                          { return e->mp1; }
   ulong ID() const                          { return e->mp2; }
};

class JThreadHandler : public JHandler
{
 public:
   JThreadHandler() : JHandler() {}
   BOOL handle( JEventData *e);

   virtual BOOL threadOver( const JThreadEvent &) { return false; }
};

// classes to start member functions as threads
/* Example ------------------------------------------------------------------

   class MyBigClass
   {
      JMemberThread<MyBigClass> thread;

      myclass()
      {
         // start a couple of threads
         thread.start( &MyBigClass::threadfn, this);
         thread.start( &MyBigClass::threadfn, this);
      }

      ulong threadfn()
      {
         // runs in a new thread
      }
   };

   ----------------------------------------------------------------------- */

template <class T>
class JMemberThread : public JThread
{
   typedef ulong (T::* pmf)();
   pmf           pThreadfn;
   T            *obj;
   JEventSem     sem;

 public:
   JMemberThread() : sem( false, true) {}
   void start( pmf p, T *object, JWindow *c = 0,
               ulong id = 0, ulong prio = 0, ulong d = 0)
   {
      pThreadfn = p;
      obj = object;
      sem.reset();
      JThread::start( c, id, prio, d);
      sem.wait();
   }

   ulong operator () ()
   {
      T *t = obj;
      pmf pfnn = pThreadfn;
      sem.post();
      // member data for this class may have gone out of scope
      return (t->*pfnn)();
   }
};

#endif
