/*
 * JVSet.hpp
 *
 * Value-set GUI component
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

/* Valuesets - used to provide multi-exclusive selection for text, bitmaps,
   icons, colours or custom things. Includes drag'n'drop support via
   subclassing, as usual. On creation, a vset is told the default item type;
   this can be changed for individual items. The coordinate system is 1-based,
   from the top-left.
*/

#ifndef _jvset_h
#define _jvset_h

#include "JLib.h"
#include "JControl.hpp"
#include "JDrag.hpp"
#include "JHelp.hpp"

class JStr;
class JIcon;
class JSize;
class JPoint;
class JColour;
class JBitmap;
class JWindow;
class JDragSet;
class JDragItem;
class JCtlEvent;

class JValueset : public JControl
{
   ulong datum( const JPoint &c) const;
   void  datam( const JPoint &c, ushort fs, ulong d) const;
   void  go();

 public:

   // style
   static const unsigned long
      normal,
      border,
      itemBorder,
      scaleBitmaps,
      custom;

   // default item type
   enum itemType {
      bitmap = 0x1,
      icon   = 0x2,
      text   = 0x4,
      colour = 0x8
   };

   JValueset( ulong hwnd);
   JValueset( JWindow *parent, ulong id);
   JValueset( JWindow *parent, const JPoint &pos, const JPoint &size, ulong id,
              const JSize &celSize, itemType i, ulong style = normal);

   // set / get cell contents. Exception thrown if format's incorrect.
   JValueset &setColour( const JPoint &cel, const JColour &col);
   JValueset &setIcon( const JPoint &cel, const JIcon &ic);
   JValueset &setBitmap( const JPoint &cel, const JBitmap &bmp);
   JValueset &setStr( const JPoint &cel, char *txt);
   JColour    asColour( const JPoint &cel) const;
   JIcon      asIcon( const JPoint &cel) const;
   JBitmap    asBitmap( const JPoint &cel) const;
   JStr       asStr( const JPoint &cel) const;

   itemType   typeOf( const JPoint &cel) const;

   JValueset  &hilight( const JPoint &cel);
   JPoint      chosen() const;

   // enablement
   JValueset &enable( const JPoint &cel, BOOL yes = true);
   JValueset &disable( const JPoint &cel) { return enable( cel, false); }

   // custom-making; see JVSetODH.hpp for handler details
   JValueset &makeCustom( const JPoint &cel);

   // dimensions
   JSize cells();
   JValueset &setSize( const JSize &s);

   // event handling
   BOOL event( const JCtlEvent &e);
   virtual BOOL lostFocus()                 { return false; }
   virtual BOOL gainedFocus()               { return false; }
   // doubleclick
   virtual BOOL opened( const JPoint &)     { return false; }
   // cursor/mouse selection
   virtual BOOL selected( const JPoint &)   { return false; }
   // for the vset
   virtual JHelpID help()                   { return JHelpID( 0); }
   // use 'new' or NULL
   virtual JDragItem *initDrag( const JPoint &) { return 0; }
   virtual JDropOk isDropOk( const JPoint &, JDragSet &)
      { return JDropOk( JDrag::no_location, JDrag::def); }
   virtual void dragLeave( const JPoint &) {}
   virtual JHelpID dropHelp( const JPoint &, JDragSet &)
      { return JHelpID( 0); }
   virtual void drop( const JPoint &, JDragSet &) {}
};

#endif
