/*
 * JWindow.hpp
 *
 * Basic window class definition
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#ifndef _jwindow_h
#define _jwindow_h

#include "JLib.h"
#include "JPParam.hpp"
#include "JMParam.hpp"
#include "JCoord.hpp"
#include "JSWP.hpp"
#include "JStr.hpp"

class JWindowTimer;
class JWindowData;
class JEventData;
class JWindowPS;
class JHandler;
class JControl;
class JProfile;
class JEvent;
class JColour;
class JFont;

// All windows - controls, frames, etc. all derive from this class.
// It is the responsibility of the descendent to get a valid hwnd and then
// call the JWindow::subclass method, typically in their ctors.

// Using any method other than the destructor or setHwnd of a JWindow which
// hasn't been given a hwnd will produce undefined behaviour...

class JWindow
{
   friend void * _System JLibFnWp( ulong, ulong, void *, void *);
   friend class JLoopyHandler;
   friend class JDrawLoopyHandler;
   friend class JTimerHandler;
   friend class JWindowPtr;

   // copying considered harmful...
   JWindow( const JWindow &) {}

   BOOL checkFirstClass();
   void makeSecondClass();

 protected:
   // interface for subclassers to use

   // no-arg constructor
   JWindow();
   JWindow &subclass();                    // enable handers
   JWindow &setautodelete( BOOL b = true); // destroy pm window in dtor
   JWindow &setHwnd( ulong hwnd);          // set the pm window handle

   // this should be thought of as read-only; change at your peril.
   class Data;
   Data *wdata;

 public:
   // construct from a given hwnd
   JWindow( ulong hwnd, BOOL autodelPMWindow = false, BOOL subclass = false);
   // construct from the window at the given position on the desktop
   JWindow( const JPoint &pt, BOOL autodelPMwin = false, BOOL sub = false);
   virtual ~JWindow();

   void destroy();

   struct Data
   {
       ulong hwnd;
       ushort  id;
   };

   // window style flags, included here for name space-niceness
   static const unsigned long
      normal,
      syncpaint,
      animate,
      clipChildren,
      clipSiblings,
      clipToParent,
      saveBits,
      visible,
      minimized;

   JWindow &setStyle( unsigned long flags, BOOL or = true);
   BOOL     hasStyle( unsigned long flags) const;

   // enablement
   JWindow &enable( BOOL ok = true);
   JWindow &disable() { return enable( false); }

   // visibility
   virtual JWindow &show( BOOL ok = true);
   virtual JWindow &hide() { return show( false); }

   // focus
   JWindow &activate();

   // z-ordering
   JWindow &toTop();
   JWindow &toBottom();
   JWindow &goBehind( JWindow *w) { return setPosition( JSWP( w)); }

   // validity - for checking in horrid situations
   BOOL isValid() const;

   // resizing, moving - parent's coordinate system
   JWindow &moveTo( const JPoint &p)      { return setPosition( JSWP( p)); }
   JWindow &resize( const JSize &p)       { return setPosition( JSWP( p)); }
   JWindow &reshape( const JPoint &pos, const JSize &size) { return setPosition( JSWP( pos, size)); }
   JWindow &reshape( const JRect &rect)   { return setPosition( JSWP( rect)); }
   JWindow &setPosition( const JSWP &swp);
   JWindow &moveOverCursor();
   JWindow &centreOver( JWindow *w);

   // current sizes & positions: uses parent's coordinate system
   JRect    rect() const        { return position().rect(); }
   JSize    size() const        { return position().size(); }
   JPoint   origin() const      { return position().pos(); }
   JSWP     position() const;

   // window ID
   virtual const ushort &ID() const { return wdata->id; }
   virtual JWindow &setID( ushort id);

   // state queries
   BOOL isShowing() const;
   BOOL isVisible() const;
   BOOL isEnabled() const;
   BOOL hasFocus() const;
   BOOL isMinimized() const  { return hasStyle( JWindow::minimized); }

   // owner & parent control
   JWindow *owner() const;
   JWindow &setOwner( JWindow *w);
   JWindow *parent() const;
   JWindow &setParent( JWindow *w);

   // message processing - attaching a handler
   virtual JWindow &attach( JHandler *h);
   virtual JWindow &detach( JHandler *h);
   JMR      defaultAction( const JEvent &e);
   JMR      defaultAction( ulong msg, const JMP &mp1 = JMP(), const JMP &mp2 = JMP());
   JMR      sendEvent( const JEvent &e) const;
   JWindow &postEvent( const JEvent &e);
   JMR      sendEvent( ulong msg, const JMP &mp1 = JMP(), const JMP &mp2 = JMP()) const;
   JWindow &postEvent( ulong msg, const JMP &mp1 = JMP(), const JMP &mp2 = JMP());

   // return true to say 'yup, I've handled the event guv.'
   // default returns false, and the handlers/defwp get called.
   virtual BOOL preHandleEvents( JEventData *) { return false; }
   // called after handlers/def proc
   virtual void postHandleEvents( JEventData *) { }

   static JMR  SendEvent( ulong hwnd, ulong msg, const JMP &mp1 = JMP(), const JMP &mp2 = JMP());
   static void PostEvent( ulong hwnd, ulong msg, const JMP &mp1 = JMP(), const JMP &mp2 = JMP());

   // painting support
   // NB to create a PS for a window, construct a JWindowPS object with the
   // window as a parameter - see JPSpace.hpp
   enum clipType {
      siblings = 0x08,
      children = 0x10
   };

   static JWindowPS getDesktopPS();

   JWindow &invalidate( const JRect &rcl = JRect());
   JWindow &invalidate( ulong region);
   JWindow &validate( const JRect &rcl);
   JWindow &validate( ulong region);
   JRect    updateRect() const;
   ulong    updateRegion() const;
   JWindow &lock( BOOL l = true);
   JWindow &unlock() { return lock( false); }

   // contents setting
   virtual JWindow &setText( const char *text);
   virtual JStr     getText() const;

   // presentation parameters
   JWindow &setFont( const JFont &f, ulong pointsize);
   JWindow &setFontNameSize( const char *string);
   JStr     getFontNameSize();
   JWindow &removeFont();
   JWindow &setColour( JPP::colour, const JColour &color);
   JColour  getColour( JPP::colour, BOOL pure = false);

   // pointer capture
   JWindow &capturePointer();
   JWindow &releasePointer();
   BOOL     hasPointer();

   // window coordinate mapping
   JPoint fromDesktopCoords( const JPoint &from);
   JPoint toDesktopCoords( const JPoint &from);
   static JPoint convertPos( JWindow &from, JWindow &to, const JPoint &pt);

   // window timers
   JWindow &startTimer( JWindowTimer *w, ulong msInterval);
   JWindow &resetTimer( JWindowTimer *timer, ulong msInterval);
   JWindow &stopTimer( JWindowTimer *w);

   // child manipulation
   JWindow &adopt( JControl *c);

   // returns 0 if child not found
   virtual JWindow *childWithID( ushort id);

   // implicit & explicit conversions from JWindow -> hwnd
   ulong    handle() const { return wdata->hwnd; }
   operator ulong () const { return handle(); }

   // desktop & object window accessors
   static JWindow theDesktopWindow;
   static JWindow theObjectWindow;

   int operator == (const JWindow &op2) { return handle() == op2.handle(); }

   // child window cursor class
   class cursor
   {
      ulong    hEnum;
      JWindow *wnd;
      JWindow *curr;
      void     check();

    public:
      cursor( JWindow *w);
     ~cursor();
      cursor &toFirst();
      cursor &toNext();
      BOOL isValid() const;
      JWindow *current();
   };
};

class JActionForbidden : public JException
{
 public:
   JActionForbidden();
};

class JDuplicateWindow : public JException
{
 public:
   JDuplicateWindow();
};

// if you have a PM window handle (HWND) that *may* be that of one of your
// controls for which you've created a JWindow [derived] object, use this
// class to get a pointer to the correct JWindow, which may be cast as you
// wish.  If you use JWindow::ctor#2, then bad things will happen.

// You shouldn't need to use this class unless you're interacting with bare
// PM functions (in which case there's a case for putting it in the main
// library)

class JWindowPtr
{
   JWindow *win;

 public:
   JWindowPtr( ulong hwnd);

   // access
   operator JWindow * () { return win; }
   operator const JWindow * () const { return win; }
   JWindow *operator -> () { return win; }
   const JWindow *operator -> () const { return win; }
};

class JWindowTimer
{
 public:
   virtual void operator () ( JWindow *wnd) = 0;
};

#endif
