/*
 * Paint - an image editor for OS/2 using JLib by John Fairhurst
 *
 * Colours.cpp - code for colour manager
 *               See doc/Colours
 *
 * Version history: 2/2/97  First Version
 *                  2/3/97  Rewrite
 *
 */

#pragma priority( 102)  // after PM & after the manager

#include <JLib.h>
#include <JCBox.hpp>
#include <JListBox.hpp>
#include <JPtr.hpp>

#include "Colours.hpp"
#include "ColourBase.hpp"
#include "ColourModels.hpp"
#include "PaintMgr.hpp"

// Colour manager class. Payoff from virtual ctor 'nonsense' elsewhere --------
ColourManager::ColourManager() : current( 0)
{
   // add our colour models. WBN to be dynamic here...
   cmps.add( JPtr<ColourModelProvider>( new NRGBProvider()));

   if( cmps.containsElementWithKey( mgr.settings.colourModel()))
      current = *cmps.elementWithKey( mgr.settings.colourModel());
   else {
      // if the default colour model is invalid, set it right
      CMPKeySet::cursor c( cmps);

      c.toFirst();
      mgr.settings.colourModel( (*c.current())->name());
      current = *c.current();
   }
}

JColour ColourManager::pick( JWindow *owner, const JColour &def)
{ return pick( current, owner, def); }

JColour ColourManager::pick( char *n, JWindow *o, const JColour &d)
{ return pick( *cmps.elementWithKey( n), o, d); }

JColour ColourManager::pick( ColourModelProvider *provider,
                             JWindow *owner, const JColour &def)
{
   // create an instance of the default model
   auto_ptr<ColourModel> arrgh = provider->create( owner, def);

   // run it. Test for app being shut down.
   if( JDialog::idCancel == arrgh->showModally() || !arrgh->isValid()) {
      throw Cancelled();
   }

   return arrgh->current();
}

// User management ------------------------------------------------------------
JMenu *ColourManager::fillMenu( JMenu *m)
{
   // Add each model to the menu
   CMPKeySet::cursor c( cmps);

   for_cursor( c)
      m->addItem( (*c.current())->name(), (*c.current())->ID());

   // set the default for the menu, and remember it for later
   m->makeCascade( current->ID());
   menulist.add( m);

   return m;
}

JComboBox *ColourManager::fillCBox( JComboBox *cb)
{
   // empty the combobox
   cb->listbox().empty();

   // insert each model into the combobox
   CMPKeySet::cursor c( cmps);

   for_cursor( c)
      cb->listbox().insert( JListBox::item( (*c.current())->name(),
                                            (*c.current())->ID()));

   // select the default - ugh method
   JListBox::cursor c2( &cb->listbox(), JListBox::all);

   for( c2.toFirst(); c2.isValid(); c2.toNext())
      if( c2->handle() == current->ID())
         cb->listbox().select( c2->index());

   return cb;
}

// This gets called by the palette settings page as it 'dumps' itself.
ColourManager &ColourManager::setDefault( char *s)
{
   // update our records
   current = *cmps.elementWithKey( s);

   // update the menus we know about
   JBag<JMenu*>::cursor c( menulist);

   for_cursor( c)
      (*c.current())->setDefault( current->ID());

   return *this;
}

// Exception ------------------------------------------------------------------
Cancelled::Cancelled() : JException( "Dlg cancelled", JExn::base)
{}

// Object ---------------------------------------------------------------------
ColourManager colourMgr;
