/*
 * Paint - an image editor for OS/2 using JLib by John Fairhurst
 *
 * Painting.cpp - the frame-window class that provides the 'interface' to
 * a painting.
 *
 * Version history: 26/1/97  First Version
 *
 */

#include <JPM.hpp>
#include <JDC.hpp>
#include <JDialog.hpp>
#include <JCoord.hpp>
#include <JViewer.hpp>
#include <JToolbar.hpp>
#include <JStatbar.hpp>
#include <JResID.hpp>
#include <JCmdH.hpp>
#include <JBitmap.hpp>
#include <JMsgBox.hpp>
#include <JColour.hpp>
#include <JCanvas.hpp>
#include <JSystem.hpp>
#include <JFont.hpp>
#include <JTrace.hpp>

#include "PaintMgr.hpp"
#include "Painting.hpp"
#include "Palette.hpp"
#include "ColourModels.hpp"
#include "Colours.hpp"
#include "ResID.h"

#define menu_count      10
#define menu_edit        0
#define menu_context     1
#define menu_tools1      2
#define menu_tools2      3
#define menu_style1      4
#define menu_style2      5
#define menu_masks1      6
#define menu_masks2      7
#define menu_pickc1      8
#define menu_pickc2      9

#define MAX( a, b) ((a) < (b) ? (b) : (a))
#define MIN( a, b) ((a) < (b) ? (a) : (b))

Painting::Painting( const JBitmap &bmp)
         : JFrame( ".Untitled", JFrame::normal | JFrame::noByteAlign),
           statbar( new JStatbar( this)),
           name( ".Untitled"), datasafe( true),
           toolbar( new JToolbar( this, idm_toolbar)),
           mouseHandler( this)
{ setup( bmp); }

Painting::Painting( char *filename) : JFrame( filename,
                                              JFrame::normal |
                                              JFrame::noByteAlign),
                                      statbar( new JStatbar( this)),
                                      toolbar( new JToolbar( this, idm_toolbar)),
                                      name( filename), datasafe( true),
                                      mouseHandler( this)
{
   // want to allow for other image formats, threadily too
// JBitmap bmp( filename);
   JBitmapReader reader( filename);
   int i = reader.elements();
   if( i == 0)
      JSystem::beep( 40, 40);
   else {
      for( int j = 1; j < i; j++)
         mgr.newPainting( **(reader[ j]));  // memory leak here...
      setup( **(reader[ 0]));
   }
}

void Painting::setup( const JBitmap &bmp)
{
   // the minimum size bitmap we can edit is 7x2 in JSystem::scrollWidth's
   JSize minPaintable( JSystem::scrollWidth * 7, JSystem::scrollWidth * 2);
   JSize maxSize( bmp.size());

   if( !(bmp.size() >= minPaintable)) {
      // create a bitmap to the minPaintable size, & blit ours onto a corner
      JSize newSize( MAX( minPaintable.x, bmp.size().x),
                     MAX( minPaintable.y, bmp.size().y));

      picture = new JBufferedCanvas( newSize, bmp.bpp(),
                                     &JWindow::theObjectWindow);
      picture->ps()->drawBitmapAt( bmp, JPoint( 0, newSize.y - bmp.size().y));
      maxSize = newSize;
   }
   else {
      picture = new JBufferedCanvas( maxSize, bmp.bpp(),
                                     &JWindow::theObjectWindow);
      picture->ps()->drawBitmapAt( bmp, JPoint());
   }
   activeSize = bmp.size();

   // Set up the main window hierarchy ----------------------------------------
   setOwner( mgr.father());

   // load the PM frame furniture
   setIcon( id_ico_paint);
   setAccel( id_acceltable);
   setMenu( idm_mainmenu);

   // set up a couple of frame extension windows
   addFrameExtension( statbar);
   addFrameExtension( toolbar);

   // make the statbar look nice
   statbar->window()->setColour( JPP::foreground, JColour::black);
   statbar->window()->setFont( JFont( "Helv"), 8);

   // create the viewer, and set sizes as appropriate
   picture->resize( maxSize);
   viewer = new JViewer( this, picture, JPoint(), JPoint(), 0);
   setClient( viewer);
   setMaxClientSize( maxSize);

   // Make sure our window size isn't invalid
   JSize sz = client()->size();
   if( !( sz <= maxSize))
      sizeClientTo( JSize( MIN( sz.x, maxSize.x),
                           MIN( sz.y, maxSize.y)));

   // set up menus ------------------------------------------------------------
   menus = new JMenu * [ menu_count];

   JMenu *m = getMenu();

   #define LOADMENU( i, j) menus[ i] = new JMenu( JResID( j))

   // the edit submenu - select mode context menu
   LOADMENU( menu_edit, idm_edit);
   m->addSubMenu( idm_edit, menus[ menu_edit]);
   menus[ menu_edit]->getSubMenu( idm_savesel).makeCascade( idm_saveselasbmp);

   // the usual context menu
   LOADMENU( menu_context, idm_context);

   // tools menus
   LOADMENU( menu_tools1, idm_tools);
   LOADMENU( menu_tools2, idm_tools);
   menus[ menu_context]->addSubMenu( idm_tools, menus[ menu_tools1]);
   m->addSubMenu( idm_tools, menus[ menu_tools2]);

   // style menus
   LOADMENU( menu_style1, idm_style);
   LOADMENU( menu_style2, idm_style);
   menus[ menu_context]->addSubMenu( idm_style, menus[ menu_style1]);
   m->addSubMenu( idm_style, menus[ menu_style2]);

   // masks menus
   LOADMENU( menu_masks1, idm_masks);
   LOADMENU( menu_masks2, idm_masks);
   menus[ menu_context]->addSubMenu( idm_masks, menus[ menu_masks1]);
   m->addSubMenu( idm_masks, menus[ menu_masks2]);

   // Colour picking menus; the colour manager looks after them
   LOADMENU( menu_pickc1, idm_pickcolour);
   LOADMENU( menu_pickc2, idm_pickcolour);
   menus[ menu_context]->addSubMenu( idm_pickcolour, menus[ menu_pickc1]);
   m->getSubMenu( idm_picture).addSubMenu( idm_pickcolour, menus[ menu_pickc2]);
   colourMgr.fillMenu( menus[ menu_pickc1]);
   colourMgr.fillMenu( menus[ menu_pickc2]);

   // misc menus things
   m->getSubMenu( idm_file).getSubMenu( idm_saveas).makeCascade( idm_saveasbmp);

   // last bit of setup -------------------------------------------------------
   picture->attach( &mouseHandler);
   showInTasklist();

   // create the palette window
   palette = new Palette( this);

}

Painting::~Painting()
{
   picture->detach( &mouseHandler);
   delete palette;
   delete picture;
   delete viewer;
   delete statbar;
   delete toolbar;
   // !! Need to deregister menus with the colour manager
   for( int i = 0; i < menu_count; i++)
      delete menus[ i];
}

Painting &Painting::show( BOOL f)
{
   JWindow::show( f);
   if( f && mgr.settings.showPalette())
      palette->moveOverCursor().show().activate(); // !! shurely not
   return *this;
}

// when the user asks to close the window, we tell the manager about it.
// He immediately calls us back. Never quit the q due to a window-close
BOOL Painting::closing()
{
   mgr.closeMe( this);
   return true;
}

// called when the manager thinks we ought to close. If data isn't safe, we
// can pop up a dialog to make sure things are okay.
// return true to say yes, go ahead and close us down. You callous bastard.
BOOL Painting::close()
{
   if( !datasafe) {
      JMsgBox mbox( 0, JVStr( "'%s' is unsaved. Quit anyway?", name.buffer()),
                    "Paint Question", JMsgBox::yesNo, JMsgBox::question);
      if( JMsgBox::no == mbox.open( this))
         return false;
   }
   return true;
}

// handle a myriad of menu commands
BOOL Painting::command( const JCmdEvent &e)
{
   ushort v = e.value();

   JSystem::beep( 40, 40);

   // First check for a colour selection
   if( v > idm_pickcolour && v <= idm_pickcolour_max) {
      try {
         JColour c = colourMgr.pick( getMenu()->itemText( v), this);
         palette->addColour( c);
         picture->ps()->setColour( c);
      } catch( Cancelled) {
      }
      return true;
   }

   switch( v) {
      case idm_show_palette:
         palette->request();
         return true;
      case idm_settings:
         mgr.showSettings();
         return true;
      case idm_complete:
      {
         ulong i = picture->bitmap()->bpp();
         trace << JVStr( "%i", i).buffer();
         break;
      }
      case idm_saveasbmp:
      case idm_saveasjpeg:
      case idm_saveasgif:
      {
         JSaveDlg *savedlg = new JSaveDlg( this, 0, name);
         savedlg->run( this, v);
         return true;
      }
      case idm_quicksave:
      {
         short defaultft = getMenu()->getSubMenu( idm_file).
                                      getSubMenu( idm_saveas).
                                      getDefault();

         // !! this runs in the main thread
         return savingImminent( 0, name, defaultft);
      }
      case idm_quit:
         mgr.closeAllWindows();
         return true;
      default:
         setMessage( JVStr( "Got a command '%d'", v));
   }
   return false;
}

Painting &Painting::setMessage( char *txt)
{
   statbar->window()->setText( txt);
   return *this;
}

JPSpace *Painting::ps() const
{ return picture->ps(); }

int Painting::operator == ( const Painting &op2) const
{ return this == &op2; }

// File management ------------------------------------------------------------
BOOL Painting::savingImminent( JVBuffer *b, const JStr &nm, ulong id)
{
   BOOL bad = false;
   jlib_catch();
   try {

      switch( id) {
         case idm_saveasbmp:
         {
            JMemoryDC dc;
            JPSpace   pspace( dc);
            JBitmap   realBitmap( activeSize, picture->bitmap()->bpp());
            ulong     htImg( picture->bitmap()->size().y);

            pspace.select( realBitmap);
            pspace.render( JBmpImage( *picture->ps(),
                                      JRect( JPoint( 0, htImg - activeSize.y),
                                             activeSize),
                                      JRect( JPoint(), activeSize)));
            pspace.deselectBitmap();
            realBitmap.saveAs( nm);

            break;
         }
         case idm_saveasgif:
         case idm_saveasjpeg:
            JMsgBox mb( this, "Format not currently supported.");
            mb.openOverPointer();
            break;
      }
      datasafe = true;

   } catch( JException *e) {
      bad = true;
   }
   if( JPM::current()->exn) {
      jlib_catch();
      bad = true;
   }
   if( bad) {
      JMsgBox mb( this, "Error - could not save file.");
      mb.openOverPointer();
   }

   return true;
}

void Painting::savingComplete( JVBuffer *b, ulong id, BOOL cancelled)
{
   setText( name);
}

// Click handler for the main window -----------------------------------------
PaintMouseHandler::PaintMouseHandler( Painting *p) : JMouseHandler(),
                                                     painting( p),
                                                     tracking( false)
{}

BOOL PaintMouseHandler::active( const JPoint &pt)
{
   return ( pt.x <= painting->activeSize.x &&
            pt.y >= painting->picture->bitmap()->size().y-
                      painting->activeSize.y );
}

BOOL PaintMouseHandler::button1down( const JClickEvent &e)
{
   if( active( e.pos())) {
      tracking = true;
      painting->picture->capturePointer();
   }
   return false;
}

BOOL PaintMouseHandler::button1up( const JClickEvent &e)
{
   if( active( e.pos())) {
      tracking = false;
      painting->picture->releasePointer();
   }
   return false;
}

#define INCL_GPI
#include <os2.h>

BOOL PaintMouseHandler::mouseMove( const JMouseEvent &e)
{
   if( active( e.pos())) {
      if( tracking) {
        // painting->picture->ps()->render( JMove( e.pos()));
        // painting->picture->ps()->render( JLine( e.pos()));
         GpiSetPel( painting->picture->ps()->handle(), e.pos());
         painting->picture->invalidate( JRect( e.pos(), JSize( 1, 1)));
      }
   }
   return false;
}

BOOL PaintMouseHandler::button2click( const JClickEvent &e)
{
   if( active( e.pos())) {
      // offer it to the tool
      painting->menus[ menu_context]->popUp( *painting);
   } else
      painting->menus[ menu_context]->popUp( *painting);

   return false;
}
